/*
  memcmp: compare the data segments of two or more instances of a process

Mainly by Philipp Richter based on work by Philipp Reisner.

3.4.98
Thanks to joe object <thoma@Unix.CSLab.tuwien.ac.at> for a brunch
of fixes. 
*/

#include <sys/types.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <sys/ptrace.h>
#include <unistd.h>
#include <fcntl.h>

#include <asm/page.h>     /* PAGE_SIZE defined here */

#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>

#define MAPLLEN 100         /* linelength of map-file */

/* data for every process */
struct proc_data
{
  int pid;              /* the pid - this one was hard... */
  int maps;             /* count of maps */
  int mem;              /* mem file descriptor */
  unsigned long vm_begin[ 512 ];  /* array of vm beginnings */
  unsigned long vm_end[ 512 ];    /* array of vm endings */
};

/* process count */
int p_num;
struct proc_data p_data[512];

static void err_exit( char *t );
static void perr_exit( char *t );


/*
  search all instances of process (uses proc-filesystem)
*/
void all_instances( char *process )
{
  DIR *d;
  struct dirent *de;
  int pid, self;
  FILE *f;
  char path[PATH_MAX+1], comm[1024];
  int rc;
  /*  unsigned long found;*/

  self = getpid();

  if( !( d = opendir( "/proc" ) ) )
    perr_exit( "could not read /proc ?? " );

  while( ( de = readdir( d ) ) )
  {
    if( !( pid = atoi( de->d_name ) ) || pid == self )
      continue;

    sprintf( path, "/proc/%d/stat", pid );
    if( !( f = fopen( path, "r" ) ) )
      continue;

    rc = fscanf( f, "%*d (%[^)]", comm ) == 1;
    fclose( f );

    if( !rc )
      continue;

    if( strcmp( comm, process ) == 0 )
    {
      p_data[p_num].pid = pid;
      p_num++;
    }
  }
  closedir( d );
}

/*
  open the /proc/xx/mem-file for every process and store the fd in the struct
*/

static int memories_open = 0;

void open_mem()
{
  char buf[50];

  for( memories_open = 0; memories_open < p_num; memories_open++ )
  {
    sprintf( buf, "/proc/%d/mem", p_data[memories_open].pid );
    if( ( p_data[memories_open].mem = open( buf, O_RDONLY ) ) == -1 )
      err_exit( "Could not open mem file. Try to run as root." );
  }
}

/*
  close all mem files
*/
void close_mem()
{
  int i;

  for( i = 0; i < memories_open; i++ )
    close( p_data[i].mem );
}

/*
  halt all processes, so that we can read the mem files
*/

static int processes_stopped = 0;

void stop_processes()
{
  int pid, rc;

  printf( "Stopping processes\n" );
  for( processes_stopped = 0; processes_stopped < p_num; processes_stopped++ )
  {
    pid = p_data[processes_stopped].pid;
    rc = ptrace( PTRACE_ATTACH, pid, 0, 0 );
    if ( rc == -1 )
      perr_exit( "ptrace attach failed" );

    if( wait4( pid, &rc, 0, NULL ) != pid )
      printf( "wait4 failed\n" );
  }
}

/*
  continue previous stopped processes
*/
void cont_processes()
{
  int i, rc;

  if (processes_stopped == 0) return;
  printf( "Continuing processes\n" );
  for( i = 0; i < processes_stopped; i++ )
  {
    rc = ptrace( PTRACE_DETACH, p_data[i].pid, 0, 0 );
    if( rc == -1 )
      perror( "ptrace detach failed" );
  }
}

/*
  the memory maps
*/
void load_maps()
{
  int i, j, k;
  FILE *f;
  char buf[100];
  char line[ MAPLLEN ];
  char vm_perm[ 5 ];
  unsigned long vm_begin, vm_end;
  int vm_offset, dev_major, dev_minor, inode;

  for( i = 0; i < p_num; i++ )
  {
    printf( "Loading maps for process %d\n", p_data[i].pid );
    sprintf( buf, "/proc/%d/maps", p_data[i].pid );
    f = fopen( buf, "r" );
    if( !f )
      err_exit( "could not open maps file. you didn`t start me as normal  
user, did you ;-?" );
    j = k = 0;
    while( 1 )
    {
      fgets( line, MAPLLEN, f );
      if( feof( f ) )
        break;
      sscanf( line, "%lx-%lx %s %x %d:%d %d", &vm_begin, &vm_end, vm_perm,  
&vm_offset,
              &dev_major, &dev_minor, &inode );
      if( inode == 0 )
      {
        p_data[i].vm_begin[j] = vm_begin;
        p_data[i].vm_end[j]   = vm_end;
        j++;
      }
      else
        k++;
    }
    p_data[i].maps = j;
    printf( "  anonymous mappings: %d\n  file mappings:      %d\n\n", j, k );
    p_data[i].maps = j;
    fclose( f );
  }
}

#define DIF 1
#define EQU 2
#define UNR 3
/*
  compare the memory of all processes
*/
void compare_mem()
{
  int i, j, k;
  unsigned long vm_begin, vm_end;
  int found, found2, vm_end_smaller, vm_begin_bigger;
  int equal_pages, diff_pages, unr_pages;
  int map_equal_pages, map_diff_pages, map_unr_pages;
  int this_page;
  /*  char buf[100];*/
  int rc;
  char buffer1[PAGE_SIZE];
  char buffer2[PAGE_SIZE];

  equal_pages = diff_pages = unr_pages = 0;

  for( i = 0; i < p_data[0].maps; i++ )
  {
    vm_begin = p_data[0].vm_begin[i];
    vm_end = p_data[0].vm_end[i];
    printf( "Comparing mapping %lx-%lx (%ld pages, %ld KB) ", vm_begin, vm_end,
            (vm_end-vm_begin) / PAGE_SIZE, (vm_end-vm_begin) / 1024 );

    found2 = 1;
    for( j = 1; j < p_num; j++ )
    {
      found = 0;
      for( k = 0; k < p_data[j].maps; k++ )
      {
        if( p_data[j].vm_begin[k] >= vm_begin && p_data[j].vm_begin[k] <= vm_end )
        {
          found = 1;
          vm_end_smaller = ( p_data[j].vm_end[k] < vm_end );
          if( vm_end_smaller )
          {
            printf( "decreasing vm_end " );
            vm_end = p_data[j].vm_end[k];
          }
        }
        if( p_data[j].vm_end[k] >= vm_begin && p_data[j].vm_end[k] <= vm_end )
        {
          found = 1;
          vm_begin_bigger = ( p_data[j].vm_begin[k] > vm_begin );
          if( vm_begin_bigger )
          {
            printf( "increasing vm_begin " );
            vm_begin = p_data[j].vm_begin[k];
          }
        }
      }
      if( !found )
      {
        printf( "missing mapping ... skipping" );
        found2 = 0;
        break;
      }
    }

    printf("\n  ");
    if( !found2 )
      continue;

    map_diff_pages = map_equal_pages = map_unr_pages = 0;
    while( vm_begin < vm_end )
    {
      for( j = 0; j < p_num; j++ )
        if( lseek( p_data[j].mem, vm_begin, SEEK_SET ) == -1 )
          err_exit( "could not seek in mem file :-(" );

      this_page = EQU;

      if( ( rc = read( p_data[0].mem, buffer1, PAGE_SIZE ) ) < PAGE_SIZE )
        this_page = UNR;

      if( this_page != UNR )
        for( j = 1; j < p_num; j++ )
        {
          if( ( rc = read( p_data[j].mem, buffer2, PAGE_SIZE ) ) < PAGE_SIZE )
          {
            this_page = UNR;
            break;
          }
          if( memcmp( buffer1, buffer2, PAGE_SIZE ) != 0 )
          {
            this_page = DIF;
            break;
          }
        }

      switch( this_page )
      {
        case EQU:
          equal_pages++;
          map_equal_pages++;
          putchar('X');
          break;
        case DIF:
          diff_pages++;
          map_diff_pages++;
          putchar('-');
          break;
        case UNR:
        default:
          map_unr_pages++;
          unr_pages++;
          putchar('?');
          break;
      }

      vm_begin += PAGE_SIZE;
    }

    printf( "\n  -> %dp (%ldkB) equ,", map_equal_pages, map_equal_pages *  
PAGE_SIZE / 1024 );
    printf( " %dp (%ldkB) diff,", map_diff_pages, map_diff_pages * PAGE_SIZE  
/ 1024);
    printf( " %dp (%ldkB) unreadable,", map_unr_pages, map_unr_pages *  
PAGE_SIZE / 1024);
    if( map_equal_pages != 0 || map_diff_pages != 0)
      printf( " %d%% equal", 100 * map_equal_pages / ( map_equal_pages +  
map_diff_pages ));
    printf( "\n" );

  }
  printf("\nequal pages:      %d (%ldkB)\n", equal_pages, equal_pages *  
PAGE_SIZE / 1024 );
  printf( "different pages:  %d (%ldkB)\n", diff_pages, diff_pages *  
PAGE_SIZE / 1024);
  printf( "unreadable pages: %d (%ldkB)\n", unr_pages, unr_pages * PAGE_SIZE  
/ 1024);
  if( equal_pages != 0 || diff_pages != 0)
    printf( "==> %d%% are equal\n", 100 * equal_pages / ( equal_pages +  
diff_pages ));

}

void err_exit( char *t )
{
  printf( "%s\n", t );

  cont_processes();
  close_mem();

  exit( 1 );
}

void perr_exit( char *t )
{
  perror( t );

  cont_processes();
  close_mem();

  exit( 2 );
}

void usage()
{
  fprintf(stderr, "Usage: memcmp {pid1} [pid2 ...] | {process name}\n\n"
                  "Compares the pages of two or more command.\n");
  exit( 0 );
}


/* main function */
int main( int argc, char **argv )
{
  int p, fpid;

  if( argc < 2 )
    usage();

  if ( geteuid () != 0 )
    fprintf (stderr, "Warning: not super-user, let's try anyway ...\n");

  p_num = 0;

  if( ( fpid = atoi( argv[ 1 ] ) ) == 0 )
    all_instances( argv[ 1 ] );
  else
    for( p_num = 0; p_num < (argc-1); p_num++ )
    {
      if( ( p = atoi( argv[ p_num + 1 ]) ) == 0 )
        usage();
      p_data[ p_num ].pid = p;
    }

  if( p_num < 2 )
    err_exit( "need at least 2 processes..." );

  open_mem();
  stop_processes();
  load_maps();
  compare_mem();
  cont_processes();
  close_mem();

  return 0;
}

