/*
   merge_utils.c by Marnix Coppens

   This file is part of mergemem by Philipp Richter & Philipp Reisner

   mergemem is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   mergemem is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with mergemem; see the file COPYING.  If not, write to
   the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "mergemem.h"
#include "merge_utils.h"

#include <time.h>
#include <stdarg.h>

/*===============================================================================================*/

/*
 * Allocates 'size' bytes initialized with zeroes.
 * Panics if the memory is not available.
 */

void       *
safe_malloc(size_t size)
{
    void       *p;

    p = calloc(1, size);
    if (!p)
	err_exit(ERR_NOMEM, "Not enough memory\n");

#ifdef DEBUG
    nmallocs++;
    nmallsize += size;
#endif
    return p;
}

/*-----------------------------------------------------------------------------------------------*/

void
logmsg(int level, char *fmt,...)
{
    time_t      t;
    char        dbuf[100];
    static char nl_terminated = 1;

    if (loglevel & level)
    {
	va_list     pa;

	va_start(pa, fmt);
	if (flog)
	{
	    if (nl_terminated)
	    {
		t = time(0);
		strftime(dbuf, 100, "%b %d %H:%M:%S", localtime(&t));
		fprintf(flog, "%s ", dbuf);
	    }
	    vfprintf(flog, fmt, pa);
	    fflush(flog);
	}
	else
	    vfprintf(stderr, fmt, pa);
	nl_terminated = (fmt[strlen(fmt) - 1] == '\n');
	va_end(pa);
    }
}

/*-----------------------------------------------------------------------------------------------*/

static void
free_all(void)
{
    void       *p;
    MapList    *mapl, *nmapl;

    while (cmd_list)
    {
	p = cmd_list->next;
	free(cmd_list);
	cmd_list = p;
    }
    while (lib_list)
    {
	p = lib_list->next;
	free(lib_list);
	lib_list = p;
    }
    while (pid_list)
    {
	p = pid_list->next;
	free(pid_list);
	pid_list = p;
    }
    while (mrg_list)
    {
	p = mrg_list->next;
	for (mapl = ((MrgList *) mrg_list)->map; mapl; mapl = nmapl)
	{
	    nmapl = mapl->next;
	    free(mapl->chksumvalid);
	    free(mapl->chksum);
	    free(mapl);
	}
	free(mrg_list);
	mrg_list = p;
    }
}

/*-----------------------------------------------------------------------------------------------*/

void
err_exit(int exitval, char *errstr)
{
    if (exitval)
	printf("%s", errstr);

    free_all();
    if (mergemod_fd != -1)
	close(mergemod_fd);

#ifdef DEBUG
    logmsg(15, "free()    : %d\n", nfrees);
    logmsg(15, "malloc()  : %d\n", nmallocs);
    logmsg(15, "Total size: %d bytes\n", nmallsize);
#endif

    if (flog)
    {
	if (exitval)
	    logmsg(1, "Fatal error: %s. Program exited with return code %d\n",
		   errstr, exitval);
	else
	    logmsg(1, errstr);
	fclose(flog);
    }

    exit(exitval);
}

/*-----------------------------------------------------------------------------------------------*/

void
normal_exit(int dummy)
{
    logmsg(1, "mergemem stopped\n");
    err_exit(NO_ERROR, "mergemem stopped\n");
}

/*===============================================================================================*/

/*
 * All of the following find_xxx_by_zzz() routines can eventually be
 * replaced by faster equivalents using hash lookups and the like.
 *
 * Right now, a simple linear lookup does the job as well..
 */

CmdList    *
find_cmd_by_name(CmdList *cmdlist, char *cmdname)
{
    CmdList    *cmdl;

    for (cmdl = cmdlist; cmdl; cmdl = cmdl->next)
	if (strcmp(cmdl->cmd, cmdname) == 0)
	    break;

    return cmdl;
}

/*-----------------------------------------------------------------------------------------------*/

PidList    *
find_cmd_by_pid(PidList *pidlist, pid_t pid)
{
    PidList    *pidl;

    for (pidl = pidlist; pidl; pidl = pidl->next)
	if (pidl->pid == pid)
	    break;

    return pidl;
}

/*-----------------------------------------------------------------------------------------------*/

LibList    *
find_lib_by_devino(LibList *liblist, dev_t dev, ino_t ino)
{
    LibList    *libl;

    for (libl = liblist; libl; libl = libl->next)
	if ((libl->ino == ino) && (libl->dev == dev))
	    break;

    return libl;
}
/*-----------------------------------------------------------------------------------------------*/

/*
 * When using a linear search here, it's important to check for equal inodes first,
 * because this will be the most deciding. The sizes will be equal most of the time
 * except for stack frames (ino = dev = 0). The size check then makes sense because
 * if the stacks have different sizes, they're unlikely to have something in common.
 */

MapList    *
find_map_in_maplist(MapList *maplist, MapList *mapl1)
{
    MapList    *mapl2;
    dev_t       dev;
    ino_t       ino;

    dev = mapl1->dev;
    ino = mapl1->ino;

    for (mapl2 = maplist; mapl2; mapl2 = mapl2->next)
	if ((mapl2->ino == ino) && (mapl2->dev == dev) &&
	 (mapl2->vmend - mapl2->vmstart) == (mapl1->vmend - mapl1->vmstart))
	    break;

    return mapl2;
}

/*===============================================================================================*/

/*
 * The add_xxx_by_zzz() routines are actually lookup functions:
 *      if present -> return element
 *            else -> add element, return new element.
 *
 * As such, they will never return 0.
 */

CmdList    *
add_cmd_by_name(CmdList **pcmdlist, char *name)
{
    CmdList    *cmdl;

    if ((cmdl = find_cmd_by_name(*pcmdlist, name)) == 0)
    {
	cmdl = safe_malloc(sizeof(*cmdl) + strlen(name) + 1);
	cmdl->cmd = (char *) (cmdl + 1);
	strcpy(cmdl->cmd, name);

	cmdl->next = *pcmdlist;
	*pcmdlist = cmdl;
    }

    return cmdl;
}

/*-----------------------------------------------------------------------------------------------*/

PidList    *
add_cmd_by_pid(PidList **ppidlist, pid_t pid)
{
    PidList    *pidl;

    if ((pidl = find_cmd_by_pid(*ppidlist, pid)) == 0)
    {
	pidl = safe_malloc(sizeof(*pidl));
	pidl->pid = pid;
	pidl->valid = 0;

	pidl->next = *ppidlist;
	*ppidlist = pidl;
    }

    return pidl;
}

/*-----------------------------------------------------------------------------------------------*/

LibList    *
add_lib_by_devino(LibList **pliblist, dev_t dev, ino_t ino)
{
    LibList    *libl;

    if ((libl = find_lib_by_devino(*pliblist, dev, ino)) == 0)
    {
	libl = safe_malloc(sizeof(*libl));
	libl->dev = dev;
	libl->ino = ino;

	libl->next = *pliblist;
	*pliblist = libl;
    }

    return libl;
}
