/*
   merge_parse.c by Marnix Coppens

   This file is part of mergemem by Philipp Richter & Philipp Reisner

   mergemem is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   mergemem is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with mergemem; see the file COPYING.  If not, write to
   the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "mergemem.h"
#include "merge_utils.h"

#include <getopt.h>
#include <sys/stat.h>

#define OPTIONS		"ade:hf:i:l:n:p:s:x:"

static struct option long_options[] =
{
    {"all", 0, 0, 'a'},
    {"daemon", 0, 0, 'd'},
    {"nice", 1, 0, 'e'},
    {"logfile", 1, 0, 'f'},
    {"help", 0, 0, 'h'},
    {"loglevel", 1, 0, 'l'},
    {"name", 1, 0, 'n'},
    {"pid", 1, 0, 'p'},
    {"shlib", 1, 0, 's'},
    {"dontmerge", 1, 0, 'x'},
    {0, 0, 0, 0}
};

/*===============================================================================================*/

static void
usage(char *name)
{
    printf("Usage: %s [options]...\n\n", name);
    err_exit(ERR_USAGE,
	     "Calls kernel module to locate and merge common memory of different processes.\n"
	     "\nOptions are:\n"
	     "  -a, --all                merge all commands\n"
	     "  -e LEV, --nice LEV       set the nice level the daemon runs under. The default\n"
	     "                           is 18 (which is considered to be quake-safe ;-)\n"
	     "  -f FILE, --logfile FILE  log to FILE\n"
	     "  -i INT, --interval INT   use INT as interval for the last command. Set as\n"
	     "                           default interval if there was no command before.\n"
	     "  -l LEV, --loglevel LEV   set the log level (0..15)\n"
     "  -p PID, --pid PID        use pid PID as one of the pids to merge.\n"
     "  -n CMD, --name CMD       use all pids refering to the command CMD\n"
	"  -x CMD, --dontmerge CMD  don't merge processes of this command\n"
	     "\nGeneral options are:\n"
	     "  -h, --help               print this help and exit\n"
	     "  -v, --version            print version and exit\n"
	     "\nIf run without parameters, the defaults are\n"
	     "\t-a -i 60 -l 1 -f " LOGFILE "\n");
}

/*-----------------------------------------------------------------------------------------------*/

void
parse_options(int argc, char **argv)
{
    struct stat stbuf;
    int         c, value;
    pid_t       pid;
    char       *str;
    PidList    *pidl;
    CmdList    *cmdl;

    while ((c = getopt_long(argc, argv, OPTIONS, long_options, 0)) != -1)
    {
	switch (c)
	{
		/* all commands */
	    case 'a':
		mergeall = TRUE;
		break;

		/* run as daemon */
	    case 'd':
		run_as_daemon = TRUE;
		break;

		/* set the nice level */
	    case 'e':
		value = strtol(optarg, &str, 10);
		if (str == optarg || value < -19 || value > 19)
		    err_exit(ERR_NICE, "nicelevel must be between -19 and 19\n");
		nicelevel = value;
		break;

		/* logfile */
	    case 'f':
		strncpy(logfile, optarg, PATH_MAX - 1);
		logfile[PATH_MAX - 1] = '\0';
		break;

		/* interval */
	    case 'i':
		value = strtol(optarg, &str, 10);
		if (str == optarg || value < 1)
		    err_exit(ERR_INTERVAL, "interval must be greater than 0\n");
		if (cmd_list)
		    cmd_list->interval = value;
		else
		    interval = value;
		break;

		/* loglevel */
	    case 'l':
		value = strtol(optarg, &str, 10);
		if (str == optarg || value < 0 || value > 15)
		    err_exit(ERR_LOG, "loglevel must be between 0 and 15\n");
		loglevel = value;
		break;

		/* command by name */
	    case 'n':
		cmdl = add_cmd_by_name(&cmd_list, optarg);
		cmdl->interval = interval;
		break;

		/* command by pid */
	    case 'p':
		pid = (pid_t) atoi(optarg);
		if (pid < 2)
		    err_exit(ERR_PID, "pid must be a number greater than 1.\n");
		add_cmd_by_pid(&pid_list, pid);
		break;

		/* merge shared library's bss only */
	    case 's':
		if ((stat(optarg, &stbuf) == -1) || !S_ISREG(stbuf.st_mode))
		    err_exit(ERR_LIB, "Invalid library.\n");
		add_lib_by_devino(&lib_list, stbuf.st_dev, stbuf.st_ino);
		break;

		/* exclude command by name */
	    case 'x':
		add_cmd_by_name(&ign_list, optarg);
		break;

	    case 'h':
	    case '?':
	    default:
		usage(argv[0]);
		break;
	}
    }

    if (optind < argc)
	err_exit(ERR_ARGS, "Too many arguments.\n");

    /*
     * in single mode: if mergeall is TRUE, get rid of both cmd_list and pid_list.
     * in daemon mode: if specific processes were specified, there can be no mergeall.
     * in either mode: if both lists are empty, merge all processes.
     */
    if (!cmd_list && !pid_list)
	mergeall = TRUE;
    else if (run_as_daemon)
	mergeall = FALSE;
    else if (mergeall)
    {
	while (cmd_list)
	{
	    cmdl = cmd_list->next;
	    free(cmd_list);
	    cmd_list = cmdl;
	}
	while (pid_list)
	{
	    pidl = pid_list->next;
	    free(pid_list);
	    pid_list = pidl;
	}
    }
}
