/*
  mergemem.c
  
  This file is part of mergemem by Philipp Richter & Philipp Reisner

  mergemem is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.
  
  mergemem is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with mergemem; see the file COPYING.  If not, write to
  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

*/


#include "mergemem.h"

/*
  Some global variables
*/

struct CmdList *cmd_list = NULL;
struct PidList *pid_list = NULL;
struct DontMerge *dont_merge = NULL;
char logfile[PATH_MAX] = "/var/log/mergemem.log";
int loglevel = 2;
int nicelevel = 18;
unsigned long int interval = 60;
int run_as_daemon = FALSE;
int mergeall = FALSE;
int onlycmd = FALSE;
int noconfig = FALSE;

int logfile_set = FALSE;
int loglevel_set = FALSE;
int interval_set = FALSE;
int nice_set = FALSE;

#ifdef DEBUG
int debug_mode = FALSE;
#endif

int logmsg_was_n = TRUE;

FILE *logf = NULL;
/* for statistics */
int saved_pages;
int chk_runs;
int sum_runs;
/* fd for proc file*/
int mergemod_fd;

/* --__--__--__--__--__--__--__-- */

int main( int argc, char **argv )
{
  int retval, modver;

  /* initialize parameters */
  if( argc == 1)
  {
    mergeall = TRUE;
    loglevel = 1;
  }
  else
    parse_options( argc, argv );
  
#ifdef DEBUG
  if(debug_mode)
    dump_vars();
#endif

  /* open logfile*/
  if( run_as_daemon && loglevel > 0 && (logf = fopen(logfile, "a")) == 0 )
  {
    setvbuf(logf, 0, _IONBF, 0);
    perror("could not open logfile for writing. logging turned off.\n");
    loglevel = 0;
  }
	
  /* check if we are already running */
  if(run_as_daemon)
    check_running();

  /* open mergemod file */
  if( (mergemod_fd = open("/dev/mergemem", O_RDONLY)) == -1 )
  {
    if(logf)
      logmsg(1, "/dev/mergemem not found. insert mergemod.o into kernel\n");
    err_exit("/dev/mergemem not found. insert mergemod.o into kernel\n");
  }
  
  /* check the version*/
  retval = ioctl( mergemod_fd, MERGEMEM_CHECK_VER, &modver );
  if(retval || modver != MOD_VERSION)
    err_exit("module incompatible with program\n");
    
  logmsg(3, "mergemod module has version %d\n", modver );
  logmsg(1, "mergemem started\n");

  if(run_as_daemon)
  {
    pid_t p;
    p = fork();
    if(p == -1)
      err_exit("could not fork daemon\n");
    if(p == 0)
    {
      signal(SIGHUP, &cfg_reread);
      signal(SIGTERM, &ok_exit);
      wait_loop();
    }
  }
  else
  {
    merge_singleshot();
    ok_exit();
  }

  return( 0 );
}

void usage( char *name )
{
  printf( "Usage: %s [options]...\n\n", name );
  printf( "Call kernel module to merge memory of equal commands\n\n"
	  "Options are:\n"
	  "  -d, --daemon             run in daemon mode (uses config file for finetuning\n"
	  "                           see manpage for details). If used, this must be the\n"
	  "                           first option. Default is singleshot-mode.\n\n"
	  "In daemon mode the following options are enabled:\n"
	  "  -n CMD, --name CMD       use all pids refering to the command CMD\n"
	  "  -i INT, --interval INT   use INT as interval for the last command. Set as\n"
	  "                           default interval if there was no command before.\n"
	  "  -x CMD, --dontmerge CMD  don't merge processes of this command\n"
	  "  -o, --onlycmds           merge ONLY processes of commands given on the command\n"
	  "                           line or config file\n"
	  "  -c, --noconfig           don't read the config file\n"
	  "  -e LEV, --nice LEV       set the nice level, the daemon runs under. the default\n"
	  "                           is 18 (which is considered to be quake-safe ;-)\n"
	  "  -l LEV, --loglevel LEV   set the log level (0,1,2,3)\n"
	  "  -f FILE, --logfile FILE  log to FILE\n\n"
	  "In singleshot mode following options are enabled:\n"
	  "  -a, --all                merge all commands\n"
	  "  -p PID, --pid PID        use pid PID as one of the pids to merge. at least 2\n"
	  "                           or more times required.\n"
	  "  -n CMD, --name CMD       use all pids refering to the command CMD\n"
	  "  -l LEV, --loglevel LEV   set the log level (0,1,2,3)\n\n"
	  "General options are:\n"
	  "  -h, --help               print this help and exit\n"
	  "  -v, --version            print version and exit\n"
#ifdef DEBUG
	  "  -g, --debug              put some debugging messages to stdout\n"
#endif
	  "\nIf run without parameters, the default is --all --loglevel 1\n"  );
  exit( 0 );
}

void parse_options( int argc, char **argv )
{
  int c, p, lev, i;
  unsigned long inter;
  char *t;
  struct PidList *pl;
  struct CmdList *cl;
  struct DontMerge *dm;
  
  static struct option long_options[] =
  {
    { "daemon",    0, 0, 'd' },
    
    { "name",      1, 0, 'n' },
    { "interval",  1, 0, 'i' },
    { "dontmerge", 1, 0, 'x' },
    { "onlycmds",  0, 0, 'o' },
    { "noconfig",  0, 0, 'c' },
    { "nice",      1, 0, 'e' },
    { "loglevel",  1, 0, 'l' },
    { "logfile",   1, 0, 'f' },
		
    { "all",       0, 0, 'a' },
    { "pid",       1, 0, 'p' },

    { "version",   0, 0, 'v' },
    { "help",      0, 0, 'h' },

#ifdef DEBUG
    { "debug",     0, 0, 'g' },
#endif

    { 0, 0, 0, 0 }
  };
	
  i = 0;
  while(1)
  {
#ifdef DEBUG
#define OPTS "dn:i:x:ocl:f:ap:vhge:"
#else
#defint OPTS "dn:i:x:ocl:f:ap:vhe:"
#endif
    c = getopt_long( argc, argv, OPTS, long_options, 0 );
    if( c == -1 )
      break;
    
    switch( c )
    {
      /* daemon mode */
      case 'd':
	if(i > 0)
	  err_exit("-d / --daemon must be the first option\n");
	if( getuid() != 0 )
	  err_exit("only root may run mergemem in daemon mode\n");
	run_as_daemon = TRUE;
	break;
	
	/* all commands */
      case 'a':
	if(run_as_daemon)
	  err_exit("option -a / --all not allowed in daemon mode\n");
	mergeall = TRUE;
	break;
	
	/* dontmerge */
      case 'x':
	if(!run_as_daemon)
	  err_exit("option -x / --dontmerge not allowed in singleshot mode\n");
	if( (dm = malloc(sizeof(struct DontMerge) + strlen(optarg) + 1)) == NULL )
	  err_exit("memory exhausted\n");
	dm->cmd = (char*)dm+sizeof(struct DontMerge);
	strcpy( dm->cmd, optarg );
	dm->next = dont_merge;
	dont_merge = dm;
	break;
	
	/* interval */
      case 'i':
	inter = strtol(optarg, &t, 10);
	if(t == optarg || inter < 1)
	  err_exit("interval must be greater than 0\n");
	if(cmd_list == NULL)
	{
	  interval = inter;
	  interval_set = TRUE;
	}
	else
	  cmd_list->iv = inter;
	break;
	
	/* onlycmd */
      case 'o':
	if(!run_as_daemon)
	  err_exit("option -x / --dontmerge not allowed in singleshot mode\n");
	onlycmd = TRUE;
	break;
	
	/* no config */
      case 'c':
	if(!run_as_daemon)
	  err_exit("option -x / --dontmerge not allowed in singleshot mode\n");
	noconfig = TRUE;
	break;
	
	/* pid */
      case 'p':
	if(run_as_daemon)
	  err_exit("option -p / --pid not allowed in daemon mode\n");
	if((p = atoi(optarg)) < 2)
	  err_exit("pid must be greater that 1\n");
	if((pl = malloc(sizeof(struct PidList))) == 0)
	  err_exit("memory exhausted\n");
	pl->pid = p;
	pl->next = pid_list;
	pid_list = pl;
	break;

	/* command name */
      case 'n':
	if( (cl = malloc(sizeof(struct CmdList) + strlen(optarg) + 1)) == NULL )
	  err_exit("memory exhausted\n");
	cl->cmd = (char*)cl+sizeof(struct CmdList);
	strcpy( cl->cmd, optarg );
	cl->iv = 0;
	cl->cntr = 0;
	cl->next = cmd_list;
	cmd_list = cl;
	break;
	
	/* loglevel */
      case 'l':
	lev = strtol(optarg, &t, 10);
	if( t == optarg || lev < 0 || lev > 3)
	  err_exit("loglevel must be between 0 and 3\n");
	loglevel = lev;
	loglevel_set = TRUE;
	break;

	/* logfile */
      case 'f':
	if(!run_as_daemon)
	  err_exit("option -f / --logfile not allowed in singleshot mode\n");
	strncpy(logfile, optarg, PATH_MAX-1);
	logfile[PATH_MAX-1] = '\0';
	logfile_set = TRUE;
	break;
	
	/* set the nice level */
      case 'e':
	lev = strtol(optarg, &t, 10);
	if( t == optarg || lev < -19 || lev > 19)
	  err_exit("nice level must be between -19 and 19\n");
	nicelevel = lev;
	nice_set = TRUE;
	break;
	
	/* version */
      case 'v':
	t = strrchr( *argv, '/')+1;
	if( t == (char*)1 ) t = *argv;
	printf( "This is version " VERSION " of %s.\n"
		"Report bugs and errors to\n"
		"Philipp Richter <philipp@mondoshawan.ml.org> or\n"
		"Philipp Reisner <e9525415@stud2.tuwien.ac.at>.\n", t );
	exit(0);
	
#ifdef DEBUG
	/* debug*/
      case 'g':
	debug_mode = TRUE;
	break;
#endif

	/* unknown */
      default:
	err_exit("unknown option. i'm confused ...\n");
	
	/* help */
      case 'h':
      case '?':
	usage( *argv );
    }

    i++;
  }
  
  if( optind < argc )
    err_exit( "too much arguments ...\n" );
  
  /* read the config file */
  if(run_as_daemon)
  {
    if(!noconfig)
      read_cfg_file();
    if(setpriority(PRIO_PROCESS, 0, nicelevel))
      logmsg(1, "could not set priority.\n");
  }

  /* if not run as daemon and pidlist & cmdlist empty -> merge all*/
  if(!run_as_daemon && cmd_list == NULL && pid_list == NULL)
    mergeall = TRUE;

  /* if all, delete command list */
  if(mergeall)
  {
    if(!loglevel_set)
      loglevel = 1;
    while(cmd_list)
    {
      cl = cmd_list->next;
      free(cmd_list);
      cmd_list = cl;
    }
  }
}

/*
  read the config file /etc/mergemem.cfg	
*/
void read_cfg_file()
{
  FILE *f;
  char buf[PATH_MAX], *buf1, *buf2, *buf3, *t;
  int line;

  /* no cfg file or not allowed to read? */
  if( !(f = fopen( CFG_NAME, "r")) )
    return;

  line = 0;
  while( fgets(buf, sizeof(buf), f) ) 
  {
    buf1 = buf;
    line++;
    if( (buf2 = cfg_next_token( &buf1 )) == NULL) continue;
    if( strchr( "#%\n", *buf1) ) continue;

    if( strcmp(buf1, "command") == 0)
    {
      struct CmdList *cl;
      unsigned long int iv;
      if( (buf1 = cfg_next_token( &buf2 )) == NULL)
	cfg_error(line, "argument 1 for 'command' missing");
      if( (buf3 = cfg_next_token( &buf1 )) == NULL)
	cfg_error(line, "argument 2 for 'command' missing");
      if( (iv = atol(buf1)) < 1 )
	cfg_error(line, "interval must be greater than zero");
      /* buf2 = name, buf1 = interval */
      
      for(cl=cmd_list; cl != NULL && (strcmp(cl->cmd, buf2) != 0); cl=cl->next) ;
      if(cl) continue;
      
      if( (cl = malloc(sizeof(struct CmdList) + strlen(buf2) + 1)) == NULL )
	err_exit("memory exhausted\n");
      cl->cmd = (char*)cl+sizeof(struct CmdList);
      strcpy( cl->cmd, buf2 );
      cl->iv = iv;
      cl->cntr = 0;
      cl->next = cmd_list;
      cmd_list = cl;
    } else if( strcmp(buf1, "dontmerge") == 0)
    {
      int foundone = 0;
      struct DontMerge *dm;
      while( (buf1 = cfg_next_token( &buf2 )) )
      {
	for(dm=dont_merge; dm != NULL && (strcmp(dm->cmd, buf2) != 0); dm=dm->next) ;
	if(dm) 
	{
	  foundone = 1;
	  buf2 = buf1;
	  continue;
	}
	if( (dm = malloc(sizeof(struct DontMerge) + strlen(buf2) + 1)) == NULL)
	  err_exit("memory exhausted\n");
	dm->cmd = (char*)dm+sizeof(struct DontMerge);
	strcpy( dm->cmd, buf2 );
	dm->next = dont_merge;
	dont_merge = dm;
	foundone = 1;
	buf2 = buf1;
      }
      if( !foundone )
	cfg_error(line, "argument(s) for 'dontmerge' missing");
    } else if( strcmp(buf1, "interval") == 0)
    {
      unsigned long int iv;
      if(interval_set)
	continue;
      if( (buf1 = cfg_next_token( &buf2 )) == NULL)
	cfg_error(line, "argument for 'interval' missing");
      if( (iv = atol(buf2)) < 1 )
	cfg_error(line, "interval must be greater than 0");
      interval = iv;
    } else if( strcmp(buf1, "logfile") == 0)
    {
      if(logfile_set)
	continue;
      if( (buf1 = cfg_next_token( &buf2 )) == NULL)
	cfg_error(line, "argument for 'logfile' missing");
      strncpy( logfile, buf2, PATH_MAX - 1);
      logfile[PATH_MAX-1] = '\0';
    } else if( strcmp(buf1, "loglevel") == 0)
    {
      int lev;
      if(loglevel_set)
	continue;
      if( (buf1 = cfg_next_token( &buf2 )) == NULL)
	cfg_error(line, "argument for 'loglevel' missing");
      
      lev = strtol(buf2, &t, 10);
      if( t == buf2 || lev < 0 || lev > 3)
	cfg_error(line, "'loglevel' must be between 0 and 3");
      loglevel = lev;
    } else if( strcmp(buf1, "nice") == 0)
    {
      int lev;
      if(nice_set)
	continue;
      if( (buf1 = cfg_next_token( &buf2 )) == NULL)
	cfg_error(line, "argument for 'nice' missing");

      lev = strtol(buf2, &t, 10);
      if( t == buf2 || lev < -19 || lev > 19)
	cfg_error(line, "'nice' must be between -19 and 19");
      nicelevel = lev;
    } else if( strcmp(buf1, "onlycmd") == 0)
    {
      onlycmd = TRUE;
    } else
      cfg_error(line, "illegal tag");
  }
	
  fclose(f);
}

/*
  print error message and abort
*/
void cfg_error( int line, const char *text )
{
  logmsg(1, "error in %s, line %d: %s\n", CFG_NAME, line, text);
  free_all();
  exit(1);
}

/*
  find next token in str, skip spaces end set a \0 after last nonspace char
*/
char *cfg_next_token( char **str )
{
  char *end;
  while( **str && isspace(**str) ) { (*str)++; }
  end = *str;
  while( *end && !isspace( *end) ) end++;
  if( *str == end ) return NULL;
  *end++ = '\0';
  return end;
}

/*
  free all malloced memory
*/
void free_all()
{
  struct DontMerge *dm;
  struct CmdList *cl;
  struct PidList *pl;
  while(dont_merge)
  {
    dm = dont_merge->next;
    free(dont_merge);
    dont_merge = dm;
  }
  while(cmd_list)
  {
    cl = cmd_list->next;
    free(cmd_list);
    cmd_list = cl;
  }
  while(pid_list)
  {
    pl = pid_list->next;
    free(pid_list);
    pid_list = pl;
  }
}

void err_exit( const char *msg )
{
  if( mergemod_fd != -1 )
    close( mergemod_fd );
  
  printf(msg);
  free_all();
  if(logf)
  {
    logmsg(1, "FATAL ERROR: %s. giving up my existence...\n", msg);
    fclose(logf);
  }
  exit(1);
}

void ok_exit()
{
  if( mergemod_fd != -1 )
    close( mergemod_fd );

  logmsg(1, "mergemem stopped\n");
  free_all();
  exit(0);
}

void logmsg( int lev, char *fmt, ...)
{
  time_t t;
  char dbuf[100];

  if( loglevel >= lev )
  {
    va_list pa;
    va_start( pa, fmt );
    if(logmsg_was_n)
    {
      t = time(0);
      (size_t) strftime(dbuf, 100, "%b %d %H:%M:%S", localtime(&t));
      if(logf) fprintf( logf , "%s ", dbuf);
    }
    vfprintf( logf ? logf : stderr, fmt, pa );
    va_end( pa );
    if(logf) fflush(logf);
    logmsg_was_n = (fmt[strlen(fmt)-1] == '\n');
  }
}

void cfg_reread()
{
  logmsg(1, "rereading cfg-file (SIGHUP). ignoring commandline options.\n");
  signal(SIGHUP, &cfg_reread);
  free_all();
  cmd_list = NULL;
  dont_merge = NULL;
  loglevel_set = FALSE;
  interval_set = FALSE;
  read_cfg_file();
#ifdef DEBUG
  if(debug_mode)
    dump_vars();
#endif
}

/*
  this is the wait loop which is running in the forked process of the daemon
  it`s not perfect but it does it`s job...
*/
void wait_loop()
{
  struct CmdList *cl;
  unsigned long cntr = 0;

  while(1)
  {
    if(cntr == 0)
    {
      if(!onlycmd)
      {
	/* merge all commands except dontmerge and commands with own intervals */
	cntr = interval;
	logmsg(2, "merging commands\n");
	merge_all_commands();
      }
    }
    for(cl = cmd_list; cl; cl=cl->next)
    {
      if(cl->cntr == 0)
      {
	merge_command( cl->cmd );
	cl->cntr = cl->iv;
      }
      cl->cntr--;
    }
    cntr--;
    sleep(1);
  }
}

void merge_singleshot()
{
  struct CmdList *cl;
	
  if(mergeall)
    merge_all_commands();
  else
  {
    /* go through pid list */
    merge_pids( &pid_list );
    for(cl = cmd_list; cl; cl=cl->next)
      merge_command( cl->cmd );
  }
}

/*
  merge all commands which are not in the CmdList & DontMerge list...
*/
void merge_all_commands()
{
  struct CmdList *ctm = NULL, *c, *cl;
  struct DontMerge *dm;
  /*  int found;*/
  /*  struct PidList *pl = NULL, *pl2;*/
  DIR *d;
  struct dirent *de;
  int pid, self;
  FILE *f;
  char path[PATH_MAX+1], comm[1024];
  int rc;

  self = getpid();

  if(!(d = opendir("/proc")))
    err_exit("could not read /proc. kernel proc support required...\n");

  while((de = readdir(d)))
  {
    if(!( pid = atoi(de->d_name)) /*|| pid == self*/ )
      continue;

    sprintf( path, "/proc/%d/stat", pid );
    if( !( f = fopen( path, "r" ) ) )
      continue;

    rc = fscanf( f, "%*d (%[^)]", comm ) == 1;
    fclose( f );

    if( !rc )
      continue;

    for( c = ctm; c != NULL; c = c->next )
    {
      if( strcmp( c->cmd, comm ) == 0)
      {
	c->cntr++;
	break;
      }
    }

    if( c ) /* if found -> next */
      continue;

    /* ensure we dont have this command in the command list*/
    for( cl = cmd_list; cl != NULL; cl = cl->next )
      if( strcmp( cl->cmd, comm ) == 0)
	break;
    if( cl )
      continue;

    /* ensure we dont have this command in the dont merge list*/
    for( dm = dont_merge; dm != NULL; dm = dm->next )
      if( strcmp( dm->cmd, comm ) == 0)
	break;
    if( dm )
      continue;

    if( (c = malloc(sizeof(struct CmdList) + strlen(comm) + 1)) == NULL )
      err_exit("memory exhausted\n");
    c->cmd = (char*)c+sizeof(struct CmdList);
    strcpy( c->cmd, comm );
    c->next = ctm;
    c->cntr = 1;
    ctm = c;
  }
  closedir( d );

  if( ctm ) /* if more than one pid */
  {
    for( c = ctm; c != NULL; c = c->next )
    {
      if( c->cntr > 1 )
      {
	logmsg(1, "merging %s (%d instances)\n", c->cmd, c->cntr );
	merge_command( c->cmd );
      }
    }
  }

  while(ctm)
  {
    c = ctm;
    ctm = ctm->next;
    free( c );
  }
}

/*
  get all pids of a command and call memory merge routine
*/
void merge_command( const char *cmd )
{
  struct PidList *pl = NULL, *pl2;
  DIR *d;
  struct dirent *de;
  int pid, self, first;
  FILE *f;
  char path[PATH_MAX+1], comm[1024];
  int rc;
  /*  unsigned long found;*/

  logmsg(2, "%s is running as ", cmd);
  self = getpid();

  if( !( d = opendir( "/proc" ) ) )
    err_exit("could not read /proc. kernel proc support required...\n");

	first = TRUE;
  while( ( de = readdir( d ) ) )
  {
    if( !( pid = atoi( de->d_name ) ) || pid == self )
      continue;

    sprintf( path, "/proc/%d/stat", pid );
    if( !( f = fopen( path, "r" ) ) )
      continue;
		
    rc = fscanf( f, "%*d (%[^)]", comm ) == 1;
    fclose( f );

    if( !rc )
      continue;

    if( strcmp( comm, cmd ) == 0 )
    {
      if( (pl2 = malloc(sizeof(struct PidList)) ) == NULL)
	err_exit( "memory exhausted\n" );
      pl2->next = pl;
      pl = pl2;
      pl2->pid = pid;
      if(!first) logmsg(2, ",");
      logmsg(2, "%d", pid );
      first = FALSE;
    }
  }
  closedir( d );
  logmsg(2, "\n");

  /* now we have built a list of pids from the command */

  if( pl == NULL ) 
  {
    if( !run_as_daemon )
      logmsg(1, "no processes of command %s found\n", cmd );
    return;
  }

  if( pl->next ) /* if more than one pid */
    merge_pids( &pl );

  while(pl)
  {
    pl2 = pl;
    pl = pl->next;
    free( pl2 );
  }
}

void merge_pids( struct PidList **pl )
{
  if( !*pl ) return;
  if( !(*pl)->next ) return;

  saved_pages = 0;
  chk_runs = 0;
  sum_runs = 0;

  if( init_maps( *pl ) )
    merge_memory( *pl );

  logmsg(1, "> Saved %d pages (%d kb)\n", saved_pages, saved_pages * PAGE_SIZE / 1024);
/*  logmsg(2, "** check runs: %d\n", chk_runs);
  logmsg(2, "** sum runs: %d\n", sum_runs);*/

}

int init_maps( struct PidList *pl )
{
  struct PidList *p;
  char buf[100];
  FILE *f;
  unsigned long vm_begin, vm_end;
  int vm_offset, dev_major, dev_minor, inode;
  char line[ MAPLLEN ];
  char vm_perm[ 5 ];
  int i, j;

  if( !pl )
    return FALSE;

  /* read the map files in memory */
  for( p = pl; p != NULL; p = p->next )
  {
    i = j = 0; /* this is the mapcounter */
    logmsg(3, "loading maps for process %d\n", p->pid );
    sprintf( buf, "/proc/%d/maps", p->pid );
    f = fopen( buf, "r" );
    if( !f )
    {
      logmsg(1, "could not open maps file. uuuugh...\n");
      return FALSE;
    }
    while( 1 )
    {
      fgets( line, MAPLLEN, f );
      if( ferror ( f ) ) break; /* FIXME! In this case the process is dead!
                                 * => remove it from PID list! 
                                 */
      if( feof( f ) ) break;
      sscanf( line, "%lx-%lx %s %x %d:%d %d", &vm_begin, &vm_end, vm_perm, &vm_offset,
	      &dev_major, &dev_minor, &inode );
      if( inode == 0 )
      {
	p->vm_begin[i] = vm_begin;
	p->vm_end[i]   = vm_end;
	i++;
      }
      else
	j++;
    }
    logmsg(3, "process has %d anonymous and %d file mappings\n", i, j);
    p->maps = i;
    fclose( f );
  }
  return TRUE;
}

void merge_memory( struct PidList *pl )
{
  int i, k;
  unsigned long vm_begin, vm_end;
  int found, found2, vm_begin_smaller, vm_end_bigger;
  struct PidList *p;

  if( !pl ) return;
  if( !pl->next ) return;

  /* reset counter for statistics */
  for( p = pl; p != NULL; p = p->next )
    p->pages_mrg = p->pages_err = p->pages_nmrg = p->pages_ashr = 
      p->pages_mov_to = p->pages_mov_from = 0;

  /* test if the other pids have the map also and if to correct the beginning and ending */
  for( i = 0; i < pl->maps; i++ )
  {
    vm_begin = pl->vm_begin[i];
    vm_end   = pl->vm_end[i];
    logmsg(3, "comparing mapping %lx-%lx (%ld pages, %ld kb)\n", vm_begin, vm_end,
	   (vm_end-vm_begin) / PAGE_SIZE, (vm_end-vm_begin) / 1024 );

    found2 = 1;
    for( p = pl->next; p != NULL; p=p->next )
    {
      found = 0;
      for( k = 0; k < p->maps; k++ )
      {
	/* correct the beginning if required*/
	if( p->vm_begin[k] <= vm_begin && p->vm_end[k] >= vm_begin )
	{
	  found = 1;
	  vm_begin_smaller = ( p->vm_begin[k] < vm_begin );
	  if( vm_begin_smaller )
	  {
	    vm_begin = p->vm_begin[k];
	    logmsg(3, "decreasing vm_begin to %lx\n", vm_begin );
	  }
	}
	/* correct the ending if required*/
	if( p->vm_end[k] >= vm_end && p->vm_begin[k] <= vm_end )
	{
	  found = 1;
	  vm_end_bigger = ( p->vm_end[k] > vm_end );
	  if( vm_end_bigger )
	  {
	    vm_end = p->vm_end[k];
	    logmsg(3, "increasing vm_end to %lx\n", vm_end );
	  }
	}
      }
      if( !found )
      {
	logmsg(3, "missing mapping ... skipping\n" );
	found2 = 0;
	break;
      }
    }
    if( !found2 )
      continue;

    /* ok, now the map exists in every process and the start/endings have been corrected */

    /* this is the main function in this little program */
    /* here the checksums for every page are generated, then compared and then */
    /* the pages are merged... */
    check_and_merge( vm_begin, vm_end, pl );

    for( p = pl; p != NULL; p=p->next )
    {
      p->pages_mrg += p->mpages_mrg;
      p->pages_err += p->mpages_err;
      p->pages_nmrg += p->mpages_nmrg;
      p->pages_ashr += p->mpages_ashr;
      p->pages_mov_to += p->mpages_mov_to;
      p->pages_mov_from += p->mpages_mov_from;
    }
  }

  for( p = pl; p != NULL; p = p->next )
    logmsg(2, ">%5d: %3dp merged, %3dp shared, %3dp misses, %3dp not merged, m:%3dp t, %3dp f\n",
	   p->pid, p->pages_mrg, p->pages_ashr, p->pages_err, p->pages_nmrg,p->pages_mov_to,p->pages_mov_from);
}

/*
  first create checksums for the current page. later the checksums are compared and all
  equal one are merged. this means if you merge a command with 10 processes
  and the checksum of one page looks like this:

  pid       0  1  2  3  4  5  6  7  8  9
  checksum 10 10 20 30 10 20 20 40 30 30

  then the following pids will be merged:

  1.  (10)  0,1,4
  2.  (20)  2,5,6
  3.  (30)  3,8,9
  4.  (40)  none

*/
void check_and_merge( unsigned long beg, unsigned long end, struct PidList *pl )
{
  struct PidList *p, *fp;
  int n = 0;
  int fpnew;
  unsigned long *ids;
  int nids;
  int i, j, fnd;
  unsigned long m, chk;
  struct mergemem_mmem mm_mmem;
  struct mergemem_chksum mm_chk;
  int cs_count;
  int fp_maps;

  for(p = pl; p != NULL; p = p->next)
  {
    p->mpages_mrg = p->mpages_err = p->mpages_nmrg = p->mpages_ashr = 
      p->mpages_mov_to = p->mpages_mov_from = 0;
    n++;
  }
  
  if( (ids = (unsigned long *)malloc( sizeof(unsigned long) * n)) == NULL )
    err_exit("memory exhausted\n");                     /* aaaaaarrrggghhh  */

  chk_runs++;

  i = 0;
  /* walk through the mapping*/
  for(m = beg; m < end; m += PAGE_SIZE)
  {
    /* generate the checksums for every process */
    chk = 1L;
    for(p = pl; p != NULL; p = p->next)
    {
      mm_chk.pid = p->pid;
      mm_chk.addr = m;

      if(ioctl(mergemod_fd, MERGEMEM_GEN_CHECKSUM, &mm_chk) == MERGEMEM_SUCCESS)
      {
	p->chksum = mm_chk.chksum;
	p->nr_of_refs = mm_chk.nrefs;
      }
      else 
      {
	p->chksum = chk++;
	p->nr_of_refs = 0;
      }

    }

    nids = 0;
    /* walk through the processes */
    for(p = pl; p != NULL; p = p->next)
    {
      /* look if we found the checksum already */
      fnd = FALSE;
      for(j = 0; j < nids; j++)
	if(ids[j] == p->chksum)
	{
	  fnd = TRUE;
	  p->id = j;
	  break;
	}
      /* if we didnt find it ... */
      if(!fnd)
      {
	ids[nids] = p->chksum;
	p->id = nids;
	nids++;
      }
    }

    /* 
       now all pid structs have set the id var to 0,1,2,3... according to the first second... 
       checksum found. all pids with the same id can be merged 
    */

    for(j = 0; j < nids; j++)
    {
      fp = NULL;
      fpnew = TRUE;

      /* find optimal fp */
      cs_count=0;
      fp_maps=0;
      for(p = pl; p != NULL; p = p->next)
      {
	if( p->id == j )
	{
	  cs_count++;
	  if(fp)
          {
	    if(p->nr_of_refs > fp_maps)
	    {
	      fp = p;
	      fp_maps = p->nr_of_refs;
	    }
	  }
	  else 
	  {
	    fp = p;
	    fp_maps = p->nr_of_refs;
	  }
	}
      }      

      if(cs_count > 1)
      {	 
	for(p = pl; p != NULL; p = p->next)
	{
	  if(p->id == j)
	  {
	    if(fp != p)
	    {
	      mm_mmem.pid1 = fp->pid;
	      mm_mmem.addr1 = m;
	      mm_mmem.pid2 = p->pid;
	      mm_mmem.addr2 = m;
	      switch(ioctl(mergemod_fd, MERGEMEM_MERGE_MEM, &mm_mmem))
	      {
		case MERGEMEM_SUCCESS:
		  saved_pages++;
		  p->mpages_mrg++;
		  if(fpnew) fp->mpages_mrg++;
		  break;
		case MERGEMEM_ALREADYSH:
		  p->mpages_ashr++;
		  if(fpnew) fp->mpages_ashr++;
		  break;
		case MERGEMEM_MOVED:
		  p->mpages_mov_from++;
		  fp->mpages_mov_to++;
		  break;
		default:
		  p->mpages_err++;
		  if(fpnew) fp->mpages_err++;
	      }
	      fpnew = FALSE;
	    }
	  }
	}
      }
    }

    i++;
  }
  
  for( p = pl; p != NULL; p = p->next )
  {
    p->mpages_nmrg = i - (p->mpages_mrg + p->mpages_ashr + p->mpages_err + p->mpages_mov_from );
    logmsg(3, " %5d: %3dp merged, %3dp shared, %3dp misses, %3dp not merged, m:%3dp t, %3dp f\n",
	   p->pid, p->mpages_mrg, p->mpages_ashr, p->mpages_err, p->mpages_nmrg,p->mpages_mov_to,p->mpages_mov_from);
  }

  free( ids );
}

void check_running()
{
  DIR *d;
  struct dirent *de;
  int pid, self;
  FILE *f;
  char path[PATH_MAX+1], comm[1024];
  int rc;

  self = getpid();

  if( !( d = opendir( "/proc" ) ) )
    err_exit("could not read /proc. kernel proc support required...\n");

  while( ( de = readdir( d ) ) )
  {
    if( !( pid = atoi( de->d_name ) ) || pid == self )
      continue;

    sprintf( path, "/proc/%d/stat", pid );
    if( !( f = fopen( path, "r" ) ) )
      continue;

    rc = fscanf( f, "%*d (%[^)]", comm ) == 1;
    fclose( f );

    if( !rc )
      continue;

    if( strcmp( comm, "mergemem" ) == 0 )
    {
      printf( "mergemem is already running as pid %d. exiting...\n", pid );
      exit( 1 );
    }
  }
  closedir( d );

}

#ifdef DEBUG
void dump_vars()
{
  struct CmdList *cl;
  struct PidList *pl;
  struct DontMerge *dm;
   
  printf("Dump-Vars:\n");
  printf("Daemon mode: %d\nLoglevel %d\nLogfile: %s\n", run_as_daemon, loglevel, logfile);
  printf("interval: %ld, mergeall: %d, onlycmd: %d, noconfig: %d,\n"
	 "logfile_set: %d, loglevel_set: %d, interval_set: %d\n",
	 interval, mergeall, onlycmd, noconfig, logfile_set, loglevel_set, interval_set);
  printf("command list:\n");
  for(cl=cmd_list; cl!=NULL; cl=cl->next)
    printf("* name: %s, interval: %ld\n", cl->cmd, cl->iv);
  printf("dontmerge list:\n");
  for(dm=dont_merge; dm!=NULL; dm=dm->next)
    printf("* name: %s\n", dm->cmd);
  printf("pid list:\n");
  for(pl=pid_list; pl!=NULL; pl=pl->next)
    printf("* pid: %d\n", pl->pid);
}
#endif
