/*
  mergeall 

  grep '000 00:00 0$' /proc/[0-9]* /maps | mergeall 10
                                  ^ remove space
  This file is part of mergemem by Philipp Richter & Philipp Reisner

  mergemem is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.
  
  mergemem is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with mergemem; see the file COPYING.  If not, write to
  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

*/


#include "mergemem.h"

/*
  Some global variables
*/
int loglevel = 2;
int nicelevel = 18;

int logmsg_was_n = TRUE;

FILE *logf = NULL;
/* for statistics */
int saved_pages;
/* fd for proc file*/
int mergemod_fd;

struct PageInfo
{
  unsigned long chksum;
  int pid;
  unsigned long vaddr;
};

int main( int argc, char **argv )
{
  int retval, modver, htblsize;
  struct PageInfo * htbl;
  int pid;
  unsigned long vm_begin, vm_end, m;
  char vm_perm[20];
  int vm_offset, dev_major, dev_minor, inode;
  struct { int saved; int alreadyshared; int moved; int other; int hashed; int hashmiss; int notequal; } pg;
  pg.saved = pg.alreadyshared = pg.moved = pg.other = pg.notequal = pg.hashed = pg.hashmiss = 0;
  if (argc != 2)
    fprintf(stderr, "mergeall kbytes"), exit(1);
  htblsize = atoi(argv[1]);
  if (! htblsize > 1)
    fprintf(stderr, "mergall kbytes; bits must be > 1, found %d",htblsize), exit(1);
  htblsize = (htblsize*1024) / sizeof(struct PageInfo);
  /* open mergemod file */
  if(setpriority(PRIO_PROCESS, 0, nicelevel))
    logmsg(1, "could not set priority.\n");
  if( (mergemod_fd = open("/dev/mergemem", O_RDONLY)) == -1 )
  {
    if(logf)
      logmsg(1, "/dev/mergemem not found. insert mergemod.o into kernel\n");
    err_exit("/dev/mergemem not found. insert mergemod.o into kernel\n");
  }
  
  /* check the version*/
  retval = ioctl( mergemod_fd, MERGEMEM_CHECK_VER, &modver );
  if(retval || modver != MOD_VERSION)
    err_exit("module incompatible with program\n");
    
  logmsg(3, "mergemod module has version %d\n", modver );
  logmsg(1, "mergemem started\n");
  htbl = calloc(htblsize, sizeof(struct PageInfo));
  if (htbl == NULL)
    fprintf(stderr, "mergeall: cannot allocate htbl of size %d",htblsize), exit(1);
  while ( (retval = scanf("/proc/%d/maps:%lx-%lx %s %x %d:%d %d\n",
			  &pid, &vm_begin, &vm_end, vm_perm, &vm_offset, &dev_major, &dev_minor, &inode ) ) != EOF )
    if (retval != 8)
      {
	fprintf(stderr, "mergall: bad input, only %d parameters match\n", retval);
	break;
      }
    else
      for (m = vm_begin; m < vm_end; m += PAGE_SIZE)
	{
	  struct mergemem_chksum mm_chk;

	  mm_chk.pid = pid;
	  mm_chk.addr = m;
	  if(ioctl(mergemod_fd, MERGEMEM_GEN_CHECKSUM, &mm_chk) == MERGEMEM_SUCCESS)
	    {
	      unsigned long hash;
	      pg.hashed++;
	      hash = mm_chk.chksum % htblsize;
	      /* logmsg(2,"hash: %d = %d %% %d\n", hash, mm_chk.chksum, htblsize); */
	      if (htbl[hash].pid == 0)
		{
		  htbl[hash].pid = pid;
		  htbl[hash].chksum = mm_chk.chksum;
		  htbl[hash].vaddr = m;
		}
	      else
		if (htbl[hash].chksum == mm_chk.chksum)
		  { /* try to merge */
		    struct mergemem_mmem mm_mmem;
		    mm_mmem.pid1 = pid;
		    mm_mmem.addr1 = m;
		    mm_mmem.pid2 = htbl[hash].pid;
		    mm_mmem.addr2 = htbl[hash].vaddr;
		    /* logmsg(2,"%d %ld + %d %ld\n",mm_mmem.pid1,mm_mmem.addr1,mm_mmem.pid2,mm_mmem.addr2); */
		    switch(ioctl(mergemod_fd, MERGEMEM_MERGE_MEM, &mm_mmem))
		      {
		      case MERGEMEM_SUCCESS:
			pg.saved++;
			break;
		      case MERGEMEM_ALREADYSH:
			pg.alreadyshared++;
			break;
		      case MERGEMEM_MOVED:
			pg.moved++;
			break;
		      case MERGEMEM_NOTEQUAL:
			pg.notequal++;
			break;
		      default:
			pg.other++;
		      }
		  }
	      else
		pg.hashmiss++;
	    }
	}
  logmsg(2,"hashed = %d, hashmiss = %d, saved = %d, alreadyshared = %d, moved = %d, notequal = %d, other = %d\n",
	 pg.hashed, pg.hashmiss, pg.saved, pg.alreadyshared, pg.moved, pg.notequal, pg.other);
  return(0);
}

/*
  print error message and abort
*/
void cfg_error( int line, const char *text )
{
  logmsg(1, "error in %s, line %d: %s\n", CFG_NAME, line, text);
  exit(1);
}


void err_exit( const char *msg )
{
  if( mergemod_fd != -1 )
    close( mergemod_fd );
  
  printf(msg);
  exit(1);
}

void ok_exit()
{
  if( mergemod_fd != -1 )
    close( mergemod_fd );

  logmsg(1, "mergemem stopped\n");
  exit(0);
}

void logmsg( int lev, char *fmt, ...)
{
  time_t t;
  char dbuf[100];

  if( loglevel >= lev )
  {
    va_list pa;
    va_start( pa, fmt );
    if(logmsg_was_n)
    {
      t = time(0);
      (size_t) strftime(dbuf, 100, "%b %d %H:%M:%S", localtime(&t));
      if(logf) fprintf( logf , "%s ", dbuf);
    }
    vfprintf( logf ? logf : stderr, fmt, pa );
    va_end( pa );
    if(logf) fflush(logf);
    logmsg_was_n = (fmt[strlen(fmt)-1] == '\n');
  }
}
