#include "mergelib.h"

#include <stdlib.h>
#include <getopt.h>

static int loglevel=MERGEMEM_LOG_STATS;
static char *logfile="/var/log/mergelib_demo";
static pid_t pid1, pid2; 
static long addr1, addr2;  

static BOOL merge_area = NO; 
static BOOL cross_merge = NO; 

static struct option long_options[] = {
    { "loglevel", 1, 0, 'v' }, 
    { "logfile", 1, 0, 'l' }, 
    { "echolog", 0, 0, 'e' },
    { "quiet", 0, 0, 'q' },
    { "test", 0, 0, 't' },
    { "ignore-errors", 0, 0, 'f' },
    { "help", 0, 0, 'h' },
    { "version", 0, 0, 'V' }, 
    { "area", 0, 0, 'r' }, 
    { "crossarea", 0, 0, 'C' }, 
    { 0, 0, 0, 0} 
};

void show_version(char *name)
{
	int version=mergemem_modversion();

	fprintf (stderr, "Hi, this is %s. \n", name); 
	if (version==-1) {
		fprintf (stderr, "[Module not insmod'ed.]\n"); 
	}

	fprintf (stderr, "I want version #%d\n", mergemem_myversion());
	fprintf (stderr, "... found version #%d\n", version);
}


void show_help(char *name)
{
	fprintf (stderr, "Usage: %s [options] pid1 addr1 pid2 addr2\n", name);
	fprintf (stderr, "For valid mapping, see /proc/[pid]/maps\n"); 
	fprintf (stderr, "addr are virtual addresses and in hex notation\n\n"); 
	
	fprintf (stderr, "  -v, --loglevel:      Set verbosity (-v1 errors only -v5 debug)\n"); 
	fprintf (stderr, "  -l, --logfile:       Specify name for logfile\n"); 
	fprintf (stderr, "  -e, --echolog:       Print diagnostic messages to stderr\n"); 
	fprintf (stderr, "  -q  --quiet:         Don't print any messages\n"); 
	fprintf (stderr, "  -t, --test:          Don't really merge pages, just look if it's possible (not supported)\n"); 
	fprintf (stderr, "  -f, --ignore-errors: Proceed when there is a fatal error\n"); 
	fprintf (stderr, "  -r, --area:          Merge pid1[addr1 addr2] pid2 [addr1 add2]"); 
	fprintf (stderr, "  -C, --crossarea:     Merge any pages in [addr1 addr2]"); 
	fprintf (stderr, "  -h  --help:          Display this message\n"); 
	fprintf (stderr, "  -V  --version:       Report version info of module\n");  
	fprintf (stderr, "\nQuestions? Comments? Subscribe to mergemem@mondoshawan.ml.org\n");
}

void parse_cmdline (int argc, char ** argv)
{
#define OPTS "v:l:eqtfrhVC"
	while (1) {
		switch (getopt_long( argc, argv, OPTS, long_options, 0 )) {
		case 'v':
			loglevel = atoi (optarg);
			break;
		case 'l': 
			logfile = malloc (strlen (optarg)+1);
			strcpy (logfile, optarg);
			break; 

		case 't': mergemem_always_probe = YES; break; 
		case 'e': mergemem_echolog = YES; break; 
		case 'q': mergemem_echolog = NO; break; 
		case 'f': mergemem_always_exit_on_error (NO); break; 
		case 'C': cross_merge = YES; 
		case 'r': merge_area = YES; break; 
		case 'h': 
			show_help(*argv); 
			exit(0); 
		case 'V': 
			show_version(*argv); 
			exit(0); 

		case -1: goto options_finished; /* like a break 2; */
		case 0: break; 	/* getopt_long set a variable */
		default:
			fprintf (stderr, "Usage: %s [options] pid1 addr1 pid2 addr2\n", *argv);
			exit (1); 
		}
	}

options_finished:
	if (optind+4 != argc) {
		fprintf (stderr, "Usage: %s [options] pid1 addr1 pid2 addr2\n", *argv);
		fprintf (stderr, "Or %s --help for help.", *argv); 
		exit (1); 
	}
	pid1=atoi(argv[optind]); 
	sscanf (argv[optind+1], "%lx", &addr1); 
	pid2=atoi(argv[optind+2]); 
	sscanf (argv[optind+3], "%lx", &addr2); 
}

void show_chksums (int pid, long addr)
{
	mergemem_log (MERGEMEM_LOG_STATS, "Checksum for PID %d [%lx]: %lx\n", 
		pid, addr, mergemem_checksum (pid, addr)); 
}

int main(int argc, char ** argv)
{
	parse_cmdline(argc, argv);
	mergemem_logtofile (logfile, loglevel); 

	show_chksums (pid1, addr1); 
	show_chksums (pid2, addr2); 
	/*	mergemem_init();  called automagically */

	if (!merge_area) {
		mergemem_merge (pid1, addr1, pid2, addr2);
	} else {
		int cnt; 
/* use addr2 as length */
		if (cross_merge) {
			 cnt=mergemem_cross_merge_area(pid1, addr1, addr2-addr1+1, pid2, 0);
		} else {
			 cnt=mergemem_merge_area(pid1, addr1, addr2-addr1+1, pid2, 0);
		}
		mergemem_log (MERGEMEM_LOG_STATS, "%d pages merged successfully.\n", cnt); 
	}

	/*	mergemem_print_stats(0); */
	mergemem_print_stats(1);
	return 0; 
}

	
	
