/*
  mergelib.c
  
  This file is part of mergemem by Philipp Richter & Philipp Reisner
  Created by Joe Object (e9226558@stud1.tuwien.ac.at).

  mergemem is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.
  
  mergemem is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with mergemem; see the file COPYING.  If not, write to
  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "mergelib.h"

#include <stdio.h>
#include <errno.h>
#include <assert.h>
#include <stdarg.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>

#include <asm/page.h>

int mergemem_errno=-1; 
static BOOL exit_on_err=NO; 
FILE *mergemem_logfile=NULL; 
int mergemem_loglevel=MERGEMEM_LOG_EVENT;
BOOL mergemem_echolog=NO; 
static BOOL mergemem_logfileerr=NO;
BOOL mergemem_always_probe=NO;

#if (__GNUC__ == 2) && (__GNUC_MINOR__ >= 5)
#define _log_err(x) log_err((x), __FUNCTION__)
#elif
#define _log_err(x) log_err((x), "get gcc")
#endif


static int fd=-1;

const char *mergemod_errmsgs[MERGEMEM_LAST_ERROR + 2] = {
  "Operation successful",
  "No such pid1",
  "No such pid2",
  "Page 1 not in memory",
  "Page 2 not in memory",
  "Pages not equal",
  "Pages shared already",
  "Page 2 mapped more than once (obsolete)",
  "Not an anonymous mapping",
  "Page locked (being swapped out)",
  "Merged, but still references to page",
     /* insert new exit codes here, the last one is reserved: */
  "Error conditions (permission, /etc ...)"
}; 

const char *mergelib_errmsgs[MERGELIB_LAST_ERROR-MERGELIB_ERROR_OFFSET+1] = {
  "mergemod not found. Try insmod mergemod.o",
  "mergemod and my versions differ.\n"
    "Please get the correct version from mondoshawan.ml.org/mergemem",
  "aehh ... someone forgot to mergemem_init(), or it failed, right?", 
  "some system error."
}; 

/* We have two of them, since one might be interested in 'local' stats (like
   how many pages merged for a particular process) as well as in 'global'
   ones.
   In addition, when just probing only the *local* stats are updated. 
   The global ones always reflect pages that are really merged. */

int mergemem_stats[2][MERGEMEM_LAST_ERROR+2];
char mergemem_stattimes[2][20]; 

const char *stat_str[] = {"Global", "Local"}; 

const char *loglevel_str[] = {
"Messg",
"Error", 
"Warn ", 
"Stats",
"Info ", 
"Debug"}; 

const char *mergemem_errmsg(void)
{
	static char s[50];

	if (mergemem_errno==MERGELIB_SYS_ERROR) {
		if (errno<sys_nerr) return sys_errlist[errno];
		else return "Unknown system error."; 
	}

	if (mergemem_errno>=0 && mergemem_errno<=MERGEMEM_LAST_ERROR)
		return mergemod_errmsgs[mergemem_errno];

	if (mergemem_errno>=MERGELIB_ERROR_OFFSET && 
	    mergemem_errno<=MERGELIB_LAST_ERROR)
		return mergelib_errmsgs[mergemem_errno-MERGELIB_ERROR_OFFSET];
	else {
		sprintf (s, "Unknown mergemem error (%d)", mergemem_errno);
		return s; 
	}
	assert(0); 
}


void mergemem_perror(char *s)
{
	fprintf (stderr, "%s: %s\n", s, mergemem_errmsg()); 
}

static char * now (char * s)
{
 	time_t t; 
	static char dbuf[100]; 

 	time(&t); 
	(size_t) strftime(s?s:dbuf, 100, "%b %d %H:%M:%S", localtime(&t));
	return s?s:dbuf; 
}

void mergemem_log(unsigned int level, const char *fmt, ...)
{
	va_list pa;
	static BOOL logmsg_was_n=YES; 

	if ((mergemem_logfile==NULL) && 
	   !(mergemem_echolog || mergemem_logfileerr)) return;

	if (level <= mergemem_loglevel) {
		if(logmsg_was_n)
		{
			fprintf(mergemem_logfile, "%s [%s]: ", now(NULL), loglevel_str[level]);
    		}

		va_start( pa, fmt );
		if (mergemem_logfile) vfprintf(mergemem_logfile, fmt, pa );
		if (mergemem_echolog || mergemem_logfileerr) 
			vfprintf(stderr, fmt, pa );
		va_end( pa );
		fflush(mergemem_logfile);
		logmsg_was_n = (fmt[strlen(fmt)-1] == '\n');
	}
}


static int log_err(int error, char * func)
	/* Whereever error conditions could arise, use this -- 
	   if error==0 nothing happens. Sets mergemem_errno and
	   logs error. 
	   The mergemem_xxx (from 0 to 13 currently) are infomationals. 
	*/
{	
	int level; 

	if (error==0) return 0; 
	if (error<0) mergemem_errno=MERGELIB_SYS_ERROR; 
	else mergemem_errno=error;

	if (mergemem_errno>=MERGELIB_ERROR_OFFSET)
		level = MERGEMEM_LOG_ERROR;
	else  
		level = MERGEMEM_LOG_NOTICE;

	mergemem_log (level, "%s(): %s.\n", func, mergemem_errmsg()); 

	if (exit_on_err && level==MERGEMEM_LOG_ERROR) {
		mergemem_log (level, "Exiting with exit status 1.\n"); 
		exit(1); 
	}

	return error;
}


void mergemem_always_exit_on_error(BOOL flag)
{
	exit_on_err = flag;
}


int mergemem_modversion(void)
{
	int modver=-1;
	
	if ((fd==-1) && mergemem_init()) {
		_log_err(MERGELIB_FD_NOT_OPEN); 
		return -1;
	}
	if (ioctl(fd, MERGEMEM_CHECK_VER, &modver) != 0) {
		_log_err(MERGELIB_SYS_ERROR); 
		return -1; 
	}
 
	return modver; 
}

int mergemem_myversion(void)
{
	return MOD_VERSION;
}

const char * mergemem_myversion_info(void)
{
#ifdef VERSION_INFO
	return VERSION_INFO;
#else 
	return "No info available.";
#endif
}



static int mergemem_do_merge(pid_t pid1, long addr1, pid_t pid2, long addr2, BOOL probe)
	/* Attempt to merge those virtual mappings. Returns 0 if
	   successful */
{
	static struct mergemem_mmem mm_mmem; 	
	int ret; 

	if ((fd==-1) && mergemem_init()) return (_log_err(MERGELIB_FD_NOT_OPEN));
	if (mergemem_always_probe) probe=YES; 

	addr1 &= PAGE_MASK; 	/* well ... not *really* neccessary. */
	addr2 &= PAGE_MASK; 

	mm_mmem.pid1 = pid1;
	mm_mmem.addr1 = addr1;
	mm_mmem.pid2 = pid2;
	mm_mmem.addr2 = addr2;

	mergemem_log (MERGEMEM_LOG_EVENT, "merging pid %d 0x%08lx with pid %d 0x%08lx %s\n", pid1, addr1, pid2, addr2, probe?"[probe]":"");
	/*
	if (probe) ret=_log_err(ioctl(fd, MERGEMEM_MERGE_PROBE, &mm_mmem));
	else     */

	ret=_log_err(ioctl(fd, MERGEMEM_MERGE_MEM, &mm_mmem));

	if ((ret>=MERGEMEM_SUCCESS) && (ret<=MERGEMEM_LAST_ERROR)) {
		if (!probe || mergemem_always_probe) 
			mergemem_stats[0][ret]++;
		mergemem_stats[1][ret]++;
	} else {
		if (!probe || mergemem_always_probe) 
			mergemem_stats[0][MERGEMEM_LAST_ERROR+1]++;
		mergemem_stats[1][MERGEMEM_LAST_ERROR+1]++;
	}

	return ret; 
}


int mergemem_merge(pid_t pid1, long addr1, pid_t pid2, long addr2)
{
	return mergemem_do_merge(pid1, addr1, pid2, addr2, NO);
}

int mergemem_merge_probe(pid_t pid1, long addr1, pid_t pid2, long addr2)
{
	return mergemem_do_merge(pid1, addr1, pid2, addr2, YES);
}

int mergemem_merge_area(pid_t pid1, long addr, long length,
			pid_t pid2, long offset)
{
	int i, cnt; 
	
	if ((fd==-1) && mergemem_init()) return (_log_err(MERGELIB_FD_NOT_OPEN));
	if (length) length = ((length-1) >> PAGE_SHIFT) + 1; 
	else return 0;  

	cnt=0;
	for (i=0;i<length;++i, addr+=PAGE_SIZE) {
		if (mergemem_do_merge (pid1, addr, pid2,addr+offset,NO)==MERGEMEM_SUCCESS) ++cnt; 
	}
	return cnt; 
}


int mergemem_cross_merge_area(pid_t pid1, long addr, long length,
			      pid_t pid2, long offset)
{
	int i, j, cnt; 
	
	if ((fd==-1) && mergemem_init()) return (_log_err(MERGELIB_FD_NOT_OPEN));
	if (!length) return 0;

	cnt=0;
	for (i=0;i<=length;i+=PAGE_SIZE) {
		for (j=0;j<=length;j+=PAGE_SIZE) 
			if (mergemem_do_merge (pid1, addr+i, pid2,addr+offset+j,NO)==MERGEMEM_SUCCESS) ++cnt; 
	}
	return cnt; 
}

unsigned long mergemem_checksum (pid_t pid, long addr)
	/* Returns the checksum of page addr in process pid */
{
	struct mergemem_chksum mm_chksum; 

	if ((fd==-1) && mergemem_init()) return (_log_err(MERGELIB_FD_NOT_OPEN));
	mm_chksum.pid=pid; 
	mm_chksum.addr=addr; 
	if (_log_err(ioctl(fd, MERGEMEM_GEN_CHECKSUM, &mm_chksum))) return -1; 

	return mm_chksum.chksum;  /* maybe -1 */
}	

int mergemem_get_chksums (pid_t pid, long addr, long length, long *sums)
{
	int i, cnt; 

	if (length) length = ((length-1) >> PAGE_SHIFT) + 1; 
	else return 0;  

	for (i=addr+length, cnt=0; i>=addr; i-=PAGE_SIZE, ++cnt) {
		*sums++ = mergemem_checksum (pid, i);
	}
	return cnt; 
}
  

/*---------------------           statistics          ----------------------*/


void mergemem_reset_stats (unsigned int set)
{
	if (set>1) return;

	now (mergemem_stattimes[set]);
	bzero(&mergemem_stats[set], sizeof (mergemem_stats[set]));
}


void mergemem_log_stats (unsigned int set)
{
	int i, total;

	if (set>1) return;

	mergemem_log (MERGEMEM_LOG_STATS, "%s statistics since %s %s\n", 
		stat_str[set], mergemem_stattimes[set], 
		mergemem_always_probe?"[probe]":""); 

	total=0; 
	for (i=0;i<=MERGEMEM_LAST_ERROR+1;++i) {
		mergemem_log (MERGEMEM_LOG_STATS, "\t%s: %d\n", 
			mergemod_errmsgs[i], mergemem_stats[set][i]);  
		total+=mergemem_stats[set][i];
	}
	mergemem_log (MERGEMEM_LOG_STATS, "\tTotal merge attempts: %d\n", 
		total); 
}


void mergemem_print_stats (unsigned int set)	/* ?? upgrade to log */
{
	int i, total;

	if (set>1) return;

	fprintf (stderr, "%s statistics since %s %s\n", 
		stat_str[set], mergemem_stattimes[set], 
		mergemem_always_probe?"[probe]":""); 

	total=0; 
	for (i=0;i<=MERGEMEM_LAST_ERROR+1;++i) {
		fprintf (stderr, "\t%s: %d\n", 
			mergemod_errmsgs[i], mergemem_stats[set][i]);  
		total+=mergemem_stats[set][i];
	}
	fprintf (stderr, "\tTotal merge attempts: %d\n", 
		total); 
}


static void stat_signal (int sig)
{
	mergemem_log_stats (MERGEMEM_GLOBAL_STATS);
	signal (SIGUSR1, stat_signal);  /* POSIX 2 BSD */
}


int mergemem_init(void)
   /* Opens fd, checks version. Returns non-zero error code on failure. */

{	
	int ver; 

	mergemem_reset_stats(0); 
	mergemem_reset_stats(1); 

	if ((fd = open("/dev/mergemem", O_RDONLY)) == -1) 
		return (_log_err(MERGELIB_NO_MODULE));

	if((ver=mergemem_modversion ()) != MOD_VERSION) {
		mergemem_log (MERGEMEM_LOG_WARN, "Module version number %d, library expects version number %d\n", ver, MOD_VERSION);
		_log_err(MERGELIB_VERSION_MISMATCH); 
	/* This is NOT an error! */
	}
	return 0; 
}	

int mergemem_logtofile (char * file, unsigned int level)
{
	if (mergemem_logfile!=NULL) fclose (mergemem_logfile); 
	if (file) {
		if ((mergemem_logfile=fopen(file, "a"))==NULL) {
			perror ("opening logfile"); 
			fprintf (stderr, "mergelib: couldn't open log file %s, logging to stderr.\n", file); 
			mergemem_logfileerr=YES; 
			return -1;
		}
	} else mergemem_logfile=NULL; 

	if (level > MERGEMEM_LOG_DEBUG) level = MERGEMEM_LOG_DEBUG;
	mergemem_loglevel = level; 
	mergemem_log (MERGEMEM_LOG_MESSAGE, "Logging at level %s\n", loglevel_str[level]);
	if (fd != -1) {
		mergemem_log (MERGEMEM_LOG_WARNING, "Should call logtofile() before init(). Gotta miss importmant info.\n"); 
	}
	signal (SIGUSR1, stat_signal);

	return 0; 
}
