/* -*- c -*- */

/*
 * value.c
 *
 * chpp
 *
 * Copyright (C) 1997-1998 Mark Probst
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <assert.h>

#include "dynstring.h"

#include "value.h"

#ifdef DEBUG_VALUE_FREE
#define CHECK_VALUE_FREE(v)    assert(!(v)->isFreed)
#else
#define CHECK_VALUE_FREE(v)
#endif

value*
allocValue (void)
{
    value *theValue = (value*)malloc(sizeof(value));

#ifdef DEBUG_VALUE_FREE
    theValue->isFreed = 0;
#endif

    return theValue;
}

void
freeValue (value *theValue)
{
#ifdef DEBUG_VALUE_FREE
    theValue->isFreed = 1;
#else
    free(theValue);
#endif
}

value*
valueNewUndefined (void)
{
    value *theValue = allocValue();

    theValue->type = VALUE_UNDEFINED;

    return theValue;
}

value*
valueNewInternal (internalSet setFunc, internalGet getFunc)
{
    value *theValue = allocValue();

    theValue->type = VALUE_INTERNAL;

    theValue->v.internal.setFunc = setFunc;
    theValue->v.internal.getFunc = getFunc;

    return theValue;
}

value*
valueNewBuiltIn (builtIn function, int evalParams)
{
    value *theValue = allocValue();

    theValue->type = VALUE_BUILT_IN;

    theValue->v.builtIn.function = function;
    theValue->v.builtIn.evalParams = evalParams;

    return theValue;
}

value*
valueNewScalar (dynstring *scalar)
{
    value *theValue = allocValue();

    theValue->type = VALUE_SCALAR;

    theValue->v.scalar.scalar = dsCopy(scalar);

    return theValue;
}

value*
valueNewScalarFromCString (const char *scalar)
{
    value *theValue = allocValue();

    theValue->type = VALUE_SCALAR;

    theValue->v.scalar.scalar = dsNewFrom(scalar);

    return theValue;
}

value*
valueNewList (void)
{
    value *theValue = allocValue();

    theValue->type = VALUE_LIST;

    theValue->v.list.list = avlNew();

    return theValue;
}

value*
valueNewHash (void)
{
    value *theValue = allocValue();

    theValue->type = VALUE_HASH;

    theValue->v.hash.hash = hash_new(32);

    return theValue;
}

value*
valueNewUserDefined (int numParams, int minVarArgs, int maxVarArgs,
		     dynstring *paramNames, char metaChar, dynstring *function)
{
    value *theValue = allocValue();
    int i;

    theValue->type = VALUE_USER_DEFINED;

    theValue->v.userDefined.numParams = numParams;
    theValue->v.userDefined.minVarArgs = minVarArgs;
    theValue->v.userDefined.maxVarArgs = maxVarArgs;

    if (maxVarArgs != 0)
	++numParams;

    theValue->v.userDefined.paramNames = (dynstring*)malloc(sizeof(dynstring) * numParams);
    for (i = 0; i < numParams; ++i)
	theValue->v.userDefined.paramNames[i] = dsCopy(&paramNames[i]);

    theValue->v.userDefined.metaChar = metaChar;
    theValue->v.userDefined.function = dsCopy(function);

    return theValue;
}

value*
valueNewScalarFromValue (value *theValue)
{
    CHECK_VALUE_FREE(theValue);

    switch (theValue->type)
    {
	case VALUE_UNDEFINED :
	    return valueNewScalarFromCString("");

	case VALUE_INTERNAL :
	    return valueNewScalarFromCString("<internal>");

	case VALUE_BUILT_IN :
	    return valueNewScalarFromCString("<builtIn>");

	case VALUE_SCALAR :
	    return valueCopy(theValue);

	case VALUE_LIST :
	    return valueNewScalarFromCString("<list>");

	case VALUE_HASH :
	    return valueNewScalarFromCString("<hash>");

	case VALUE_USER_DEFINED :
	    return valueNewScalarFromCString("<userDefined>");
    }

    return 0;
}

value*
valueCopy (value *theValue)
{
    value *theCopy = allocValue();

    CHECK_VALUE_FREE(theValue);

    theCopy->type = theValue->type;

    switch (theValue->type)
    {
	case VALUE_UNDEFINED :
	case VALUE_INTERNAL :
	case VALUE_BUILT_IN :
	    *theCopy = *theValue;
	    return theCopy;

	case VALUE_SCALAR :
	    theCopy->v.scalar.scalar = dsCopy(&theValue->v.scalar.scalar);
	    return theCopy;

	case VALUE_LIST :
	    {
		avlNode *node;
		int i = 1;

		theCopy->v.list.list = avlNew();

		node = avlFirst(theValue->v.list.list, 1);
		while (node != 0)
		{
		    avlInsert(theCopy->v.list.list, valueCopy((value*)avlKey(node)), i++);
		    node = avlNext(node);
		}
	    }
	    return theCopy;
	    
	case VALUE_HASH :
	    {
		hstate state;
		value *aValue;
		char *aKey;

		theCopy->v.hash.hash = hash_new(hash_count(theValue->v.hash.hash));
		
		state = hash_state(theValue->v.hash.hash);
		while ((aValue = (value*)hash_next(&state, &aKey)) != 0)
		    hash_insert(theCopy->v.hash.hash,
				strcpy((char*)malloc(strlen(aKey) + 1), aKey),
				valueCopy(aValue));
	    }
	    return theCopy;

	case VALUE_USER_DEFINED :
	    {
		int i;

		theCopy->v.userDefined.numParams = theValue->v.userDefined.numParams;
		theCopy->v.userDefined.paramNames =
		    (dynstring*)malloc(sizeof(dynstring*) * theValue->v.userDefined.numParams);
		for (i = 0; i < theValue->v.userDefined.numParams; ++i)
		    theCopy->v.userDefined.paramNames[i] = dsCopy(&theValue->v.userDefined.paramNames[i]);
		theCopy->v.userDefined.metaChar = theValue->v.userDefined.metaChar;
		theCopy->v.userDefined.function = theValue->v.userDefined.function;
	    }
	    return theCopy;

	default :
	    assert(0);
    }

    return 0;
}

void
valueFreeContents (value *theValue)
{
    CHECK_VALUE_FREE(theValue);

    switch (theValue->type)
    {
	case VALUE_SCALAR :
	    dsFree(&theValue->v.scalar.scalar);
	    break;

	case VALUE_LIST :
	    {
		avlNode *node = avlFirst(theValue->v.list.list, 1);

		while (node != 0)
		{
		    valueFree((value*)avlKey(node));
		    node = avlNext(node);
		}

		avlFree(theValue->v.list.list);
	    }
	    break;

	case VALUE_HASH :
	    {
		hstate state = hash_state(theValue->v.hash.hash);
		value *aValue;

		do
		{
		    char *key;

		    aValue = hash_next(&state, &key);

		    if (aValue != 0)
		    {
			free(key);
			valueFree(aValue);
		    }
		} while (aValue != 0);

		hash_free(theValue->v.hash.hash);
	    }
	    break;

	case VALUE_USER_DEFINED :
	    {
		int i;

		for (i = 0; i < theValue->v.userDefined.numParams; ++i)
		    dsFree(&theValue->v.userDefined.paramNames[i]);
		free(theValue->v.userDefined.paramNames);
		dsFree(&theValue->v.userDefined.function);
	    }
	    break;
    }
}

void
valueFree (value *theValue)
{
    CHECK_VALUE_FREE(theValue);

    valueFreeContents(theValue);
    freeValue(theValue);
}

int
valueCompare (value *value1, value *value2)
{
    CHECK_VALUE_FREE(value1);
    CHECK_VALUE_FREE(value2);

    /* this should do a more sophisticated comparison */

    return strcmp(valueTransformToScalar(value1)->v.scalar.scalar.data,
		  valueTransformToScalar(value2)->v.scalar.scalar.data);
}

value*
valueAssign (value *dest, value *src, int destroy)
{
    CHECK_VALUE_FREE(dest);
    CHECK_VALUE_FREE(src);

    valueFreeContents(dest);

    if (destroy)
    {
	*dest = *src;
	freeValue(src);
    }
    else
    {
	value *copy = valueCopy(src);

	*dest = *copy;
    }

    return dest;
}

int
valueHashCount (value *theValue)
{
    CHECK_VALUE_FREE(theValue);

    assert(theValue->type == VALUE_HASH);

    return hash_count(theValue->v.hash.hash);
}

void
valueHashDefine (value *theValue, dynstring *key, value *defValue)
{
    value *hashEntry;

    CHECK_VALUE_FREE(theValue);
    CHECK_VALUE_FREE(defValue);

    assert(theValue->type == VALUE_HASH);

    hashEntry = (value*)hash_lookup(theValue->v.hash.hash, key->data);
    if (hashEntry != 0)
	valueFree(hashEntry);
    hash_insert(theValue->v.hash.hash, key->data, defValue);
}

value*
valueHashLookup (value *theValue, dynstring *key)
{
    CHECK_VALUE_FREE(theValue);

    assert(theValue->type == VALUE_HASH);

    return (value*)hash_lookup(theValue->v.hash.hash, key->data);
}

int
valueListLength (value *theValue)
{
    CHECK_VALUE_FREE(theValue);

    assert(theValue->type == VALUE_LIST);

    return avlNumElements(theValue->v.list.list);
}

value*
valueListGetElement (value *theValue, int index)
{
    CHECK_VALUE_FREE(theValue);

    assert(theValue->type == VALUE_LIST);
    assert(index >= 0 && index < avlNumElements(theValue->v.list.list));

    return (value*)avlGet(theValue->v.list.list, index + 1);
}

void
valueListSetElement (value *theValue, int index, value *defValue)
{
    CHECK_VALUE_FREE(theValue);
    CHECK_VALUE_FREE(defValue);

    assert(theValue->type == VALUE_LIST);
    assert(index >= 0);

    if (index >= avlNumElements(theValue->v.list.list))
    {
	int i = index - avlNumElements(theValue->v.list.list);

	for (; i > 0; --i)
	    avlInsert(theValue->v.list.list, 0, avlNumElements(theValue->v.list.list) + 1);

	avlInsert(theValue->v.list.list, defValue, index + 1);
    }
    else
    {
	value *listEntry;

	listEntry = (value*)avlGet(theValue->v.list.list, index + 1);
	if (listEntry != 0)
	    valueFree(listEntry);
	avlDelete(theValue->v.list.list, index + 1);
	avlInsert(theValue->v.list.list, defValue, index + 1);
    }
}

void
valueListDeleteElement (value *theValue, int index)
{
    value *element;

    CHECK_VALUE_FREE(theValue);

    assert(theValue->type == VALUE_LIST);
    assert(index >= 0 && index < avlNumElements(theValue->v.list.list));

    element = avlGet(theValue->v.list.list, index + 1);
    valueFree(element);
    avlDelete(theValue->v.list.list, index + 1);
}

void
valueListAppendList (value *list, value *appended, int destroy)
{
    int index = avlNumElements(list->v.list.list);
    avlNode *node = avlFirst(appended->v.list.list, 1);

    CHECK_VALUE_FREE(list);
    CHECK_VALUE_FREE(appended);

    while (node != 0)
    {
	value *theValue;

	if (destroy)
	    theValue = (value*)avlKey(node);
	else
	    theValue = valueCopy((value*)avlKey(node));

	avlInsert(list->v.list.list, theValue, ++index);

	node = avlNext(node);
    }
    
    if (destroy)
    {
	avlFree(appended->v.list.list);
	freeValue(appended);
    }
}

void
valueListClear (value *theValue)
{
    avlNode *node;

    CHECK_VALUE_FREE(theValue);

    assert(theValue->type == VALUE_LIST);

    node = avlFirst(theValue->v.list.list, 1);
    while (node != 0)
    {
	value *entry = (value*)avlKey(node);

	valueFree(entry);

	node = avlNext(node);
    }

    avlFree(theValue->v.list.list);
    theValue->v.list.list = avlNew();
}

value*
valueTransformToScalar (value *theValue)
{
    CHECK_VALUE_FREE(theValue);

    if (theValue->type != VALUE_SCALAR)
    {
	value *scalarValue = valueNewScalarFromValue(theValue);

	valueFreeContents(theValue);
	*theValue = *scalarValue;
    }

    return theValue;
}

const char*
cStringForValueType (int type)
{
    static char *valueTypes[] =
        {
	    "undefined", 
	    "internal",
	    "built-in",
	    "scalar",
	    "list",
	    "hash",
	    "user-defined"
	};

    return valueTypes[type];
}
