/* -*- c -*- */

/*
 * symtab.c
 *
 * chpp
 *
 * Copyright (C) 1997-1998 Mark Probst
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#include "error.h"
#include "internals.h"

#include "symtab.h"

#define HASH_PRIME           1009

scopeLevel *nearestScope;
scopeLevel *globalScope;

symbol *symbolHashTable[HASH_PRIME];

scopeLevel *freeScopes = 0;
symbol *freeSymbols = 0;

inline unsigned long
elfHash (const char *str)
{
    const unsigned char *name = (const unsigned char*)str;
    unsigned long h = 0,
	g;

    while (*name)
    {
	h = (h << 4) + *name++;
	if ((g = h & 0xf0000000))
	    h ^= g >> 24;
	h &= ~g;
    }

    return h;
}

inline scopeLevel*
allocScopeLevel (void)
{
    scopeLevel *scope;

    if (freeScopes != 0)
    {
	scope = freeScopes;
	freeScopes = freeScopes->previous;
    }
    else
	scope = (scopeLevel*)malloc(sizeof(scopeLevel));

    return scope;
}

inline void
freeScopeLevel (scopeLevel *scope)
{
    scope->previous = freeScopes;
    freeScopes = scope;
}
    
inline symbol*
allocSymbol (void)
{
    symbol *entry;

    if (freeSymbols != 0)
    {
	entry = freeSymbols;
	freeSymbols = freeSymbols->scopeNext;
    }
    else
	entry = (symbol*)malloc(sizeof(symbol));

    return entry;
}

inline void
freeSymbol (symbol *entry)
{
    entry->scopeNext = freeSymbols;
    freeSymbols = entry;
}

void
insertLocalSymbolIntoHash (symbol *entry)
{
    entry->hash = elfHash(entry->name.data) % HASH_PRIME;

    entry->hashNext = symbolHashTable[entry->hash];
    symbolHashTable[entry->hash] = entry;
}

void
insertGlobalSymbolIntoHash (symbol *entry)
{
    entry->hash = elfHash(entry->name.data) % HASH_PRIME;

    entry->hashNext = 0;
    if (symbolHashTable[entry->hash] == 0)
	symbolHashTable[entry->hash] = entry;
    else
    {
	symbol *aSymbol = symbolHashTable[entry->hash];

	while (aSymbol->hashNext != 0)
	    aSymbol = aSymbol->hashNext;
	aSymbol->hashNext = entry;
    }
}

void
removeSymbolFromHash (symbol *entry)
{
    if (symbolHashTable[entry->hash] == entry)
	symbolHashTable[entry->hash] = entry->hashNext;
    else
    {
	symbol *aSymbol = symbolHashTable[entry->hash];

	while (aSymbol->hashNext != entry)
	    aSymbol = aSymbol->hashNext;
	assert(aSymbol != 0);
	aSymbol->hashNext = entry->hashNext;
    }
}

void
initSymtab (void)
{
    int i;

    nearestScope = allocScopeLevel();
    nearestScope->firstSymbol = 0;
    nearestScope->previous = 0;
    globalScope = nearestScope;

    for (i = 0; i < HASH_PRIME; ++i)
	symbolHashTable[i] = 0;
}

symbol*
lookupValueSymbolOfType (dynstring *name, int valueType)
{
    unsigned long hash = elfHash(name->data) % HASH_PRIME;
    symbol *entry;

    for (entry = symbolHashTable[hash]; entry != 0; entry = entry->hashNext)
	if (strcmp(name->data, entry->name.data) == 0)
	{
	    assert(entry->type == SYMBOL_VALUE);

	    if (entry->value->type == valueType)
		return entry;
	    else
	    {
		issueError(ERRMAC_SYMBOL_WRONG_TYPE, name->data,
			   cStringForValueType(entry->value->type),
			   cStringForValueType(valueType));
		return 0;
	    }
	}

    return 0;
}

symbol*
lookupSymbol (dynstring *name)
{
    unsigned long hash = elfHash(name->data) % HASH_PRIME;
    symbol *entry;

    for (entry = symbolHashTable[hash]; entry != 0; entry = entry->hashNext)
	if (strcmp(name->data, entry->name.data) == 0)
	    return entry;

    return 0;
}

int
existsSymbol (dynstring *name)
{
    return lookupSymbol(name) != 0;
}

symbol*
lookupFunction (dynstring *name)
{
    unsigned long hash = elfHash(name->data) % HASH_PRIME;
    symbol *entry;

    for (entry = symbolHashTable[hash]; entry != 0; entry = entry->hashNext)
    {
	if (strcmp(name->data, entry->name.data) == 0
	    && entry->type == SYMBOL_VALUE
	    && (entry->value->type == VALUE_BUILT_IN || entry->value->type == VALUE_USER_DEFINED))
	    return entry;
    }

    return 0;
}

void
defineScalar (symbol *entry, dynstring *defValue)
{
    assert(entry->type == SYMBOL_VALUE
	   && (entry->value->type == VALUE_INTERNAL || entry->value->type == VALUE_SCALAR));

    if (entry->value->type == VALUE_SCALAR)
    {
	valueFree(entry->value);
	entry->value = valueNewScalar(defValue);
    }
    else if (entry->value->type == VALUE_INTERNAL)
    {
	value *scalarValue;

	if (entry->value->v.internal.setFunc == 0)
	{
	    issueError(ERRMAC_SET_INTERNAL, entry->name.data);
	    return;
	}

	scalarValue = valueNewScalar(defValue);
	entry->value->v.internal.setFunc(scalarValue);
	valueFree(scalarValue);
    }
}

symbol*
defineScalarInScope (scopeLevel *scope, dynstring *name, dynstring *value)
{
    symbol *entry;

    for (entry = scope->firstSymbol; entry != 0; entry = entry->scopeNext)
	if (strcmp(name->data, entry->name.data) == 0
	    && entry->type == SYMBOL_VALUE
	    && (entry->value->type == VALUE_SCALAR || entry->value->type == VALUE_INTERNAL))
	{
	    defineScalar(entry, value);
	    return entry;
	}

    entry = allocSymbol();

    entry->name = dsCopy(name);
    entry->type = SYMBOL_VALUE;
    entry->value = valueNewScalar(value);

    entry->scopeNext = scope->firstSymbol;
    scope->firstSymbol = entry;
    entry->scope = scope;

    if (scope == nearestScope)
	insertLocalSymbolIntoHash(entry);
    else if (scope == globalScope)
	insertGlobalSymbolIntoHash(entry);
    else
	assert(0);

    return entry;
}

symbol*
defineVariableInScope (scopeLevel *scope, dynstring *name, value *value)
{
    symbol *entry;

    for (entry = scope->firstSymbol; entry != 0; entry = entry->scopeNext)
	if (strcmp(name->data, entry->name.data) == 0)
	{
	    valueFree(entry->value);
	    entry->value = value;

	    return entry;
	}

    entry = allocSymbol();

    entry->name = dsCopy(name);
    entry->type = SYMBOL_VALUE;
    entry->value = value;

    entry->scopeNext = scope->firstSymbol;
    scope->firstSymbol = entry;
    entry->scope = scope;

    if (scope == nearestScope)
	insertLocalSymbolIntoHash(entry);
    else if (scope == globalScope)
	insertGlobalSymbolIntoHash(entry);
    else
	assert(0);

    return entry;
}

symbol*
defineLocalScalar (dynstring *name, dynstring *value)
{
    return defineScalarInScope(nearestScope, name, value);
}

symbol*
defineLocalVariable (dynstring *name, value *defValue)
{
    return defineVariableInScope(nearestScope, name, defValue);
}

symbol*
defineGlobalScalar (dynstring *name, dynstring *value)
{
    return defineScalarInScope(globalScope, name, value);
}

symbol*
defineScalarForCurrentScope (dynstring *name, dynstring *value)
{
    symbol *entry;

    entry = lookupValueSymbolOfType(name, VALUE_SCALAR);
    if (entry != 0)
	defineScalar(entry, value);
    else
	return defineGlobalScalar(name, value);

    return entry;
}

symbol*
defineVariableForCurrentScope (dynstring *name, value *value)
{
    symbol *entry;

    entry = lookupValueSymbolOfType(name, value->type);
    if (entry != 0)
    {
	valueFree(entry->value);
	entry->value = value;
    }
    else
	return defineVariableInScope(globalScope, name, value);

    return entry;
}

symbol*
defineGlobalInternal (dynstring *name, internalSet setFunc, internalGet getFunc)
{
    symbol *entry = allocSymbol();

    entry->name = dsCopy(name);
    entry->type = SYMBOL_VALUE;
    entry->value = valueNewInternal(setFunc, getFunc);

    entry->scopeNext = globalScope->firstSymbol;
    globalScope->firstSymbol = entry;
    entry->scope = globalScope;

    insertGlobalSymbolIntoHash(entry);

    return entry;
}

symbol*
defineGlobalBuiltIn (dynstring *name, builtIn function, int evalParams)
{
    symbol *entry = allocSymbol();

    entry->name = dsCopy(name);
    entry->type = SYMBOL_VALUE;
    entry->value = valueNewBuiltIn(function, evalParams);

    entry->scopeNext = globalScope->firstSymbol;
    globalScope->firstSymbol = entry;
    entry->scope = globalScope;

    insertGlobalSymbolIntoHash(entry);

    return entry;
}

symbol*
defineGlobalUserDefined (dynstring *name, int numArgs, int minVarArgs, int maxVarArgs,
			 dynstring *args, dynstring *value)
{
    symbol *entry = lookupFunction(name);

    if (entry != 0 && entry->type == SYMBOL_VALUE && entry->value->type == VALUE_BUILT_IN)
    {
	issueError(ERRMAC_REDEFINE_BUILT_IN, name->data);
	return 0;
    }
    else if (entry != 0)
	valueFree(entry->value);
    else
    {
	entry = allocSymbol();

	entry->name = dsCopy(name);
	entry->type = SYMBOL_VALUE;

	entry->scopeNext = globalScope->firstSymbol;
	globalScope->firstSymbol = entry;
	entry->scope = globalScope;
	
	insertGlobalSymbolIntoHash(entry);
    }

    entry->value = valueNewUserDefined(numArgs, minVarArgs, maxVarArgs, args, *metaChar, value);

    return entry;
}

void
undefineSymbol (symbol *entry)
{
    removeSymbolFromHash(entry);

    if (entry->scope->firstSymbol == entry)
	entry->scope->firstSymbol = entry->scopeNext;
    else
    {
	symbol *aSymbol = entry->scope->firstSymbol;

	while (aSymbol->scopeNext != entry)
	    aSymbol = aSymbol->scopeNext;

	aSymbol->scopeNext = entry->scopeNext;
    }

    dsFree(&entry->name);

    if (entry->type == SYMBOL_VALUE)
	valueFree(entry->value);

    freeSymbol(entry);
}

void
pushScopeLevel (void)
{
    scopeLevel *scope = allocScopeLevel();

    scope->firstSymbol = 0;
    scope->previous = nearestScope;
    nearestScope = scope;
}

void
popScopeLevel (void)
{
    symbol *entry = nearestScope->firstSymbol;
    scopeLevel *previous = nearestScope->previous;

    while (entry != 0)
    {
	symbol *next = entry->scopeNext;

	removeSymbolFromHash(entry);

	dsFree(&entry->name);

	if (entry->type == SYMBOL_VALUE)
	    valueFree(entry->value);

	freeSymbol(entry);

	entry = next;
    }

    freeScopeLevel(nearestScope);
    nearestScope = previous;

    assert(nearestScope != 0);
}

symbol*
lookupHash (dynstring *name, int define)
{
    symbol *entry = lookupValueSymbolOfType(name, VALUE_HASH);

    if (define && entry == 0)
    {
	value *hash = valueNewHash();

	entry = defineVariableForCurrentScope(name, hash);
    }
    else if (entry == 0)
    {
	issueError(ERRMAC_UNDEFINED, name);
	return 0;
    }

    return entry;
}

/*
void
hashDefine (symbol *entry, dynstring *key, dynstring *defValue)
{
    value *hashEntry = valueNewScalar(defValue);

    assert(entry->type == SYMBOL_VALUE && entry->value->type == VALUE_HASH);

    valueHashDefine(entry->value, key, hashEntry);
}

dynstring*
hashLookup (symbol *entry, dynstring *key)
{
    value *hashEntry;

    assert(entry->type == SYMBOL_VALUE && entry->value->type == VALUE_HASH);

    hashEntry = valueHashLookup(entry->value, key);

    return &hashEntry->v.scalar.scalar;
}
*/

symbol*
lookupList (dynstring *name, int define)
{
    symbol *entry = lookupValueSymbolOfType(name, VALUE_LIST);

    if (define && entry == 0)
    {
	value *list = valueNewList();

	entry = defineVariableForCurrentScope(name, list);
    }
    else if (entry == 0)
    {
	issueError(ERRMAC_UNDEFINED, name);
	return 0;
    }

    return entry;
}

int
listLength (symbol *entry)
{
    assert(entry->type == SYMBOL_VALUE && entry->value->type == VALUE_LIST);

    return valueListLength(entry->value);
}

dynstring*
listGetElement (symbol *entry, int index)
{
    value *listEntry;

    assert(entry->type == SYMBOL_VALUE && entry->value->type == VALUE_LIST);
    assert(index >= 0 && index < valueListLength(entry->value));

    listEntry = valueListGetElement(entry->value, index);

    assert(listEntry->type == VALUE_SCALAR);

    return &listEntry->v.scalar.scalar;
}

void
listSetElement (symbol *entry, int index, dynstring *defValue)
{
    value *listEntry;

    assert(entry->type == SYMBOL_VALUE && entry->value->type == VALUE_LIST);
    assert(index >= 0);

    listEntry = valueNewScalar(defValue);
    valueListSetElement(entry->value, index, listEntry);
}

void
listDeleteElement (symbol *entry, int index)
{
    assert(entry->type == SYMBOL_VALUE && entry->value->type == VALUE_LIST);
    assert(index >= 0 && index < valueListLength(entry->value));

    valueListDeleteElement(entry->value, index);
}

void
listClear (symbol *entry)
{
    assert(entry->type == SYMBOL_VALUE && entry->value->type == VALUE_LIST);

    valueListClear(entry->value);
}
