/* -*- c -*- */

/*
 * macros.c
 *
 * chpp
 *
 * Copyright (C) 1997-1998 Mark Probst
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "dynstring.h"
#include "input.h"
#include "symtab.h"
#include "internals.h"
#include "error.h"
#include "arith.h"
#include "main.h"

#include "macros.h"

void
eatUntil (inputReader *ir, const char *delimiters, outputWriter *ow)
{
    dynstring ds;
    int consume = 1;

    do
    {
	if (consume)
	    if (NEXT_CHAR(ir) == EOF) return;
	
	if (strchr(delimiters, THIS_CHAR(ir)))
	    return;
	
	if (consume)
	{
	    OUT_CHAR(ow, THIS_CHAR(ir));
	}
	else
	    consume = 1;

	if (THIS_CHAR(ir) == *metaChar)
	{
	    if (NEXT_CHAR(ir) == EOF) return;

	    OUT_CHAR(ow, THIS_CHAR(ir));
	    switch (THIS_CHAR(ir))
	    {
		case '<' :
		    ds = eatIntoDSUntil(ir, "([{=>");
		    OUT_STRING(ow, ds.data, ds.length);
		    if (THIS_CHAR(ir) == EOF) { dsFree(&ds); return; }
		    OUT_CHAR(ow, THIS_CHAR(ir));

		    if (THIS_CHAR(ir) == '('
			&& ds.length == 0)
		    {
			dsFree(&ds);

			eatUntil(ir, ")", ow);
			if (THIS_CHAR(ir) == EOF) return;
			OUT_CHAR(ow, THIS_CHAR(ir));
			if (NEXT_CHAR(ir) == EOF) return;
			OUT_CHAR(ow, THIS_CHAR(ir));

			if (strchr("([{=>", THIS_CHAR(ir)) == 0)
			    assert(0);
		    }
		    else
			dsFree(&ds);

		    while (THIS_CHAR(ir) != '>')
		    {
			if (THIS_CHAR(ir) == '=')
			{
			    eatUntil(ir, ">", ow);
			    if (THIS_CHAR(ir) == EOF) return;
			    OUT_CHAR(ow, THIS_CHAR(ir));
			}
			else if (THIS_CHAR(ir) == '(')
			{
			    while (1)
			    {
				eatUntil(ir, ",)", ow);
				if (THIS_CHAR(ir) == EOF) return;
				OUT_CHAR(ow, THIS_CHAR(ir));
				if (THIS_CHAR(ir) == ')') break;
			    }
			    if (NEXT_CHAR(ir) == EOF) return;
			    if (THIS_CHAR(ir) != '>')
				issueError(ERRMAC_UNMATCHED_MACROCALL);
			    OUT_CHAR(ow, THIS_CHAR(ir));
			}
			else if (THIS_CHAR(ir) == '[')
			{
			    eatUntil(ir, "]", ow);
			    if (THIS_CHAR(ir) == EOF) return;
			    OUT_CHAR(ow, THIS_CHAR(ir));
			    if (NEXT_CHAR(ir) == EOF) return;
			    OUT_CHAR(ow, THIS_CHAR(ir));
			}
			else if (THIS_CHAR(ir) == '{')
			{
			    eatUntil(ir, "}", ow);
			    if (THIS_CHAR(ir) == EOF) return;
			    OUT_CHAR(ow, THIS_CHAR(ir));
			    if (NEXT_CHAR(ir) == EOF) return;
			    OUT_CHAR(ow, THIS_CHAR(ir));
			}
			else
			{
			    issueError(ERRMAC_UNMATCHED_MACROCALL);
			    return;
			}
		    }
		    break;
		    
		case '\"' :               /* " */
		    while (1)
		    {
			if (NEXT_CHAR(ir) == EOF) return;
			OUT_CHAR(ow, THIS_CHAR(ir));
			if (THIS_CHAR(ir) == quoteChar)
			{
			    if (NEXT_CHAR(ir) == EOF) return;
			    OUT_CHAR(ow, THIS_CHAR(ir));
			}
			else if (THIS_CHAR(ir) == '\"')       /* " */
			    break;
		    }
		    break;

		case '[' :
		    eatUntil(ir, "]", ow);
		    if (THIS_CHAR(ir) == EOF) return;
		    OUT_CHAR(ow, THIS_CHAR(ir));
		    break;

		case '{' :
		    eatUntil(ir, "}", ow);
		    if (THIS_CHAR(ir) == EOF) return;
		    OUT_CHAR(ow, THIS_CHAR(ir));
		    break;

		default :
		    if (!isalnum(THIS_CHAR(ir)) && THIS_CHAR(ir) != '_')
			break;

		    ds = dsNew();
		    dsAppendChar(&ds, THIS_CHAR(ir));
		    do
		    {
			if (NEXT_CHAR(ir) == EOF) break;
			if (!isalnum(THIS_CHAR(ir)) && THIS_CHAR(ir) != '_')
			    break;
			dsAppendChar(&ds, THIS_CHAR(ir));
			OUT_CHAR(ow, THIS_CHAR(ir));
		    } while (THIS_CHAR(ir) != EOF);

		    if (THIS_CHAR(ir) == EOF) return;

		    if (strchr(delimiters, THIS_CHAR(ir))) /* is this always correct? */
			return;

		    OUT_CHAR(ow, THIS_CHAR(ir));
		    
		    if (ds.length > 0)
		    {
			symbol *entry = lookupSymbol(&ds);

			dsFree(&ds);

			if (entry != 0)
			{
			    while (1)
			    {
				if (THIS_CHAR(ir) == '(')
				{
				    do
				    {
					eatUntil(ir, ",)", ow);
					if (THIS_CHAR(ir) == EOF) return;
					OUT_CHAR(ow, THIS_CHAR(ir));
				    } while (THIS_CHAR(ir) != ')');

				    break;
				}
				else if (THIS_CHAR(ir) == '[')
				{
				    eatUntil(ir, "]", ow);
				    if (THIS_CHAR(ir) == EOF) return;
				    OUT_CHAR(ow, THIS_CHAR(ir));
				}
				else if (THIS_CHAR(ir) == '{')
				{
				    eatUntil(ir, "}", ow);
				    if (THIS_CHAR(ir) == EOF) return;
				    OUT_CHAR(ow, THIS_CHAR(ir));
				}
				else if (THIS_CHAR(ir) == *metaChar)
				{
				    consume = 0;
				    break;
				}
				else
				    break;
			    }
			}
			else if (THIS_CHAR(ir) == *metaChar)
			    consume = 0;
		    }
		    else
			dsFree(&ds);
	    }
	}
    } while (THIS_CHAR(ir) != EOF);
}

value**
parseArguments (inputReader *ir, int *numArgs, int evalParams)
{
    int allocated = 8;
    value **args = (value**)malloc(allocated * sizeof(value*));

    *numArgs = 0;

    while (1)
    {
	dynstring ds;

	if (evalParams)
	{
	    if (THIS_CHAR(ir) == EOF)
		return args;
	    ds = eatIntoDSUntil(ir, ",)");
	    dsRemoveOuterWS(&ds);
	    args[*numArgs] = evalDSIntoValue(&ds);
	    dsFree(&ds);
	}
	else
	{
	    ds = eatIntoDSUntil(ir, ",)");
	    args[*numArgs] = valueNewScalar(&ds);
	    dsFree(&ds);
	}

	if (THIS_CHAR(ir) == EOF) return args;
	++*numArgs;
	if (THIS_CHAR(ir) == ')')
	    break;
	if (*numArgs == allocated)
	{
	    allocated += 8;
	    args = (value**)realloc(args, allocated * sizeof(value*));
	}
    }

    return args;
}

void
freeArguments (int numArgs, value **args)
{
    int i;

    for (i = 0; i < numArgs; ++i)
	valueFree(args[i]);
    free(args);
}

void
callFunction (value *function, int numArgs, value **args, outputWriter *ow)
{
    if (function->type == VALUE_BUILT_IN)
	function->v.builtIn.function(numArgs, args, ow);
    else if (function->type == VALUE_USER_DEFINED)
    {
	int i;
	char *oldMetaChar = metaChar;
	char newMetaChar = function->v.userDefined.metaChar;

	if (numArgs < function->v.userDefined.numParams + function->v.userDefined.minVarArgs
	    || (numArgs > function->v.userDefined.numParams + function->v.userDefined.maxVarArgs
		&& function->v.userDefined.maxVarArgs >= 0))
	    issueError(ERRMAC_WRONG_NUM_ARGS_USER, numArgs, function->v.userDefined.numParams);
	else
	{
	    int lastParam;

	    pushScopeLevel();
	    for (i = 0; i < function->v.userDefined.numParams; ++i)
		defineLocalVariable(&function->v.userDefined.paramNames[i], valueCopy(args[i]));
	    lastParam = i;

	    if (function->v.userDefined.maxVarArgs != 0)
	    {
		value *list = valueNewList();

		for (; i < numArgs; ++i)
		    valueListSetElement(list, i - lastParam, valueCopy(args[i]));
		    
		defineLocalVariable(&function->v.userDefined.paramNames[lastParam], list);
	    }

	    metaChar = &newMetaChar;
	    evalDSIntoOW(&function->v.userDefined.function, ow);
	    metaChar = oldMetaChar;
	    popScopeLevel();
	}
    }
    else
	assert(0);
}

void
evalVariable (symbol *entry, outputWriter *ow)
{
    if (entry->value->type == VALUE_SCALAR)
    {
	OUT_STRING(ow, entry->value->v.scalar.scalar.data,
		   entry->value->v.scalar.scalar.length);
    }
    else if (entry->value->type == VALUE_INTERNAL)
	entry->value->v.internal.getFunc(ow);
    else
	assert(0);
}

dynstring
evalVariableIntoDS (symbol *entry)
{
    dynstring val = dsNew();
    outputWriter ow = owNewDynstring(&val);

    evalVariable(entry, &ow);
    owFreeDynstring(&ow);

    return val;
}

void
outValue (outputWriter *ow, value *theValue)
{
    if (theValue->type == VALUE_INTERNAL)
	theValue->v.internal.getFunc(ow);
    else
	OUT_VALUE(ow, theValue); 
}
    
void
evalUntil (inputReader *ir, const char *delimiters, outputWriter *ow)
{
    dynstring ds;
    value *currentValue,
	*valueToBeFreed = 0;
    symbol *entry;
    int consume = 1;

    do
    {
	if (consume)
	{
	    if (NEXT_CHAR(ir) == EOF) return;
	}
	else
	{
	    if (THIS_CHAR(ir) == EOF) return;
	    consume = 1;
	}

	if (delimiters != 0 && strchr(delimiters, THIS_CHAR(ir)))
	    return;

	if (THIS_CHAR(ir) == *metaChar)
	{
	    if (NEXT_CHAR(ir) == EOF) { OUT_CHAR(ow, *metaChar); return; }
	    if (THIS_CHAR(ir) == *metaChar)
	    {
		OUT_CHAR(ow, *metaChar);
	    }
	    else
	    {
		switch (THIS_CHAR(ir))
		{
		    case '<' :
			ds = evalIntoDSUntil(ir, "([{=>");

			if (THIS_CHAR(ir) == EOF) { dsFree(&ds); return; }

			if (ds.length == 0 && THIS_CHAR(ir) == '(')
			{
			    dsFree(&ds);

			    valueToBeFreed = currentValue = evalIntoValueUntil(ir, ")");

			    if (THIS_CHAR(ir) == EOF) { valueFree(currentValue); return; }
			    if (NEXT_CHAR(ir) == EOF) { valueFree(currentValue); return; }
			}
			else
			{
			    entry = lookupSymbol(&ds);
			    if (entry == 0)
				currentValue = 0;
			    else
			    {
				assert(entry->type == SYMBOL_VALUE);
				currentValue = entry->value;
			    }

			    if (currentValue == 0)
			    {
				if (THIS_CHAR(ir) == '=')
				{
				    value *theValue = evalIntoValueUntil(ir, ">");
			    
				    defineVariableForCurrentScope(&ds, theValue);
				}
				else
				    /* at this point we should skip until '>' */
				    issueError(ERRMAC_UNDEFINED, ds.data);
			    }

			    dsFree(&ds);

			    if (currentValue == 0)
				break;
			}

			
			while (THIS_CHAR(ir) != '>')
			{
			    if (THIS_CHAR(ir) == '=')
			    {
				if (valueToBeFreed != 0)
				    /* i guess we should skip until '>' */
				    issueError(ERRMAC_NOT_LVALUE);
				else
				{
				    value *theValue = evalIntoValueUntil(ir, ">");

				    if (currentValue->type == VALUE_INTERNAL)
				    {
					currentValue->v.internal.setFunc(theValue);
					valueFree(theValue);
				    }
				    else
					valueAssign(currentValue, theValue, 1);
				}

				currentValue = 0;
			    }
			    else if (THIS_CHAR(ir) == '{')
			    {
				dynstring index;

				index = evalIntoDSUntil(ir, "}");

				if (THIS_CHAR(ir) == EOF)
				{
				    dsFree(&index);
				    if (valueToBeFreed != 0) valueFree(valueToBeFreed);
				    return;
				}
				if (NEXT_CHAR(ir) == EOF)
				{
				    dsFree(&index);
				    if (valueToBeFreed != 0) valueFree(valueToBeFreed);
				    return;
				}

				if (THIS_CHAR(ir) == '=')
				{
				    if (valueToBeFreed != 0)
					/* i guess we should skip until '>' */
					issueError(ERRMAC_NOT_LVALUE);
				    else
				    {
					value *hashEntryValue;

					hashEntryValue = evalIntoValueUntil(ir, ">");

					if (currentValue->type != VALUE_HASH)
					{
					    issueError(ERRMAC_VALUE_WRONG_TYPE,
						       cStringForValueType(currentValue->type),
						       cStringForValueType(VALUE_HASH));
					    valueFree(hashEntryValue);
					}
					else
					    valueHashDefine(currentValue, &index,
							    hashEntryValue);
				    }

				    currentValue = 0;
				}
				else
				{
				    if (currentValue->type != VALUE_HASH)
					/* this is probably not the best way to handle
					   this case */
					issueError(ERRMAC_VALUE_WRONG_TYPE,
						   cStringForValueType(currentValue->type),
						   cStringForValueType(VALUE_HASH));
				    else
				    {
					value *hashEntry = valueHashLookup(currentValue,
									   &index);

					if (hashEntry == 0)
					    issueError(ERRMAC_KEY_NOT_IN_HASH, index.data);
					else
					    currentValue = hashEntry;
				    }
				}

				dsFree(&index);
			    }
			    else if (THIS_CHAR(ir) == '[')
			    {
				dynstring indexString;
				int index;

				indexString = evalIntoDSUntil(ir, "]");
				index = atoi(indexString.data);
				dsFree(&indexString);

				if (THIS_CHAR(ir) == EOF)
				{
				    if (valueToBeFreed != 0) valueFree(valueToBeFreed);
				    return;
				}
				if (NEXT_CHAR(ir) == EOF)
				{
				    if (valueToBeFreed != 0) valueFree(valueToBeFreed);
				    return;
				}

				if (THIS_CHAR(ir) == '=')
				{
				    if (valueToBeFreed != 0)
					/* i guess we should skip until '>' */
					issueError(ERRMAC_NOT_LVALUE);
				    else
				    {
					value *listEntryValue;

					listEntryValue = evalIntoValueUntil(ir, ">");

					if (currentValue->type != VALUE_LIST)
					{
					    issueError(ERRMAC_VALUE_WRONG_TYPE,
						       cStringForValueType(currentValue->type),
						       cStringForValueType(VALUE_LIST));
					    valueFree(listEntryValue);
					}
					else
					{
					    if (index < 0)
						issueError(ERRMAC_INDEX_OUT_OF_BOUNDS, index);
					    else
						valueListSetElement(currentValue, index,
								    listEntryValue);
					}
				    }

				    currentValue = 0;
				}
				else
				{
				    if (currentValue->type != VALUE_LIST)
					/* this is probably not the best way to handle
					   this case */
					issueError(ERRMAC_VALUE_WRONG_TYPE,
						   cStringForValueType(currentValue->type),
						   cStringForValueType(VALUE_LIST));
				    else
				    {				    
					if (index < 0 || index >= valueListLength(currentValue))
					    issueError(ERRMAC_INDEX_OUT_OF_BOUNDS, index);
					else
					    currentValue = valueListGetElement(currentValue,
									       index);
				    }
				}
			    }
			    else if (THIS_CHAR(ir) == '(')
			    {
				if (currentValue->type != VALUE_USER_DEFINED
				    && currentValue->type != VALUE_BUILT_IN)
				    /* this is probably not the best way to handle
				       this case */
				    issueError(ERRMAC_VALUE_WRONG_TYPE,
					       cStringForValueType(currentValue->type),
					       cStringForValueType(VALUE_USER_DEFINED));
				else
				{
				    value **args;
				    int numArgs,
					evalParams;

				    if (currentValue->type == VALUE_USER_DEFINED)
					evalParams = 1;
				    else
					evalParams = currentValue->v.builtIn.evalParams;

				    args = parseArguments(ir, &numArgs, evalParams);

				    if (THIS_CHAR(ir) == EOF) /* ')' */
				    {
					freeArguments(numArgs, args);
					if (valueToBeFreed != 0) valueFree(valueToBeFreed);
					return;
				    }
				    if (NEXT_CHAR(ir) == EOF) /* '>' */
				    {
					freeArguments(numArgs, args);
					if (valueToBeFreed != 0) valueFree(valueToBeFreed);
					return; 
				    }

				    if (THIS_CHAR(ir) != '>')
					issueError(ERRMAC_UNMATCHED_MACROCALL);
				    
				    callFunction(currentValue, numArgs, args, ow);

				    freeArguments(numArgs, args);

				    currentValue = 0;
				}
			    }
			    else
				assert(0);       /* for the moment */
			}

			if (currentValue != 0)
			    outValue(ow, currentValue);
			if (valueToBeFreed != 0)
			{
			    valueFree(valueToBeFreed);
			    valueToBeFreed = 0;
			}

			break;

		    case '"' :
			while (1)
			{
			    if (NEXT_CHAR(ir) == EOF) return;
			    if (THIS_CHAR(ir) == quoteChar)
			    {
				char newChar;

				if (NEXT_CHAR(ir) == EOF) return;
				newChar = THIS_CHAR(ir);
				switch (THIS_CHAR(ir))
				{
				    case 'n' :
					newChar = '\n';
					break;
				    case 't' :
					newChar = '\t';
					break;
				}
				OUT_CHAR(ow, newChar);
			    }
			    else if (THIS_CHAR(ir) == '"')
				break;
			    else
				OUT_CHAR(ow, THIS_CHAR(ir));
			} 
			break;
			
		    case '[' :
			{
			    dynstring arithValue;

			    ds = evalIntoDSUntil(ir, "]");
			    if (THIS_CHAR(ir) == EOF) { dsFree(&ds); return; }
			    arithValue = arithEvalDS(&ds);
			    dsFree(&ds);
			    OUT_STRING(ow, arithValue.data, arithValue.length);
			    dsFree(&arithValue);
			}
			break;
			
		    case '{' :
			ds = evalIntoDSUntil(ir, "}");
			if (THIS_CHAR(ir) == EOF) { dsFree(&ds); return; }
			evalDSIntoOW(&ds, ow);
			dsFree(&ds);
			break;
			
		    default :
			ds = dsNew();
			do
			{
			    if (!isalnum(THIS_CHAR(ir)) && THIS_CHAR(ir) != '_')
				break;
			    dsAppendChar(&ds, THIS_CHAR(ir));
			    if (NEXT_CHAR(ir) == EOF) break;
			} while (THIS_CHAR(ir) != EOF);
			if (ds.length == 0)
			{
			    OUT_CHAR(ow, *metaChar);
			}
			else
			{
			    symbol *entry = lookupSymbol(&ds);

			    if (entry == 0)
			    {
				OUT_CHAR(ow, *metaChar);
				OUT_STRING(ow, ds.data, ds.length);
				dsFree(&ds);
				consume = 0;
				break;
			    }

			    dsFree(&ds);

			    assert(entry->type == SYMBOL_VALUE);
			    currentValue = entry->value;

			    while (currentValue != 0)
			    {
				if (THIS_CHAR(ir) == '(')
				{
				    if (currentValue->type != VALUE_USER_DEFINED
					&& currentValue->type != VALUE_BUILT_IN)
					issueError(ERRMAC_VALUE_WRONG_TYPE,
						   cStringForValueType(currentValue->type),
						   cStringForValueType(VALUE_USER_DEFINED));
				    else
				    {
					value **args;
					int numArgs,
					    evalParams;

					if (entry->value->type == VALUE_USER_DEFINED)
					    evalParams = 1;
					else
					    evalParams = entry->value->v.builtIn.evalParams;

					args = parseArguments(ir, &numArgs, evalParams);

					if (THIS_CHAR(ir) == EOF)
					{
					    freeArguments(numArgs, args);
					    return;
					}

					callFunction(entry->value, numArgs, args, ow);

					freeArguments(numArgs, args);

					if (NEXT_CHAR(ir) == EOF) return;

					currentValue = 0;
				    }
				}
				else if (THIS_CHAR(ir) == '{')
				{
				    if (currentValue->type != VALUE_HASH)
					issueError(ERRMAC_VALUE_WRONG_TYPE,
						   cStringForValueType(currentValue->type),
						   cStringForValueType(VALUE_HASH));
				    else
				    {
					dynstring index;
					value *hashEntry;

					index = evalIntoDSUntil(ir, "}");

					if (THIS_CHAR(ir) == EOF) { dsFree(&index); return; }

					hashEntry = valueHashLookup(currentValue, &index);

					if (hashEntry == 0)
					    issueError(ERRMAC_KEY_NOT_IN_HASH, index.data);
					else
					    currentValue = hashEntry;

					dsFree(&index);

					if (NEXT_CHAR(ir) == EOF)
					{
					    outValue(ow, currentValue);
					    return;
					}
				    }
				}
				else if (THIS_CHAR(ir) == '[')
				{
				    if (currentValue->type != VALUE_LIST)
					issueError(ERRMAC_VALUE_WRONG_TYPE,
						   cStringForValueType(currentValue->type),
						   cStringForValueType(VALUE_LIST));
				    else
				    {
					dynstring indexString;
					int index;

					indexString = evalIntoDSUntil(ir, "]");
					index = atoi(indexString.data);
					dsFree(&indexString);

					if (THIS_CHAR(ir) == EOF) return;

					if (index < 0 || index >= valueListLength(currentValue))
					    issueError(ERRMAC_INDEX_OUT_OF_BOUNDS, index);
					else
					    currentValue = valueListGetElement(currentValue,
									       index);
					
					if (NEXT_CHAR(ir) == EOF)
					{
					    outValue(ow, currentValue);
					    return;
					}
				    }
				}
				else
				    break;
			    }

			    if (currentValue != 0)
				outValue(ow, currentValue);
			}
			consume = 0;
		}
	    }
	}
	else
	    OUT_CHAR(ow, THIS_CHAR(ir));
    } while (THIS_CHAR(ir) != EOF);
}

void
eval (inputReader *ir, outputWriter *ow)
{
    evalUntil(ir, 0, ow);
}

dynstring
eatDSUntil (dynstring *ds, int *pos, const char *delimiters)
{
    inputReader ir = irNewDynstring(ds, *pos);
    dynstring dsVal = dsNew();
    outputWriter ow = owNewDynstring(&dsVal);

    eatUntil(&ir, delimiters, &ow);
    *pos = ((stateDynstring*)ir.state)->pos;
    irFreeDynstring(&ir);
    owFreeDynstring(&ow);

    return dsVal;
}

dynstring
eatIntoDSUntil (inputReader *ir, const char *delimiters)
{
    dynstring val = dsNew();
    outputWriter ow = owNewDynstring(&val);

    eatUntil(ir, delimiters, &ow);
    owFreeDynstring(&ow);

    return val;
}

dynstring
evalIntoDSUntil (inputReader *ir, const char *delimiters)
{
    dynstring val = dsNew();
    outputWriter ow = owNewDynstring(&val);

    evalUntil(ir, delimiters, &ow);
    owFreeDynstring(&ow);

    return val;
}

value*
evalIntoValueUntil (inputReader *ir, const char *delimiters)
{
    value *resultValue = valueNewUndefined();
    outputWriter ow = owNewValue(resultValue);

    evalUntil(ir, delimiters, &ow);
    owFreeValue(&ow);

    if (resultValue->type == VALUE_UNDEFINED)
	valueTransformToScalar(resultValue);

    return resultValue;
}

void
evalDSIntoOW (dynstring *ds, outputWriter *ow)
{
    inputReader ir = irNewDynstring(ds, 0);

    eval(&ir, ow);
    irFreeDynstring(&ir);
}

value*
evalDSIntoValue (dynstring *ds)
{
    value *resultValue = valueNewUndefined();
    outputWriter ow = owNewValue(resultValue);

    evalDSIntoOW(ds, &ow);

    owFreeValue(&ow);

    if (resultValue->type == VALUE_UNDEFINED)
	valueTransformToScalar(resultValue);

    return resultValue;
}

dynstring
evalDS (dynstring *ds)
{
    inputReader ir = irNewDynstring(ds, 0);
    dynstring dsVal = dsNew();
    outputWriter ow = owNewDynstring(&dsVal);

    eval(&ir, &ow);
    irFreeDynstring(&ir);
    owFreeDynstring(&ow);

    return dsVal;
}
