/* -*- c -*- */

/*
 * dynstring.c
 *
 * chpp
 *
 * Copyright (C) 1997-1998 Mark Probst
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <stdio.h>

#include "dynstring.h"

int powersOfTwo[32];
char *freeStrings[32];

#define MIN_EXPONENT     6
#define MIN_SIZE         (1 << MIN_EXPONENT)

#define DBG(s)           

#define EXTENSIVE_DEBUGGING

#ifdef EXTENSIVE_DEBUGGING
void
checkForFreedString (char *string)
{
    int i;

    for (i = MIN_EXPONENT; i < 32; ++i)
    {
	char *freedString = freeStrings[i];

	while (freedString != 0)
	{
	    assert(freedString != string);
	    freedString = *(char**)freedString;
	}
    }
}
#endif

void
initDynstring (void)
{
    int i;

    for (i = 0; i < 32; ++i)
    {
	powersOfTwo[i] = 1 << i;
	freeStrings[i] = 0;
    }
}

char*
allocString (int minSize, int *exponent)
{
    char *string;

    *exponent = MIN_EXPONENT;
    while (powersOfTwo[*exponent] < minSize)
	++*exponent;
    if (freeStrings[*exponent] == 0)
	string = (char*)malloc(powersOfTwo[*exponent]);
    else
    {
	string = freeStrings[*exponent];
	freeStrings[*exponent] = *(char**)string;
    }

    DBG(fprintf(stderr, "allocing string %p\n", string));

    return string;
}

inline void
freeString (char *string, int exponent)
{
#ifdef EXTENSIVE_DEBUGGING
    checkForFreedString(string);
#endif

    *(char**)string = freeStrings[exponent];
    freeStrings[exponent] = string;

    DBG(fprintf(stderr, "freeing string %p\n", string));
}

char*
reallocString (char *string, int minSize, int *exponent)
{
    int oldExponent = *exponent;
    char *newString;

    while (powersOfTwo[*exponent] < minSize)
	++*exponent;
    if (freeStrings[*exponent] == 0)
	newString = (char*)realloc(string, powersOfTwo[*exponent]);
    else
    {
	newString = freeStrings[*exponent];
	freeStrings[*exponent] = *(char**)newString;
	memcpy(newString, string, powersOfTwo[oldExponent]);
	freeString(string, oldExponent);
    }

    DBG(fprintf(stderr, "reallocing string %p to %p\n", string, newString));

    return newString;
}

dynstring
dsNew (void)
{
    dynstring ds;

    ds.data = allocString(0, &ds.allocated);
    ds.data[0] = '\0';
    ds.length = 0;

    return ds;
}

dynstring
dsNewFrom (const char *s)
{
    dynstring ds;
    int length = strlen(s);

    ds.data = allocString(length + 1, &ds.allocated);
    strcpy(ds.data, s);
    ds.length = length;

    return ds;
}

dynstring
dsNewFromBytes (const char *s, int num)
{
    dynstring ds;

    ds.data = allocString(num + 1, &ds.allocated);
    memcpy(ds.data, s, num);
    ds.data[num] = 0;
    ds.length = num;

    return ds;
}

dynstring
dsEmpty (void)
{
    static dynstring ds;

    if (ds.data == 0)
	ds = dsNew();

    return ds;
}

void
dsFree (dynstring *ds)
{
    assert(ds->data != 0);

    freeString(ds->data, ds->allocated);
    ds->data = 0;
}

dynstring
dsCopy (dynstring *dsOrig)
{
    dynstring ds;

    assert(dsOrig->data != 0);

    ds.data = allocString(dsOrig->length + 1, &ds.allocated);
    strcpy(ds.data, dsOrig->data);
    ds.length = dsOrig->length;

    return ds;
}

void
dsAppendString (dynstring *ds, const char *s, int num)
{
    assert(ds->data != 0);

    if (ds->length + num >= powersOfTwo[ds->allocated])
	ds->data = reallocString(ds->data, ds->length + num + 1, &ds->allocated);

    strncpy(ds->data + ds->length, s, num);
    ds->length += num;
    ds->data[ds->length] = '\0';
}

void
dsAppendChar (dynstring *ds, char c)
{
    assert(ds->data != 0);

    if (ds->length + 1 >= powersOfTwo[ds->allocated])
	ds->data = reallocString(ds->data, ds->length + 2, &ds->allocated);

    ds->data[ds->length] = c;
    ds->data[++ds->length] = '\0';
}

void
dsShrinkFront (dynstring *ds, int num)
{
    assert(ds->data != 0);

    if (num > ds->length)
	num = ds->length;

    memmove(ds->data, ds->data + num, ds->length + 1 - num);
    ds->length -= num;
}

void
dsShrinkRear (dynstring *ds, int num)
{
    assert(ds->data != 0);

    if (num > ds->length)
	num = ds->length;

    ds->length -= num;
    ds->data[ds->length] = '\0';
}

void
dsRemoveRearWS (dynstring *ds)
{
  int i = ds->length;

  assert(ds->data != 0);

  if( !ds->length ) return;

  while( (i > 0) && isspace(ds->data[i-1]) )
    {
      ds->data[i-1] = 0;
      i--;
      ds->length--;
    }
}

void
dsRemoveFrontWS (dynstring *ds) 
{
  int i = 0;

  while( isspace(ds->data[i])  ) {
    i++;
  }

  if( i ) dsShrinkFront( ds, i );
}

void
dsRemoveOuterWS (dynstring *ds) 
{
  dsRemoveRearWS( ds );
  dsRemoveFrontWS( ds );
}

dynstring
dsSplitWS (dynstring *ds) 
{
  dynstring ds2;
  int i = 0;

  assert(ds->data != 0);

  while( i < (ds->length-1) && ds->data[i] != ' ' && ds->data[i] != '\t' ) i++;
  if( i == ds->length - 1 ) return ds2 = dsNewFrom( "" );

  ds2 = dsNewFrom( &ds->data[i+1] );
  dsShrinkRear( ds, ds->length - i );

  return ds2;
}

