/* -*- c -*- */

/*
 * builtins/value.c
 *
 * chpp
 *
 * Copyright (C) 1997-1998 Mark Probst
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <assert.h>

#include "../value.h"
#include "../output.h"
#include "../symtab.h"
#include "../macros.h"
#include "../error.h"
#include "../internals.h"

#include "builtins.h"

void
builtInList (int numArgs, value **args, outputWriter *ow)
{
    value *theList;

    theList = valueNewList();
    if (numArgs == 1)
    {
	dsRemoveOuterWS(&valueTransformToScalar(args[0])->v.scalar.scalar);
	if (args[0]->v.scalar.scalar.length > 0)
	    valueListSetElement(theList, 0, evalDSIntoValue(&args[0]->v.scalar.scalar));
    }
    else
    {
	int i;

	for (i = 0; i < numArgs; ++i)
	    valueListSetElement(theList, i, removeWSAndEvalValueIntoValue(args[i]));
    }

    OUT_VALUE(ow, theList);
    valueFree(theList);
}

void
builtInHash (int numArgs, value **args, outputWriter *ow)
{
    value *theHash;
    int i;

    if (!(numArgs == 1 || numArgs % 2 == 0))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "hash");
	return;
    }

    if (numArgs == 1)
    {
	if (valueTransformToScalar(args[0])->v.scalar.scalar.length > 0)
	{
	    issueError(ERRMAC_WRONG_NUM_ARGS, "hash");
	    return;
	}
    }

    theHash = valueNewHash();

    for (i = 0; i < numArgs / 2; ++i)
    {
	valueTransformToScalar(args[i * 2]);
	if (valueHashLookup(theHash, &args[i * 2]->v.scalar.scalar) != 0)
	    issueWarning(WARNMAC_DUPLICATE_HASH_KEY, args[i * 2]->v.scalar.scalar.data);
	valueHashDefine(theHash, &args[i * 2]->v.scalar.scalar,
			valueCopy(args[i * 2 + 1]));
    }

    OUT_VALUE(ow, theHash);
    valueFree(theHash);
}

void
encodeValue (value *theValue, outputWriter *ow)
{
    assert(theValue != 0);

    switch (theValue->type)
    {
	case VALUE_UNDEFINED :
	    return;

	case VALUE_INTERNAL :
	    OUT_STRING(ow, "<internal>", 10);
	    break;

	case VALUE_BUILT_IN :
	    OUT_STRING(ow, "<builtIn>", 9);
	    break;

	case VALUE_SCALAR :
	    {
		int i;

		OUT_CHAR(ow, *metaChar);
		OUT_CHAR(ow, '\"');                /* " */
		for (i = 0; i < theValue->v.scalar.scalar.length; ++i)
		{
		    char c = theValue->v.scalar.scalar.data[i];
		    
		    if (c == '\"' || c == quoteChar)        /* " */
			OUT_CHAR(ow, quoteChar);
		    OUT_CHAR(ow, c);
		}
		OUT_CHAR(ow, '\"');                         /* " */
	    }
	    break;

	case VALUE_LIST :
	    {
		int i;

		OUT_CHAR(ow, *metaChar);
		OUT_STRING(ow, "list(", 5);
		if (valueListLength(theValue) > 0)
		{
		    encodeValue(valueListGetElement(theValue, 0), ow);
		    for (i = 1; i < valueListLength(theValue); ++i)
		    {
			OUT_CHAR(ow, ',');
			encodeValue(valueListGetElement(theValue, i), ow);
		    }
		}
		OUT_CHAR(ow, ')');
	    }
	    break;

	case VALUE_HASH :
	    {
		value *aValue,
		    *keyValue;
		char *aKey;
		hstate state;

		OUT_CHAR(ow, *metaChar);
		OUT_STRING(ow, "hash(", 5);
		state = hash_state(theValue->v.hash.hash);
		if ((aValue = (value*)hash_next(&state, &aKey)) != 0)
		{
		    keyValue = valueNewScalarFromCString(aKey);
		    encodeValue(keyValue, ow);
		    valueFree(keyValue);
		    OUT_CHAR(ow, ',');
		    encodeValue(aValue, ow);
		    while ((aValue = (value*)hash_next(&state, &aKey)) != 0)
		    {
			OUT_CHAR(ow, ',');
			keyValue = valueNewScalarFromCString(aKey);
			encodeValue(keyValue, ow);
			valueFree(keyValue);
			OUT_CHAR(ow, ',');
			encodeValue(aValue, ow);
		    }
		}
		OUT_CHAR(ow, ')');
	    }
	    break;

	case VALUE_USER_DEFINED :
	    OUT_STRING(ow, "<userDefined>", 13);
	    break;

	default :
	    assert(0);
    }
}

void
builtInEncode (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "encode");
	return;
    }

    encodeValue(args[0], ow);
}

void
registerValues (void)
{
    dynstring ds;

    ds = dsNewFrom("list"); defineGlobalBuiltIn(&ds, builtInList, 0);
    ds = dsNewFrom("hash"); defineGlobalBuiltIn(&ds, builtInHash, 1);
    ds = dsNewFrom("encode"); defineGlobalBuiltIn(&ds, builtInEncode, 1);
}
