/* -*- c -*- */

/*
 * builtins/stringops.c
 *
 * chpp
 *
 * Copyright (C) 1997-1998 Mark Probst
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>

#include "../regex.h"
#include "../error.h"
#include "../macros.h"
#include "../symtab.h"

#include "builtins.h"

#define CHPP_SYNTAX_BITS    (RE_BACKSLASH_ESCAPE_IN_LISTS | RE_CHAR_CLASSES | RE_INTERVALS \
                           | RE_NEWLINE_ALT | RE_NO_BK_BRACES | RE_NO_BK_PARENS \
                           | RE_NO_BK_VBAR | RE_UNMATCHED_RIGHT_PAREN_ORD)

void
builtInMatch (int numArgs, value **args, outputWriter *ow)
{
    dynstring regContent;
    const char *error;
    struct re_pattern_buffer patternBuffer;
    struct re_registers registers;
    int result,
	i;
    char resultString[64];

    if (!(numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "match");
	return;
    }

    patternBuffer.translate = 0;
    patternBuffer.fastmap = 0;
    patternBuffer.buffer = 0;
    patternBuffer.allocated = 0;

    valueTransformToScalar(args[0]);
    error = re_compile_pattern(args[0]->v.scalar.scalar.data, args[0]->v.scalar.scalar.length,
			       &patternBuffer);
    if (error != 0)
    {
	issueError(ERRMAC_INVALID_REGEX, error);
	return;
    }

    valueTransformToScalar(args[1]);
    result = re_search(&patternBuffer,
		       args[1]->v.scalar.scalar.data,
		       args[1]->v.scalar.scalar.length, 0, args[1]->v.scalar.scalar.length,
		       &registers);

    if (result >= 0)
	for (i = 0; i < registers.num_regs; ++i)
	{
	    char regNumberString[64];
	    dynstring regVarName;

	    if (registers.start[i] >= 0)
		regContent = dsNewFromBytes(args[1]->v.scalar.scalar.data + registers.start[i],
					    registers.end[i] - registers.start[i]);
	    else
		regContent = dsNew();
	    sprintf(regNumberString, "%d", i);
	    regVarName = dsNewFrom(regNumberString);
	    defineScalarForCurrentScope(&regVarName, &regContent);
	    dsFree(&regVarName);
	    dsFree(&regContent);
	}

    regfree(&patternBuffer);

    OUT_STRING(ow, resultString, sprintf(resultString, "%d", result));
}

void
builtInGsub (int numArgs, value **args, outputWriter *ow)
{
    struct re_pattern_buffer patternBuffer;
    struct re_registers registers;
    const char *error;
    int start = 0,
	result;

    if (!(numArgs == 3))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "gsub");
	return;
    }

    patternBuffer.translate = 0;
    patternBuffer.fastmap = 0;
    patternBuffer.buffer = 0;
    patternBuffer.allocated = 0;

    valueTransformToScalar(args[1]);
    error = re_compile_pattern(args[1]->v.scalar.scalar.data, args[1]->v.scalar.scalar.length,
			       &patternBuffer);
    if (error != 0)
    {
	issueError(ERRMAC_INVALID_REGEX, error);
	return;
    }

    valueTransformToScalar(args[2]);

    do
    {
	valueTransformToScalar(args[0]);
	result = re_search(&patternBuffer,
			   args[0]->v.scalar.scalar.data,
			   args[0]->v.scalar.scalar.length,
			   start, args[0]->v.scalar.scalar.length - start,
			   &registers);

	if (result >= 0)
	{
	    OUT_STRING(ow, args[0]->v.scalar.scalar.data + start, result - start);
	    OUT_STRING(ow, args[2]->v.scalar.scalar.data, args[2]->v.scalar.scalar.length);
	    start = registers.end[0];
	}
    } while (result >= 0);

    OUT_STRING(ow, args[0]->v.scalar.scalar.data + start, args[0]->v.scalar.scalar.length - start);
}

void
builtInRemovews (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "removews");
	return;
    }

    dsRemoveOuterWS(&valueTransformToScalar(args[0])->v.scalar.scalar);

    OUT_STRING(ow, args[0]->v.scalar.scalar.data, args[0]->v.scalar.scalar.length);
}

void
builtInStrlength (int numArgs, value **args, outputWriter *ow)
{
    char lengthString[64];

    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "strlength");
	return;
    }

    OUT_STRING(ow, lengthString,
	       sprintf(lengthString, "%d",
		       valueTransformToScalar(args[0])->v.scalar.scalar.length));
}

void
builtInSubstring (int numArgs, value **args, outputWriter *ow)
{
    dynstring *string = &valueTransformToScalar(args[numArgs - 1])->v.scalar.scalar;
    int start,
	length;

    if (!(numArgs == 2 || numArgs == 3))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "substring");
	return;
    }

    if (numArgs == 2)
    {
	start = atoi(valueTransformToScalar(args[0])->v.scalar.scalar.data);

	if (start >= 0)
	{
	    if (start > string->length)
		start = string->length;
	    length = string->length - start;
	}
	else
	{
	    length = -start;
	    start = string->length + start;
	}
    }
    else
    {
	start = atoi(valueTransformToScalar(args[0])->v.scalar.scalar.data);
	length = atoi(valueTransformToScalar(args[1])->v.scalar.scalar.data);

	if (start < 0)
	    start = 0;
	else if (start > string->length)
	    start = string->length;

	if (length < 0)
	    length = -length - start;

	if (start + length > string->length)
	    length = string->length - start;

	if (length < 0)
	    length = 0;
    }

    OUT_STRING(ow, string->data + start, length);
}

void
builtInStreq (int numArgs, value **args, outputWriter *ow)
{
    char resultString[64];

    if (!(numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "streq");
	return;
    }

    OUT_STRING(ow, resultString,
	       sprintf(resultString, "%d",
		       !strcmp(valueTransformToScalar(args[0])->v.scalar.scalar.data,
			       valueTransformToScalar(args[1])->v.scalar.scalar.data)));
}

void
builtInStrcmp (int numArgs, value **args, outputWriter *ow)
{
    char resultString[64];

    if (!(numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "strcmp");
	return;
    }

    OUT_STRING(ow, resultString,
	       sprintf(resultString, "%d",
		       strcmp(valueTransformToScalar(args[0])->v.scalar.scalar.data,
			      valueTransformToScalar(args[1])->v.scalar.scalar.data)));
}

void
builtInChr (int numArgs, value **args, outputWriter *ow)
{
    char c;

    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "chr");
	return;
    }

    c = atoi(valueTransformToScalar(args[0])->v.scalar.scalar.data);
    OUT_CHAR(ow, c);
}

void
builtInNumber (int numArgs, value **args, outputWriter *ow)
{
    long number,
	base;
    char numberString[32];
    int index = 0,
	isNegative = 0;

    if (!(numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "number");
	return;
    }

    valueTransformToScalar(args[0]);
    valueTransformToScalar(args[1]);

    number = atol(args[0]->v.scalar.scalar.data);
    base = atol(args[1]->v.scalar.scalar.data);

    assert(base > 1 && base <= 36);

    if (number < 0)
    {
	isNegative = 1;
	number = -number;
    }

    do
    {
	long digit = number % base;

	if (digit < 10)
	    numberString[index++] = '0' + digit;
	else
	    numberString[index++] = 'a' + digit - 10;
	
	number = number / base;
    } while (number != 0);

    if (isNegative)
	OUT_CHAR(ow, '-');
    for (--index; index >= 0; --index)
	OUT_CHAR(ow, numberString[index]);
}

void
builtInHtmlencode (int numArgs, value **args, outputWriter *ow)
{
}

void
registerStringOps (void)
{
    dynstring ds;

    ds = dsNewFrom("match"); defineGlobalBuiltIn(&ds, builtInMatch, 1);
    ds = dsNewFrom("gsub"); defineGlobalBuiltIn(&ds, builtInGsub, 1);
    ds = dsNewFrom("removews"); defineGlobalBuiltIn(&ds, builtInRemovews, 1);
    ds = dsNewFrom("strlength"); defineGlobalBuiltIn(&ds, builtInStrlength, 1);
    ds = dsNewFrom("substring"); defineGlobalBuiltIn(&ds, builtInSubstring, 1);
    ds = dsNewFrom("streq"); defineGlobalBuiltIn(&ds, builtInStreq, 1);
    ds = dsNewFrom("strcmp"); defineGlobalBuiltIn(&ds, builtInStrcmp, 1);
    ds = dsNewFrom("chr"); defineGlobalBuiltIn(&ds, builtInChr, 1);
    ds = dsNewFrom("number"); defineGlobalBuiltIn(&ds, builtInNumber, 1);

    re_syntax_options = CHPP_SYNTAX_BITS;
}
