/* -*- c -*- */

/*
 * builtins/flowctl.c
 *
 * chpp
 *
 * Copyright (C) 1997-1998 Mark Probst
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>

#include "../error.h"
#include "../macros.h"
#include "../symtab.h"
#include "../chash.h"

#include "builtins.h"

void
builtInIf (int numArgs, value **args, outputWriter *ow)
{
    value *cond;
    int boolVal;

    if (!(numArgs == 2 || numArgs == 3))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "if");
	return;
    }

    cond = removeWSAndEvalValueIntoValue(args[0]);
    boolVal = BOOL_VALUE(cond);
    valueFree(cond);
    if (boolVal)
	removeWSAndEvalValue(args[1], ow);
    else if (numArgs == 3)
	removeWSAndEvalValue(args[2], ow);
}

void
builtInIfdef (int numArgs, value **args, outputWriter *ow)
{
    dynstring var;
    int boolVal;

    if (!(numArgs == 2 || numArgs == 3))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "ifdef");
	return;
    }

    var = removeWSAndEvalValueIntoDS(args[0]);
    boolVal = existsSymbol(&var);
    dsFree(&var);
    if (boolVal)
	removeWSAndEvalValue(args[1], ow);
    else if (numArgs == 3)
	removeWSAndEvalValue(args[2], ow);
}

void
builtInIfdefkey (int numArgs, value **args, outputWriter *ow)
{
    value *hash;
    dynstring key;
    int boolVal;

    if (!(numArgs == 3 || numArgs == 4))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "ifdefkey");
	return;
    }

    hash = removeWSAndEvalValueIntoValue(args[0]);
    if (hash->type != VALUE_HASH)
    {
	issueError(ERRMAC_VALUE_WRONG_TYPE,
		   cStringForValueType(hash->type),
		   cStringForValueType(VALUE_HASH));
	valueFree(hash);
	return;
    }

    key = removeWSAndEvalValueIntoDS(args[1]);
    boolVal = (valueHashLookup(hash, &key) != 0);
    dsFree(&key);

    valueFree(hash);

    if (boolVal)
	removeWSAndEvalValue(args[2], ow);
    else if (numArgs == 4)
	removeWSAndEvalValue(args[3], ow);
}

void
builtInFor (int numArgs, value **args, outputWriter *ow)
{
    dynstring counter,
	ds;
    int start,
	stop,
	increment = 1,
	i;
    symbol *entry;
    char numberString[32];
    dynstring value;

    if (!(numArgs == 4 || numArgs == 5))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "for");
	return;
    }

    ds = removeWSAndEvalValueIntoDS(args[1]);
    start = atoi(ds.data);
    dsFree(&ds);

    ds = removeWSAndEvalValueIntoDS(args[2]);
    stop = atoi(ds.data);
    dsFree(&ds);

    if (numArgs == 5)
    {
	ds = removeWSAndEvalValueIntoDS(args[3]);
	increment = atoi(ds.data);
	dsFree(&ds);

	if (increment == 0)
	{
	    issueError(ERRMAC_NULL_INCREMENT);
	    return;
	}
    }
    else
	if (start <= stop)
	    increment = 1;
	else
	    increment = -1;

    valueTransformToScalar(args[numArgs - 1]);
    dsRemoveOuterWS(&args[numArgs - 1]->v.scalar.scalar);

    i = start;
    sprintf(numberString, "%d", i);
    value = dsNewFrom(numberString);
    counter = removeWSAndEvalValueIntoDS(args[0]);
    entry = defineScalarForCurrentScope(&counter, &value);
    dsFree(&counter);
    dsFree(&value);

    while ((increment > 0 && i <= stop) || (increment < 0 && i >= stop))
    {
	evalDSIntoOW(&args[numArgs - 1]->v.scalar.scalar, ow);

	i = atoi(entry->value->v.scalar.scalar.data);
	i += increment;

	sprintf(numberString, "%d", i);
	value = dsNewFrom(numberString);
	defineScalar(entry, &value);
	dsFree(&value);
    }
}

void
builtInForeach (int numArgs, value **args, outputWriter *ow)
{
    value *list;

    if (!(numArgs == 3))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "foreach");
	return;
    }

    list = removeWSAndEvalValueIntoValue(args[1]);

    if (valueListLength(list) > 0)
    {
	int i;
	dynstring counter = removeWSAndEvalValueIntoDS(args[0]);

	valueTransformToScalar(args[2]);
	dsRemoveOuterWS(&args[2]->v.scalar.scalar);
    
	for (i = 0; i < valueListLength(list); ++i)
	{
	    defineVariableForCurrentScope(&counter, valueCopy(valueListGetElement(list, i)));
	    evalDSIntoOW(&args[2]->v.scalar.scalar, ow);
	}

	dsFree(&counter);
    }

    valueFree(list);
}

void
builtInForeachkey (int numArgs, value **args, outputWriter *ow)
{
    value *hash;
    dynstring counter;
    int count,
	i;
    hstate state;
    char **keys;

    if (!(numArgs == 3))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "foreachkey");
	return;
    }

    hash = removeWSAndEvalValueIntoValue(args[1]);
    if (hash->type != VALUE_HASH)
    {
	issueError(ERRMAC_VALUE_WRONG_TYPE,
		   cStringForValueType(hash->type),
		   cStringForValueType(VALUE_HASH));
	valueFree(hash);
	return;
    }
    counter = removeWSAndEvalValueIntoDS(args[0]);

    valueTransformToScalar(args[2]);
    dsRemoveOuterWS(&args[2]->v.scalar.scalar);

    count = hash_count(hash->v.hash.hash);
    keys = (char**)malloc(count * sizeof(char*));
    state = hash_state(hash->v.hash.hash);
    for (i = 0; i < count; ++i)
	hash_next(&state, &keys[i]);

    for (i = 0; i < count; ++i)
    {
	dynstring key;

	key = dsNewFrom(keys[i]);
	defineScalarForCurrentScope(&counter, &key);
	dsFree(&key);
	evalDSIntoOW(&args[2]->v.scalar.scalar, ow);
    }

    free(keys);
    dsFree(&counter);
    valueFree(hash);
}

void
builtInWhile (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "while");
	return;
    }

    valueTransformToScalar(args[0]);
    dsRemoveOuterWS(&args[0]->v.scalar.scalar);
    valueTransformToScalar(args[1]);
    dsRemoveOuterWS(&args[1]->v.scalar.scalar);

    while (1)
    {
	value *cond;
	int boolVal;

	cond = evalDSIntoValue(&args[0]->v.scalar.scalar);
	boolVal = BOOL_VALUE(cond);
	valueFree(cond);

	if (!boolVal)
	    break;

	evalDSIntoOW(&args[1]->v.scalar.scalar, ow);
    }
}

void
builtInUntil (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "until");
	return;
    }

    valueTransformToScalar(args[0]);
    dsRemoveOuterWS(&args[0]->v.scalar.scalar);
    valueTransformToScalar(args[1]);
    dsRemoveOuterWS(&args[1]->v.scalar.scalar);

    while (1)
    {
	value *cond;
	int boolVal;

	cond = evalDSIntoValue(&args[0]->v.scalar.scalar);
	boolVal = BOOL_VALUE(cond);
	valueFree(cond);

	if (boolVal)
	    break;

	evalDSIntoOW(&args[1]->v.scalar.scalar, ow);
    }
}

void
builtInDowhile (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "dowhile");
	return;
    }

    valueTransformToScalar(args[0]);
    dsRemoveOuterWS(&args[0]->v.scalar.scalar);
    valueTransformToScalar(args[1]);
    dsRemoveOuterWS(&args[1]->v.scalar.scalar);

    while (1)
    {
	value *cond;
	int boolVal;

	evalDSIntoOW(&args[0]->v.scalar.scalar, ow);

	cond = evalDSIntoValue(&args[1]->v.scalar.scalar);
	boolVal = BOOL_VALUE(cond);
	valueFree(cond);

	if (!boolVal)
	    break;
    }
}

void
builtInDountil (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "dountil");
	return;
    }

    valueTransformToScalar(args[0]);
    dsRemoveOuterWS(&args[0]->v.scalar.scalar);
    valueTransformToScalar(args[1]);
    dsRemoveOuterWS(&args[1]->v.scalar.scalar);

    while (1)
    {
	value *cond;
	int boolVal;

	evalDSIntoOW(&args[0]->v.scalar.scalar, ow);

	cond = evalDSIntoValue(&args[1]->v.scalar.scalar);
	boolVal = BOOL_VALUE(cond);
	valueFree(cond);

	if (boolVal)
	    break;
    }
}

void
registerFlowCtl (void)
{
    dynstring ds;

    ds = dsNewFrom("if"); defineGlobalBuiltIn(&ds, builtInIf, 0);
    ds = dsNewFrom("ifdef"); defineGlobalBuiltIn(&ds, builtInIfdef, 0);
    ds = dsNewFrom("ifdefkey"); defineGlobalBuiltIn(&ds, builtInIfdefkey, 0);
    ds = dsNewFrom("for"); defineGlobalBuiltIn(&ds, builtInFor, 0);
    ds = dsNewFrom("foreach"); defineGlobalBuiltIn(&ds, builtInForeach, 0);
    ds = dsNewFrom("foreachkey"); defineGlobalBuiltIn(&ds, builtInForeachkey, 0);
    ds = dsNewFrom("while"); defineGlobalBuiltIn(&ds, builtInWhile, 0);
    ds = dsNewFrom("until"); defineGlobalBuiltIn(&ds, builtInUntil, 0);
    ds = dsNewFrom("dowhile"); defineGlobalBuiltIn(&ds, builtInDowhile, 0);
    ds = dsNewFrom("dountil"); defineGlobalBuiltIn(&ds, builtInDountil, 0);
}
