/* -*- c -*- */

/*
 * builtins/fileops.c
 *
 * chpp
 *
 * Copyright (C) 1997-1998 Mark Probst
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <unistd.h>
#ifdef _NEXT
#include <libc.h>
#endif

#include "../error.h"
#include "../macros.h"
#include "../symtab.h"

#include "builtins.h"

static FILE **fileTable = 0;
static int fileTableSize = 0;
static int numOpenFiles = 0;

#define INITIAL_FILE_TABLE_SIZE    8
#define FILE_TABLE_GROW_AMOUNT     8

static int
getNewFileNum (void)
{
    int fileNum;

    if (fileTable == 0)
    {
	int i;

	fileTableSize = INITIAL_FILE_TABLE_SIZE;
	fileTable = (FILE**)malloc(fileTableSize * sizeof(FILE*));
	for (i = 0; i < fileTableSize; ++i)
	    fileTable[i] = 0;
	fileNum = 0;
    }
    else if (numOpenFiles < fileTableSize)
    {
	for (fileNum = 0; fileTable[fileNum] != 0; ++fileNum)
	    ;
    }
    else
    {
	int i;

	fileNum = fileTableSize;
	fileTableSize += FILE_TABLE_GROW_AMOUNT;
	fileTable = (FILE**)realloc(fileTable, fileTableSize * sizeof(FILE*));
	for (i = fileNum; i < fileTableSize; ++i)
	    fileTable[i] = 0;
    }

    return fileNum;
}

void
builtInOpen (int numArgs, value **args, outputWriter *ow)
{
    int fileNum;
    char fileNumString[64],
	*openMode = "r";

    if (!(numArgs == 1 || numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "fopen");
	return;
    }

    if (numArgs == 2)
    {
	valueTransformToScalar(args[1]);
	openMode = args[1]->v.scalar.scalar.data;
    }

    if (strcmp(openMode, "r") != 0 && strcmp(openMode, "w") != 0)
    {
	OUT_STRING(ow, "-1", 2);
	return;
    }

    fileNum = getNewFileNum();
    fileTable[fileNum] = fopen(valueTransformToScalar(args[0])->v.scalar.scalar.data, openMode);
    if (fileTable[fileNum] == 0)
    {
	OUT_STRING(ow, "-1", 2);
	return;
    }

    ++numOpenFiles;
    OUT_STRING(ow, fileNumString, sprintf(fileNumString, "%d", fileNum));
}

void
builtInPipe (int numArgs, value **args, outputWriter *ow)
{
    int thePipe[2];
    int fileNum;
    char fileNumString[64];

    if (!(numArgs >= 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "fpipe");
	return;
    }

    if (pipe(thePipe) == -1)
    {
	issueError(ERRMAC_CALL_FAILED, "fpipe", strerror(errno), 0);
	OUT_STRING(ow, "-1", 2);
    }

    switch (fork())
    {
	case -1 :
	    issueError(ERRMAC_CALL_FAILED, "fork", strerror(errno), 0);
	    OUT_STRING(ow, "-1", 2);

	case 0 :
	    {
		int i;
		char **execArgs = (char**)malloc((numArgs + 1) * sizeof(char*));

		close(thePipe[0]);
		dup2(thePipe[1], 1);

		for (i = 0; i < numArgs; ++i)
		    execArgs[i] = valueTransformToScalar(args[i])->v.scalar.scalar.data;
		execArgs[i] = 0;

		if (execvp(execArgs[0], execArgs) == -1)
		{
		    issueError(ERRMAC_CALL_FAILED, "execvp", strerror(errno), 0);
		    exit(1);
		}
	    }
	    break;

	default :
	    close(thePipe[1]);

	    fileNum = getNewFileNum();
	    fileTable[fileNum] = fdopen(thePipe[0], "r");

	    if (fileTable[fileNum] == 0)
	    {
		issueError(ERRMAC_CALL_FAILED, "fdopen", strerror(errno), 0);
		OUT_STRING(ow, "-1", 2);
	    }

	    break;
    }

    ++numOpenFiles;
    OUT_STRING(ow, fileNumString, sprintf(fileNumString, "%d", fileNum));
}

void
builtInClose (int numArgs, value **args, outputWriter *ow)
{
    dynstring ds;
    int fileNum;

    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "fclose");
	return;
    }

    fileNum = atoi(valueTransformToScalar(args[0])->v.scalar.scalar.data);

    if (fileNum < 0 || fileNum >= fileTableSize || fileTable[fileNum] == 0)
	issueError(ERRMAC_INVALID_MACRO_ARG, "fclose", ds.data, 0);
    else
    {
	fclose(fileTable[fileNum]);
	fileTable[fileNum] = 0;
	--numOpenFiles;
    }
}

void
builtInReadline (int numArgs, value **args, outputWriter *ow)
{
    dynstring ds;
    FILE *file;
    int fileNum,
	result;

    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "fgets");
	return;
    }

    fileNum = atoi(valueTransformToScalar(args[0])->v.scalar.scalar.data);

    if (fileNum < 0 || fileNum >= fileTableSize || fileTable[fileNum] == 0)
    {
	issueError(ERRMAC_INVALID_MACRO_ARG, "fgets", ds.data);
	return;
    }

    file = fileTable[fileNum];
    do
    {
	result = fgetc(file);

	if (result != EOF)
	{
	    OUT_CHAR(ow, result);
	    if (result == '\n')
		return;
	}
    } while (result != EOF);
}

void
builtInWrite (int numArgs, value **args, outputWriter *ow)
{
    int fileNum;

    if (!(numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "fputs");
	return;
    }

    fileNum = atoi(valueTransformToScalar(args[0])->v.scalar.scalar.data);
    if (fileNum < 0 || fileNum >= fileTableSize || fileTable[fileNum] == 0)
    {
	issueError(ERRMAC_INVALID_MACRO_ARG, "fwrite", args[0]->v.scalar.scalar.data);
	return;
    }

    valueTransformToScalar(args[1]);

    fwrite(args[1]->v.scalar.scalar.data, 1, args[1]->v.scalar.scalar.length, fileTable[fileNum]);
}

void
builtInEof (int numArgs, value **args, outputWriter *ow)
{
    dynstring ds;
    FILE *file;
    int fileNum;

    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "feof");
	return;
    }

    fileNum = atoi(valueTransformToScalar(args[0])->v.scalar.scalar.data);

    if (fileNum < 0 || fileNum >= fileTableSize || fileTable[fileNum] == 0)
    {
	issueError(ERRMAC_INVALID_MACRO_ARG, "feof", ds.data, 0);
	return;
    }

    file = fileTable[fileNum];

    if (feof(file))
    {
	OUT_CHAR(ow, '1');
    }
    else
    {
	OUT_CHAR(ow, '0');
    }
}

void
builtInStat (int numArgs, value **args, outputWriter *ow)
{
    struct stat buf;

    value *theHash = valueNewHash();

    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "fstat");
	return;
    }

    if (stat(valueTransformToScalar(args[0])->v.scalar.scalar.data, &buf) == -1)
    {
	OUT_VALUE(ow, theHash);
    }
    else
    {
	static dynstring uidName,
	    gidName,
	    sizeName,
	    blksizeName,
	    blocksName,
	    atimeName,
	    mtimeName,
	    ctimeName;
	static int haveNames = 0;

	char numberString[64];

	if (!haveNames)
	{
	    haveNames = 1;
	    uidName = dsNewFrom("uid");
	    gidName = dsNewFrom("gid");
	    sizeName = dsNewFrom("size");
	    blksizeName = dsNewFrom("blksize");
	    blocksName = dsNewFrom("blocks");
	    atimeName = dsNewFrom("atime");
	    mtimeName = dsNewFrom("mtime");
	    ctimeName = dsNewFrom("ctime");
	}

	sprintf(numberString, "%u", (unsigned int)buf.st_uid);
	valueHashDefine(theHash, &uidName, valueNewScalarFromCString(numberString));

	sprintf(numberString, "%u", (unsigned int)buf.st_gid);
	valueHashDefine(theHash, &gidName, valueNewScalarFromCString(numberString));

	sprintf(numberString, "%lu", (unsigned long)buf.st_size);
	valueHashDefine(theHash, &sizeName, valueNewScalarFromCString(numberString));

	sprintf(numberString, "%lu", (unsigned long)buf.st_blksize);
	valueHashDefine(theHash, &blksizeName, valueNewScalarFromCString(numberString));

	sprintf(numberString, "%lu", (unsigned long)buf.st_blocks);
	valueHashDefine(theHash, &blocksName, valueNewScalarFromCString(numberString));

	sprintf(numberString, "%lu", (unsigned long)buf.st_atime);
	valueHashDefine(theHash, &atimeName, valueNewScalarFromCString(numberString));

	sprintf(numberString, "%lu", (unsigned long)buf.st_mtime);
	valueHashDefine(theHash, &mtimeName, valueNewScalarFromCString(numberString));

	sprintf(numberString, "%lu", (unsigned long)buf.st_ctime);
	valueHashDefine(theHash, &ctimeName, valueNewScalarFromCString(numberString));

	OUT_VALUE(ow, theHash);
    }

    valueFree(theHash);
}

void
registerFileOps (void)
{
    dynstring ds;

    ds = dsNewFrom("fopen"), defineGlobalBuiltIn(&ds, builtInOpen, 1);
    ds = dsNewFrom("fpipe"), defineGlobalBuiltIn(&ds, builtInPipe, 1);
    ds = dsNewFrom("fclose"), defineGlobalBuiltIn(&ds, builtInClose, 1);
    ds = dsNewFrom("fgets"), defineGlobalBuiltIn(&ds, builtInReadline, 1);
    ds = dsNewFrom("fputs"), defineGlobalBuiltIn(&ds, builtInWrite, 1);
    ds = dsNewFrom("feof"), defineGlobalBuiltIn(&ds, builtInEof, 1);
    ds = dsNewFrom("fstat"), defineGlobalBuiltIn(&ds, builtInStat, 1);
}
