/* -*- c -*- */

/*
 * builtins/builtins.c
 *
 * chpp
 *
 * Copyright (C) 1997-1998 Mark Probst
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#ifdef _NEXT
#include <libc.h>
#endif

#include "../error.h"
#include "../macros.h"
#include "../value.h"
#include "../symtab.h"
#include "../main.h"
#include "../depends.h"

#include "builtins.h"

extern int generateDependencies;

void
removeWSAndEvalValue (value *val, outputWriter *ow)
{
    valueTransformToScalar(val);
    dsRemoveOuterWS(&val->v.scalar.scalar);
    evalDSIntoOW(&val->v.scalar.scalar, ow);
}

value*
removeWSAndEvalValueIntoValue (value *val)
{
    value *resultValue = valueNewUndefined();
    outputWriter ow = owNewValue(resultValue);

    removeWSAndEvalValue(val, &ow);

    owFreeValue(&ow);

    return resultValue;
}

dynstring
removeWSAndEvalValueIntoDS (value *val)
{
    dynstring result = dsNew();
    outputWriter ow = owNewDynstring(&result);

    removeWSAndEvalValue(val, &ow);
    owFreeDynstring(&ow);

    return result;
}

void
builtInDefine (int numArgs, value **args, outputWriter *ow)
{
    dynstring name;

    if (!(numArgs > 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "define");
	return;
    }

    name = removeWSAndEvalValueIntoDS(args[0]);

    {
	dynstring *macroArgs = (dynstring*)malloc((numArgs - 2) * sizeof(dynstring));
	int i;
	inputReader ir;
	outputWriter owDynstring;
	int numMacroArgs,
	    minVarArgs = 0,
	    maxVarArgs = 0;

	for (i = 0; i < numArgs - 3; ++i)
	    macroArgs[i] = removeWSAndEvalValueIntoDS(args[i + 1]);

	valueTransformToScalar(args[i + 1]);
	dsRemoveOuterWS(&args[i + 1]->v.scalar.scalar);
	ir = irNewDynstring(&args[i + 1]->v.scalar.scalar, 0);
	macroArgs[i] = dsNew();
	owDynstring = owNewDynstring(&macroArgs[i]);

	evalUntil(&ir, ":", &owDynstring);
	if (THIS_CHAR(&ir) == ':')
	{
	    dynstring ds;
	    outputWriter owNumber;

	    ds = dsNew();
	    owNumber = owNewDynstring(&ds);
	    evalUntil(&ir, ":", &owNumber);
	    owFreeDynstring(&owNumber);
	    if (ds.length == 0)
		minVarArgs = 0;
	    else
		minVarArgs = atoi(ds.data);
	    dsFree(&ds);

	    ds = dsNew();
	    owNumber = owNewDynstring(&ds);
	    evalUntil(&ir, 0, &owNumber);
	    owFreeDynstring(&owNumber);
	    if (ds.length == 0)
		maxVarArgs = -1;
	    else
		maxVarArgs = atoi(ds.data);
	    dsFree(&ds);

	    numMacroArgs = numArgs - 3;
	}
	else
	    numMacroArgs = numArgs - 2;

	owFreeDynstring(&owDynstring);
	irFreeDynstring(&ir);

	valueTransformToScalar(args[numArgs - 1]);
	dsRemoveOuterWS(&args[numArgs - 1]->v.scalar.scalar);
	defineGlobalUserDefined(&name, numMacroArgs, minVarArgs, maxVarArgs, macroArgs, 
				&args[numArgs - 1]->v.scalar.scalar);

	free(macroArgs);
    }

    dsFree(&name);
}

/*
void
builtInAdefine (int numArgs, dynstring *args, outputWriter *ow)
{
    symbol *entry;
    dynstring *value;

    assert(numArgs == 3);

    entry = lookupVariable(&args[0]);
    if (entry == 0 || entry->type == SYMBOL_INTERNAL)
    {
	dynstring empty = dsEmpty();

	entry = defineVariableForCurrentScope(&args[0], &empty);
    }

    if (entry->assocArray == 0)
	entry->assocArray = hash_new(20);

    value = (dynstring*)hash_lookup(entry->assocArray, args[1].data);
    if (value != 0)
	dsFree(value);
    else
    {
	value = (dynstring*)malloc(sizeof(dynstring));
	hash_insert(entry->assocArray, args[1].data, value);
    }
    *value = dsNewFromBytes(args[2].data, args[2].length);
}
*/

void
builtInLocals (int numArgs, value **args, outputWriter *ow)
{
    int i;

    if (!(numArgs >= 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "locals");
	return;
    }

    pushScopeLevel();

    for (i = 0; i < numArgs - 1; ++i)
    {
	dynstring var,
	    value = dsEmpty();

	var = removeWSAndEvalValueIntoDS(args[i]);

	defineLocalScalar(&var, &value);

	dsFree(&var);
    }

    removeWSAndEvalValue(args[numArgs - 1], ow);

    popScopeLevel();
}

void
builtInVoid (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "void");
	return;
    }
}

void
builtInOutputenable (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "outputenable");
	return;
    }

    toplevelOutputWriter.enabled = BOOL_VALUE(args[0]);
}

void
builtInWarning (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "warning");
	return;
    }

    valueTransformToScalar(args[0]);
    issueWarning(WARNMAC_USER_WARNING, args[0]->v.scalar.scalar.data);
}

void
builtInError (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "error");
	return;
    }

    valueTransformToScalar(args[0]);
    issueError(ERRMAC_USER_ERROR, args[0]->v.scalar.scalar.data);
}

void
builtInRandom (int numArgs, value **args, outputWriter *ow)
{
    int randomNumber;
    char randString[64];

    if (!(numArgs == 0 || numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "random");
	return;
    }

    randomNumber = random();

    if (numArgs == 1)
    {
	valueTransformToScalar(args[0]);
	randomNumber %= atoi(args[0]->v.scalar.scalar.data);
    }

    OUT_STRING(ow, randString, sprintf(randString, "%d", randomNumber));
}

void
builtInDepend (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 1 || numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "depend");
	return;
    }

    if (generateDependencies)
    {
	valueTransformToScalar(args[0]);
	if (numArgs == 1)
	    addDependency(0, &args[0]->v.scalar.scalar);
	else
	{
	    valueTransformToScalar(args[1]);
	    addDependency(&args[1]->v.scalar.scalar, &args[0]->v.scalar.scalar);
	}
    }
}

void
registerBuiltIns (void)
{
    dynstring ds;

    ds = dsNewFrom("define"); defineGlobalBuiltIn(&ds, builtInDefine, 0);
    /*    ds = dsNewFrom("adefine"); defineGlobalBuiltIn(&ds, builtInAdefine, 1);*/
    ds = dsNewFrom("locals"); defineGlobalBuiltIn(&ds, builtInLocals, 0);
    ds = dsNewFrom("void"); defineGlobalBuiltIn(&ds, builtInVoid, 1);
    ds = dsNewFrom("outputenable"); defineGlobalBuiltIn(&ds, builtInOutputenable, 1);
    ds = dsNewFrom("warning"); defineGlobalBuiltIn(&ds, builtInWarning, 1);
    ds = dsNewFrom("error"); defineGlobalBuiltIn(&ds, builtInError, 1);
    ds = dsNewFrom("random"); defineGlobalBuiltIn(&ds, builtInRandom, 1);
    ds = dsNewFrom("depend"); defineGlobalBuiltIn(&ds, builtInDepend, 1);

    registerValues();
    registerFileOps();
    registerStringOps();
    registerFlowCtl();
    registerArrayOps();
    registerHashOps();
}
