/* -*- c -*- */

/*
 * builtins/arrayops.c
 *
 * chpp
 *
 * Copyright (C) 1997-1998 Mark Probst
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <assert.h>
#include <stdlib.h>

#include "../regex.h"
#include "../error.h"
#include "../macros.h"
#include "../symtab.h"

#include "builtins.h"

void
builtInListlength (int numArgs, value **args, outputWriter *ow)
{
    char lengthString[64];

    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "listlength");
	return;
    }

    if (args[0]->type != VALUE_LIST)
	issueError(ERRMAC_VALUE_WRONG_TYPE,
		   cStringForValueType(args[0]->type),
		   cStringForValueType(VALUE_LIST));
    else
	OUT_STRING(ow, lengthString, sprintf(lengthString, "%d", valueListLength(args[0])));
}

void
builtInSplit (int numArgs, value **args, outputWriter *ow)
{
    struct re_pattern_buffer patternBuffer;
    struct re_registers registers;
    const char *error;
    int pos,
	length,
	matchPos,
	elements = 0;
    value *result = valueNewList();

    if (!(numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "split");
	return;
    }

    valueTransformToScalar(args[0]);
    valueTransformToScalar(args[1]);

    patternBuffer.translate = 0;
    patternBuffer.fastmap = 0;
    patternBuffer.buffer = 0;
    patternBuffer.allocated = 0;

    error = re_compile_pattern(args[1]->v.scalar.scalar.data,
			       args[1]->v.scalar.scalar.length, &patternBuffer);
    if (error != 0)
    {
	issueError(ERRMAC_INVALID_REGEX, error);
	return;
    }

    pos = 0;
    length = args[0]->v.scalar.scalar.length;

    do
    {
	dynstring ds;
	int elementLength;

	matchPos = re_search(&patternBuffer, args[0]->v.scalar.scalar.data + pos,
			     length, 0, length, &registers);
	if (matchPos >= 0)
	    elementLength = matchPos;
	else
	    elementLength = length;

	ds = dsNewFromBytes(args[0]->v.scalar.scalar.data + pos, elementLength);
	valueListSetElement(result, elements++, valueNewScalar(&ds));
	dsFree(&ds);

	pos += elementLength + registers.end[0] - registers.start[0];
	length -= elementLength + registers.end[0] - registers.start[0];
    } while (matchPos >= 0);

    regfree(&patternBuffer);

    OUT_VALUE(ow, result);
    valueFree(result);
}

void
builtInJoin (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "join");
	return;
    }

    if (args[1]->type != VALUE_LIST)
	issueError(ERRMAC_VALUE_WRONG_TYPE,
		   cStringForValueType(args[1]->type),
		   cStringForValueType(VALUE_LIST));
    else
    {
	int length = valueListLength(args[1]);

	if (length > 0)
	{
	    int i;

	    valueTransformToScalar(args[0]);

	    OUT_VALUE(ow, valueListGetElement(args[1], 0));
	    for (i = 1; i < length; ++i)
	    {
		OUT_STRING(ow, args[0]->v.scalar.scalar.data, args[0]->v.scalar.scalar.length);
		OUT_VALUE(ow, valueListGetElement(args[1], i));
	    }
	}
    }
}

/* we should use valueCompare() here */
#define NODE_VALUE(n)    (&(((value*)(avlKey(n)))->v.scalar.scalar))
#define CMP_NODES(n,m)   strcmp(NODE_VALUE(n)->data,NODE_VALUE(m)->data)

void
quicksortArray (avlTree *tree, avlNode *l, int lIndex, avlNode *r, int rIndex)
{
    avlNode *v, *i, *j;
    int vIndex,
	iIndex,
	jIndex;
    dynstring x,
	t;

    assert(lIndex < rIndex);

    iIndex = lIndex; i = l;
    jIndex = rIndex; j = r;

    vIndex = (lIndex + rIndex) / 2; v = avlFirst(tree, vIndex);
    x = *NODE_VALUE(v);

    do
    {
	while (strcmp(NODE_VALUE(i)->data, x.data) < 0)
	{
	    ++iIndex; i = avlNext(i);
	}
	while (strcmp(x.data, NODE_VALUE(j)->data) < 0)
	{
	    --jIndex; j = avlPrevious(j);
	}

	if (iIndex <= jIndex)
	{
	    t = *NODE_VALUE(i); *NODE_VALUE(i) = *NODE_VALUE(j); *NODE_VALUE(j) = t;

	    ++iIndex; i = avlNext(i);
	    --jIndex; j = avlPrevious(j);
	}
    } while (iIndex <= jIndex);

    if (lIndex < jIndex)
	quicksortArray(tree, l, lIndex, j, jIndex);
    if (iIndex < rIndex)
	quicksortArray(tree, i, iIndex, r, rIndex);
}

void
builtInSort (int numArgs, value **args, outputWriter *ow)
{
    int length;

    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "sort");
	return;
    }

    if (args[0]->type != VALUE_LIST)
	issueError(ERRMAC_VALUE_WRONG_TYPE,
		   cStringForValueType(args[0]->type),
		   cStringForValueType(VALUE_LIST));
    else
    {
	length = valueListLength(args[0]);
	if (length >= 2)
	    quicksortArray(args[0]->v.list.list,
			   avlFirst(args[0]->v.list.list, 1), 1,
			   avlFirst(args[0]->v.list.list, length), length);

	OUT_VALUE(ow, args[0]);
    }
}

void
builtInAppend (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 2))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "append");
	return;
    }

    if (args[0]->type != VALUE_LIST)
	issueError(ERRMAC_VALUE_WRONG_TYPE,
		   cStringForValueType(args[0]->type),
		   cStringForValueType(VALUE_LIST));
    else
    {
	valueListSetElement(args[0], valueListLength(args[0]), valueCopy(args[1]));
	OUT_VALUE(ow, args[0]);
    }
}

void
builtInUniq (int numArgs, value **args, outputWriter *ow)
{
    if (!(numArgs == 1))
    {
	issueError(ERRMAC_WRONG_NUM_ARGS, "uniq");
	return;
    }

    if (args[0]->type != VALUE_LIST)
	issueError(ERRMAC_VALUE_WRONG_TYPE,
		   cStringForValueType(args[0]->type),
		   cStringForValueType(VALUE_LIST));
    else
    {
	if (valueListLength(args[0]) > 1)
	{
	    value *aValue = valueListGetElement(args[0], 0);
	    int i = 1;

	    while (i < valueListLength(args[0]))
	    {
		value *element = valueListGetElement(args[0], i);
		
		if (valueCompare(aValue, element) == 0)
		    valueListDeleteElement(args[0], i);
		else
		{
		    aValue = element;
		    ++i;
		}
	    }
	}
	OUT_VALUE(ow, args[0]);
    }
}

void
registerArrayOps (void)
{
    dynstring ds;

    ds = dsNewFrom("listlength"); defineGlobalBuiltIn(&ds, builtInListlength, 1);
    ds = dsNewFrom("split"); defineGlobalBuiltIn(&ds, builtInSplit, 1);
    ds = dsNewFrom("join"); defineGlobalBuiltIn(&ds, builtInJoin, 1);
    ds = dsNewFrom("sort"); defineGlobalBuiltIn(&ds, builtInSort, 1);
    ds = dsNewFrom("append"); defineGlobalBuiltIn(&ds, builtInAppend, 1);
    ds = dsNewFrom("uniq"); defineGlobalBuiltIn(&ds, builtInUniq, 1);
}
