/* botForth Kernel : Mar 10, 1991  Rob Chapman */

#include "botforth.h"
#include "kernel.h"
#include "extii.h"

/* Virtual Forth Machinery */
#if defined(__GNUC__) && !defined(MSDOS)
register Cell *sp asm ("%a3"),*rp asm ("%a2");        /* Stacks */
register void (***ip)() asm ("%a4"),(**tick)() asm ("%a5");     /* Threader */
#else
Cell *sp,*rp;		/* Stack pointers */
void (***ip)(), (**tick)();	/* Indirect-threaded code pointers */
#endif

/* ==== Cell ==== */

void CELL()  /* -- n */
{
	*--sp=sizeof(Cell);
}

void CELLS()  /* n -- n' */
{
	sp[0]*=sizeof(Cell);     /* CELL * */
}

/* ==== Stacks ==== */

/* ==== Data stack primitives ==== */

void SWAP()  /* m \ n -- n \ m */
{
	*--rp = sp[1];					/* OVER >R */
	sp[1] = sp[0], sp[0] = *rp++;	/* NIP R> */
}

void DUP()  /* m -- m \ m */
{
	--sp,sp[0]=sp[1];
}

void DROP()  /* m -- */
{
	sp++;
}

/* ==== Return stack primitives ==== */

void TO_R()  /* m -- */
{
	*--rp=*sp++;
}

void R()  /* -- m */
{
	*--sp=*rp;
}

void R_FROM()  /* -- m */
{
	*--sp=*rp++;
}

/* ==== Stack control ==== */
Cell *rp0_=0;

void rp_0()  /* -- a */  /* location of bottom of return stack */
{
	*--sp=(Cell)&rp0_;
}

Cell *sp0_=0;

void sp_0()  /* -- a */  /* location of bottom of parameter stack */
{
	*--sp=(Cell)&sp0_;
}


void RP_STORE()  /* -- */
{
	rp = rp0_;
}

void SP_STORE()  /* ? -- */
{
	sp=sp0_;
}

void RDEPTH()  /* -- n */
{
	*--sp = rp0_ - rp;
}

void DEPTH()  /* -- n */
{
	*--rp = sp0_ - sp;
	*--sp = *rp++;
}

/* ==== Stack macros ==== */

void NUP()  /* m \ n -- m \ m \ n */
{
	--sp,sp[0]=sp[1];   /* DUP */
	sp[1]=sp[2];      /* >R DUP R> */
}

void NIP()  /* m \ n -- n */
{
	sp[1]=sp[0],sp++;     /* SWAP DROP */
}

void OVER()  /* m \ n -- m \ n \ m */
{
	--sp,sp[0]=sp[2];    /* NUP SWAP */
}

void TUCK()  /* m \ n -- n \ m \ n */
{
	--sp,sp[0]=sp[1];     /* DUP */
	sp[1]=sp[2];
	sp[2]=sp[0];	 /* SWAP OVER */
}

void TWO_DUP()  /* m \ n -- m \ n \ m \ n */
{
	--sp,sp[0]=sp[2];     /* OVER */
	--sp,sp[0]=sp[2];    /* OVER */
}

void TWO_DROP()  /* m \ n -- */
{
	sp+=2;	  /* DROP DROP */
}

void QUESTION_DUP()  /* n -- [n] \ n */
{
	if(sp[0])
	{	  /* DUP IF */
		--sp,sp[0]=sp[1];	  /* DUP */
	}
}

/* ==== Execution of the VFM ==== */

void EXECUTE()  /* a -- */
{
	tick=(void (**)())(*sp++);
	TestTick;(**tick)();	  /* EXECUTE */
}

/* ==== ALU ==== */

/* ==== Logic ==== */

void AND()  /* m \ n -- p */
{
	sp[1]&=sp[0],sp++;
}

void OR()  /* m \ n -- p */
{
	sp[1]|=sp[0],sp++;
}

void XOR()  /* m \ n -- p */
{
	sp[1]^=sp[0],sp++;
}

void NOT()  /* m -- n */
{
	sp[0] = ~sp[0];
}

/* ==== Shift ==== */

void TWO_STAR()  /* m -- n */
{
	sp[0]<<=1;
}

void TWO_SLASH()  /* m -- n */
{
	sp[0]=sp[0]/2;
}

void U_2_SLASH()  /* m -- n */
{
	sp[0]>>=1;
}

/* ==== Math ==== */

void PLUS()  /* m \ n -- p */
{
	sp[1]+=sp[0],sp++;
}

void MINUS()  /* m \ n -- p */
{
	sp[1]-=sp[0],sp++;
}

void NEGATE()  /* m -- n */
{
	sp[0]=-(sp[0]);	 /* NOT 1 + */
}

/* ==== signed multiply and unsigned divide ==== */

void SLASH_MOD()  /* n \ m -- remainder \ quotient */
{
	*--rp=sp[0];
	sp[0]=sp[1]/(sp[0]);
	sp[1]%=*rp++;
}

void SLASH()  /* n \ m -- quotient */
{
	sp[1]/=sp[0],sp++;       /* /MOD NIP */
}

void MOD()  /* n \ m -- remainder */
{
	sp[1]%=sp[0],sp++;     /* /MOD DROP */
}

void STAR()  /* n \ m -- p */
{
	sp[1]*=sp[0],sp++;
}

/* ==== Test ==== */

/* ==== Flags ==== */

void NO()  /* -- n */  
{
	*--sp=0;
}

void YES()  /* -- n */  
{
	*--sp=-1;
}

/* ==== Zero test ==== */

void ZERO_EQUALS()  /* n -- flag */
{
	sp[0]=sp[0]==0?-1:0;	 /* 0= */
}

void ZERO_LESS_THAN()  /* n -- flag */
{
	sp[0]=(Integer)sp[0]<0?-1:0;
}

/* ==== Compare ==== */

void EQUALS()  /* n \ m -- flag */
{
	if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;
}

void LESS_THAN()  /* n \ m -- flag */
{
	if((Integer)sp[1]<(Integer)sp[0]) *++sp=-1; else *++sp=0;
}

void GREATER_THAN()  /* n \ m -- flag */
{
	if((Integer)sp[1]>(Integer)sp[0]) *++sp=-1; else *++sp=0;
}

void U_LESS_THAN()  /* n \ m -- flag */
{
	if(sp[1]<sp[0]) *++sp=-1; else *++sp=0;
}

void U_GREATER_THAN()  /* n \ m -- flag */
{
	if(sp[1]>sp[0]) *++sp=-1; else *++sp=0;
}

/* ==== Absolute, MinMax ==== */

void ABS()  /* n -- n */
{
	if((Integer)sp[0]<0) sp[0]=-(sp[0]);	/* ABS */
}

void MAX()  /* n \ m -- p */
{
	if((Integer)sp[0]>(Integer)sp[1]) sp[1]=sp[0]; sp++;	/* MAX */
}

void MIN()  /* n \ m -- p */
{
	if((Integer)sp[0]<(Integer)sp[1]) sp[1]=sp[0]; sp++;	/* MIN */
}

/* ==== Memory ==== */

/* ==== Cells ==== */

void FETCH()  /* a -- n */
{
	sp[0]=*(Cell *)(sp[0]);
}

void STORE()  /* n \ a -- */
{
	*(Cell *)(sp[0])=sp[1];
	sp+=2;   /* 2DROP */
}

void FETCH_PLUS()  /* a -- n \ a+ */
{
	--sp, sp[0] = sp[1] + sizeof(Cell);
	sp[1] = *(Cell *)sp[1];
}

void STORE_PLUS()  /* n \ a -- a+ */
{
	*(Cell *)sp[0] = sp[1];
	sp[1] = sp[0] + sizeof(Cell), sp++;
}

void FETCH_MINUS()  /* a -- n \ a- */
{
	--sp, sp[0] = sp[1] - sizeof(Cell);
	sp[1] = *(Cell *)sp[1];
}

void STORE_MINUS()  /* n \ a -- a- */
{
	*(Cell *)sp[0] = sp[1];
	sp[1] = sp[0] - sizeof(Cell), sp++;
}

void PLUS_STORE()  /* n \ addr -- */
{
	*(Cell *)(sp[0])+=sp[1];
	sp+=2;	  /* 2DROP */     /* +! */
}

/* ==== Bytes ==== */

void C_FETCH()  /* a -- c */
{
	sp[0]=(Cell)(*(Byte *)(sp[0]));
}

void C_STORE()  /* c \ a -- */
{
	*(Byte *)(sp[0])=(Byte)(sp[1]);
	sp+=2;     /* 2DROP */
}

void C_FETCH_PLUS()  /* a -- c \ a+ */
{
	--sp, sp[0] = sp[1] + 1;
	sp[1] = (Cell)(*(Byte *)sp[1]);
}

void C_STORE_PLUS()  /* c \ a -- a+ */
{
	*(Byte *)sp[0] = (Byte)sp[1];
	sp[1] = sp[0] + 1, sp++;
}

void C_FETCH_MINUS()  /* a -- c \ a- */
{
	--sp, sp[0] = sp[1] - 1;
	sp[1] = (Cell)(*(Byte *)sp[1]);
}

void C_STORE_MINUS()  /* c \ a -- a- */
{
	*(Byte *)sp[0] = (Byte)sp[1];
	sp[1] = sp[0] - 1, sp++;
}

void C_PLUS_STORE()  /* c \ addr -- */
{
	*(Byte *)(sp[0])+=(Integer)sp[1];
	sp+=2;	  /* 2DROP */     /* C+! */
}

/* ==== Bits ==== */

void PLUS_BITS()  /* bits \ addr -- */
{
	*(Byte *)sp[0] |= sp[1];
	sp += 2;
}

void MINUS_BITS()  /* bits \ addr -- */
{
	*(Byte *)sp[0] = *(Byte *)sp[0] & ~sp[1];
	sp += 2;
}

/* ==== Short ==== */

void W_FETCH()  /* a -- n */
{
	sp[0]=(Cell)(*(unsigned short *)(sp[0]));
}

void W_STORE()  /* n \ a -- */
{
	*(unsigned short *)(sp[0])=(unsigned short)(sp[1]);
	sp+=2;	  /* 2DROP */
}

/* ==== Queues == | insert | remove | end |  data space for circular queue | */
void ZERO_Q() /* q -- */  /* initialize a queue */
{
	Queue	*queue;
	
	queue = (Queue *)*sp++;
	queue->remove = queue->insert = &queue->storage[0];
}


void Q()  /* q -- n */
{
	sp[0] = *((Queue *)sp[0])->remove;
}

void Q_QUERY() /* q -- n */  /* return the size of a queue */
{
	Queue	*queue;
	
	queue = (Queue *)sp[0];
	if ( (Integer)( sp[0] = (Integer)queue->remove - (Integer)queue->insert ) < 0 )
		sp[0] += (Cell)queue->end - (Cell)&queue->end;
	sp[0] /= sizeof(Cell);
}

void Q_FROM() /* q -- n */  /* |queued item|>s  s is transferred to the data stack */
{
	Queue	*queue;
	
	queue = (Queue *)sp[0];
	sp[0] = *queue->remove--;
	if ( (Cell)queue->remove == (Cell)&queue->end )
		queue->remove = queue->end;
}

void TO_Q() /* n \ q -- */  /* q>|ueued items|  q is transferred from the data stack */
{
	Queue	*queue;
	
	queue = (Queue *)*sp++;
	*queue->insert-- = *sp++;
	if ( (Cell)queue->insert == (Cell)&queue->end )
		queue->insert = queue->end;
}

void TO_Q_STAR()  /* n \ q -- */
{
	*--sp=0;
	*--rp=*sp++;	  /* >R */
	TO_Q();
	*--sp=*rp++;   /* R> */
	sp++;	  /* DROP */
}
/* for multi access to a queue */

/* ==== Memory tools ==== */

void BL()  /* -- n */
{
	*--sp=32;
}

void CMOVE()  /* src \ dest \ count -- */
{
	memcpy((void *)sp[1], (void *)sp[2], sp[0]);
	sp+=3;
}

void LESS_THAN_CMOVE()  /* src \ dest \ count -- */
{
	memmove((void *)sp[1], (void *)sp[2], sp[0]);
	sp+=3;
}

void MOVE()  /* src \ dest \ count -- */
{
	memcpy((void *)sp[1], (void *)sp[2], sp[0]*sizeof(Cell));
	sp+=3;
}

void FILL()  /* addr \ count \ char -- */
{
	memset((void *)sp[2], (Byte)sp[0], sp[1]);
	sp+=3;
}

void ERASE()  /* addr \ count -- */
{
	*--sp=0;
	FILL();
}

void BLANKS()  /* addr \ count -- */
{
	BL();
	FILL();
}

void DIFFER_QUERY()  /* a \ a -- a+ \ a+ \ bits */
{
	--sp;
	sp[0] = (Cell)(*(Byte *)sp[1]++ ^ *(Byte *)sp[2]++);
}

void BITS_DIFFER_QUERY()  /* a \ a \ mask -- a+ \ a+ \ bits */
{
	sp[0] &= (Cell)(*(Byte *)sp[1]++ ^ *(Byte *)sp[2]++);
}

void SAME_QUERY()  /* a \ b \ length -- flag */
{
	sp[2] = memcmp((void *)sp[2], (void *)sp[1], sp[0]);
	sp += 2;
	ZERO_EQUALS();
}

void COMPARE()  /* a \ b -- f */
{
	--sp,sp[0]=sp[1];	/* DUP */
	sp[0]=(Cell)(*(Byte *)(sp[0]));	  /* C@ */
	sp[0]+=1;	/* + */
	SAME_QUERY();
}

/* ==== Dictionary ==== */

/* ==== Memory Management ==== */
Cell dp_=0;

void dp()  /* -- a */  /* end of dictionary and start of free memory */
{
	*--sp=(Cell)&dp_;
}

Cell limit_=0;

void limit()  /* -- a */  /* end of free memory */
{
	*--sp=(Cell)&limit_;
}


void HERE()  /* -- addr */
{
	dp();
	sp[0]=*(Cell *)(sp[0]);       /* @ */
}

void ALLOT()  /* n -- */
{
	dp();
	*(Integer *)(sp[0])+=(Integer)sp[1];
	sp+=2;     /* 2DROP */     /* +! */
}

/*  char-align#  TUCK +  SWAP  NOT AND */

void ALIGN()  /* a -- a' */
{
	struct{void*x;char y;void*z;}align;
	*--sp = sizeof(align) - 2 * sizeof(void*) - sizeof(char); /* 1 or 3 */
	--sp,sp[0]=sp[1];	  /* DUP */
	sp[1]=sp[2];
	sp[2]=sp[0];	 /* TUCK */
	sp[1]+=sp[0],sp++;   /* + */
	*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
	sp[0]^=-1;     /* NOT */
	sp[1]&=sp[0],sp++;    /* AND */
}

void ALIGNED()  /* -- */
{
	HERE();
	ALIGN();
	HERE();
	sp[1]-=sp[0],sp++;     /* - */
	ALLOT();
}

void C_COMMA()  /* n -- */
{
	HERE();
	*(Byte *)(sp[0])=(Byte)(sp[1]);
	sp+=2;       /* 2DROP */     /* C! */
	*--sp=1;
	ALLOT();
}

void W_COMMA()  /* n -- */
{
	ALIGNED();
	HERE();
	*(unsigned short *)(sp[0])=(unsigned short)(sp[1]);
	sp+=2;     /* 2DROP */     /* W! */
	*--sp=2;
	ALLOT();
}

void COMMA()  /* n -- */
{
	ALIGNED();
	HERE();
	*(Cell *)(sp[0])=sp[1];
	sp+=2;   /* 2DROP */     /* ! */
	*--sp=sizeof(Cell);     /* CELL */
	ALLOT();
}

/* ==== Headers ==== | link | name | tick | */
Cell latest_=0;

void latest()  /* -- a */  /* point to latest dictionary entry */
{
	*--sp=(Cell)&latest_;
}

void L_TO_NAME()  /* link -- name */
{
	sp[0]+=sizeof(Cell);	  /* CELL + */
}

void N_TO_LINK()  /* name -- link */
{
	sp[0]-=sizeof(Cell);       /* CELL - */
}

void T_TO_NAME()  /* tick -- name */
{
	for(;;)	  /* BEGIN */
	{
		*--sp = 1, ALIGN(), MINUS();  /* jump back to skip random fill byte */
		--sp,sp[0]=sp[1];       /* DUP */
		sp[0]=(Cell)(*(Byte *)(sp[0]));	  /* C@ */
		sp[0]&=128;      /* AND */  /* non-ASCII */
		if(*sp++)break;      /* UNTIL */
	}
}

void N_TO_TICK()  /* name -- tick */
{
	C_FETCH_PLUS();
	*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
	sp[0]&=31;     /* AND */
	sp[1]+=sp[0],sp++;    /* + */
	ALIGN();
}

void T_TO_LINK()  /* tick -- link */
{
	T_TO_NAME();
	N_TO_LINK();
}

void L_TO_TICK()  /* link -- tick */
{
	L_TO_NAME();
	N_TO_TICK();
}

void L_TO_LINK()  /* link -- previous */
{
	if(sp[0])
		sp[0]=*(Cell *)(sp[0]);      /* @ */
}

void TO_BODY()  /* tick -- body */
{
	CELL(), PLUS();
}

/* ==== Dictionary Searching ==== */

void NAME_QUERY()  /* string \ name -- flag */
{
	*--sp=63;
	BITS_DIFFER_QUERY();
	if(*sp++)   /* IF */
	{
		sp+=2;       /* 2DROP */
		*--sp=0;
		return;	 /* EXIT */
	}
	--sp,sp[0]=sp[1];	  /* DUP */
	sp[0]-=1;       /* - */
	sp[0]=(Cell)(*(Byte *)(sp[0]));    /* C@ */
	sp[0]&=31;       /* AND */
	SAME_QUERY();
}

Cell search_dict_=0;

void search_dict()  /* -- a */  /* cache for fast lookups */
{
	*--sp = (Cell)&search_dict_;
}

void SEARCH_DICTIONARY()  /* string \ list -- link \ yes | -- string \ no */
{
	if(sp[0] == (Cell)&latest_)
	{  /* only search hash cache if searching whole dictionary */
		OVER(), search_dict(), FETCH(), FIND();
		if(sp[0])
		{
			N_TO_LINK(), sp[2] = sp[0],  sp++,  sp[0] = -1;
			return;
		}
		else
			sp++;
		for(;;)       /* BEGIN */
		{  /* search dictionary link list */
			L_TO_LINK();
			if(!sp[0]) break;      /* DUP WHILE */
			--sp,sp[0]=sp[2];       /* OVER */
			--sp,sp[0]=sp[2];    /* OVER */
			L_TO_NAME();
			NAME_QUERY();
			if(*sp++)	 /* IF */
			{
				sp[1]=sp[0];    /* NIP DUP */
				L_TO_NAME(), search_dict(), FETCH(), INSERT();
				*--sp=-1;
				return;	  /* EXIT */
			}
		};	 /* REPEAT */
	}
	for(;;)       /* BEGIN */
	{  /* search dictionary link list */
		L_TO_LINK();
		if(!sp[0]) break;      /* DUP WHILE */
		--sp,sp[0]=sp[2];       /* OVER */
		--sp,sp[0]=sp[2];    /* OVER */
		L_TO_NAME();
		NAME_QUERY();
		if(*sp++)	 /* IF */
		{
			NIP();
			*--sp=-1;
			return;	  /* EXIT */
		}
	};	 /* REPEAT */
}

void INIT_CACHE()  /* -- */  /* initialize the dictionary cache */
{
	if(search_dict_)
	{
		search_dict(), FETCH(), EMPTY();
	}
	else
	{
		DICTIONARY(), search_dict(), STORE();
	}
}

void LATE()  /* tick -- latest tick */  /* ' word LATE EXECUTE; for late binding */
{ 
	T_TO_NAME();
	latest();
	SEARCH_DICTIONARY();
	if(*sp++)    /* IF */
	{
		L_TO_TICK();
	}
	else	  /* ELSE */
	{
		N_TO_TICK();
	}
}

/* Environment */

/* ==== Interrupt control ==== */

void MINUS_INT()  /* -- */
{
}

void MINUS_INT_QUERY()  /* -- f */
{
	*--sp=0;
}

void PLUS_INT()  /* -- */
{
}

void QUESTION_PLUS_INT()  /* f -- */
{
	sp++;    /* DROP */
}

/* ==== Baron ==== */
struct
{
	Cell *i;
	Cell *r;
	Cell *e;
	Cell q[129];
}peasantq_={&peasantq_.q[128],&peasantq_.q[128],&peasantq_.q[128]};

void peasantq()  /* -- q */  
{
	*--sp=(Cell)&peasantq_;
}

/* maximum of hex 80 peasants */

void ZERO_BARON()  /* -- */
{
	peasantq();
	ZERO_Q();
}

void TO_BARON()  /* tick -- */
{
	peasantq();
	TO_Q_STAR();
}

void BARON_FROM()  /* -- tick */
{
	peasantq();
	Q_FROM();
}

void BARON()  /* -- */
{
	peasantq();
	Q_QUERY();
	if(*sp++)      /* IF */
	{
		peasantq();
		Q_FROM();
		if((tick=(void (**)())(*sp++)) != 0)
			(**tick)();	  /* EXECUTE */
		    else
		    	RESET();
	}
}

void KILL()  /* tick -- */
{
	*--sp=-1;
	TO_BARON();
	for(;;)	 /* BEGIN */
	{
		BARON_FROM();
		if(sp[0] == -1)break;    /* WHILE */
		--sp,sp[0]=sp[2];   /* OVER */
		--sp,sp[0]=sp[2];    /* OVER */
		sp[1]^=sp[0],sp++;   /* XOR */
		if(*sp++)     /* IF */
		{
			TO_BARON();
		}
		else   /* ELSE */
		{
			sp++;      /* DROP */
		}
	};	 /* REPEAT */
	sp+=2;      /* 2DROP */
}

void RUN()  /* tick -- */
{
	--sp,sp[0]=sp[1];     /* DUP */
	KILL();
	TO_BARON();
}

/* ==== Output stream ==== */
struct
{
	Cell *i;
	Cell *r;
	Cell *e;
	Cell q[257];
}emitq_={&emitq_.q[256],&emitq_.q[256],&emitq_.q[256]};

void emitq()  /* -- q */
{
	*--sp=(Cell)&emitq_;
}

/* decimal 256, enough for one string */
Cell out_=0;

void out()  /* -- a */  /* absolute cursor position since last cursor return */
{
	*--sp=(Cell)&out_;
}

/* Note:  If the peasant which is clearing the emitq, is also filling it,
		  even indirectly, then FLUSH-EMITS will never exit.  */

void FLUSH_EMITS()  /* -- */  /* wait until emitq is empty */
{
	for(;;)	/* BEGIN */
	{
		emitq();
		Q_QUERY();
		if(!*sp++)break;       /* WHILE */
		BARON();
	}   /* REPEAT */
}

void QUESTION_WAIT()  /* -- */
{
	for(;;)	/* BEGIN */
	{
		emitq();
		Q_QUERY();
		*--sp=256;
		if(!(sp[1]==sp[0])) {sp+=2;break;}
		sp+=2;       /* truth WHILE */
		BARON();
	};   /* REPEAT */
}

void EMIT()  /* char -- */
{
	QUESTION_WAIT();
	out();
	sp[0]=*(Cell *)(sp[0]);	/* @ */
	--sp,sp[0]=sp[2];       /* OVER */
	*--sp=8;
	if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;
	if(*sp++)       /* IF */
	{
		sp[0]-=1;      /* - */
	}
	else	  /* ELSE */
	{
		--sp,sp[0]=sp[2];	  /* OVER */
		*--sp=13;
		if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;
		if(*sp++)      /* IF */
		{
			sp++;	/* DROP */
			*--sp=0;
		}
		else   /* ELSE */
		{
			--sp,sp[0]=sp[2];	  /* OVER */
			sp[0]^=10;     /* XOR */
			if(*sp++)     /* IF */
			{
				--sp,sp[0]=sp[2];    /* OVER */
				sp[0]^=255;    /* XOR */
				if(*sp++)     /* IF */
				{
					sp[0]+=1;      /* + */
				}
			}
		}
	}
	out();
	*(Cell *)(sp[0])=sp[1];
	sp+=2;   /* 2DROP */     /* ! */
	emitq();
	TO_Q();
}

/* ==== Output formatter ==== */

void _CR()  /* -- */
{
	*--sp=13;
	EMIT();
	*--sp=10;
	EMIT();
}

void QUESTION_CR()  /* -- */
{
	out();
	sp[0]=*(Cell *)(sp[0]);     /* @ */
	if(*sp++)       /* IF */
	{
		_CR();
	}
}

void SPACE()  /* -- */
{
	BL();
	EMIT();
}

void SPACES()  /* n -- */
{
	*--sp=0;
	if((Integer)sp[0]>(Integer)sp[1]) sp[1]=sp[0]; sp++;   /* MAX */
	for(*--rp=*sp++;*rp;(*rp)--)	/* FOR */
	{
		SPACE();
	}	  /* NEXT */
	rp++;
}

void COUNT()  /* addr -- addr' \ count */
{
	C_FETCH_PLUS();
	*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
}

void TYPE()  /* addr \ count -- */
{
	for(*--rp=*sp++;*rp;(*rp)--)	/* FOR */
	{
		C_FETCH_PLUS();
		*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
		EMIT();
	}    /* NEXT */
	rp++;
	sp++;   /* DROP */
}

/* ==== Number Output ==== */
Cell base_=10;

void base()  /* -- a */
{
	*--sp=(Cell)&base_;
}


void HEX()  /* -- */
{
	*--sp=16;
	base();
	*(Cell *)sp[0]=sp[1];
	sp+=2;    /* 2DROP */     /* ! */
}

void DECIMAL()  /* -- */
{
	*--sp=10;
	base();
	*(Cell *)sp[0]=sp[1];
	sp+=2;       /* 2DROP */     /* ! */
}

void PAD()  /* -- addr */
{
	HERE();
	sp[0]+=LINELENGTH;	 /* + */
	ALIGN();
}

void HOLD()  /* char -- */
{
	*--sp=-1;
	PAD();
	*(Integer *)(sp[0])+=(Integer)sp[1];
	sp+=2;     /* 2DROP */     /* +! */
	PAD();
	sp[0]=*(Cell *)(sp[0]);     /* @ */
	*(Byte *)(sp[0])=(Byte)(sp[1]);
	sp+=2;	/* 2DROP */     /* C! */
}

void START_NUMBER_CONVERSION()  /* -- */
{
	PAD();
	--sp,sp[0]=sp[1];	 /* DUP */
	*(Cell *)sp[0]=sp[1];
	sp+=2;	/* 2DROP */     /* ! */
}

void CONVERT_DIGIT()  /* n -- n */
{
	base();
	sp[0]=*(Cell *)(sp[0]);     /* @ */
	*--rp=sp[0];
	sp[0]=sp[1]/(sp[0]);
	sp[1]%=*rp++;   /* /MOD */
	*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
	*--sp=9;
	--sp,sp[0]=sp[2];	/* OVER */
	if((Integer)sp[1]<(Integer)sp[0]) *++sp=-1; else *++sp=0;
	if(*sp++)	  /* IF */
	{
		sp[0]+=7;      /* + */
	}
	sp[0]+=48;      /* + */
	HOLD();
}

void CONVERT_NUMBER()  /* n -- n */
{
	for(;;)	/* BEGIN */
	{
		CONVERT_DIGIT();
		--sp,sp[0]=sp[1];	/* DUP */
		if(!*sp++)break;	/* 0= UNTIL */
	}
}

void SIGN()  /* m \ n -- n */
{
	*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
	sp[0]=(Integer)sp[0]<0?-1:0;	 /* 0< */
	if(*sp++)      /* IF */
	{
		*--sp=45;
		HOLD();
	}
}

void END_NUMBER_CONVERSION()  /* n -- addr \ count */
{
	sp++;      /* DROP */
	PAD();
	sp[0]=*(Cell *)(sp[0]);   /* @ */
	PAD();
	--sp,sp[0]=sp[2];	/* OVER */
	sp[1]-=sp[0],sp++;   /* - */
}

void DOT_R()  /* n \ m -- */
{
	*--rp=*sp++;	  /* >R */
	START_NUMBER_CONVERSION();
	if(base_==10)	 /* IF */
	{
		--sp,sp[0]=sp[1];    /* DUP */
		if((Integer)sp[0]<0) sp[0]=-(sp[0]);    /* ABS */
		CONVERT_NUMBER();
		SIGN();
	}
	else      /* ELSE */
	{
		CONVERT_NUMBER();
	}
	END_NUMBER_CONVERSION();
	*--sp=*rp++;	  /* R> */
	--sp,sp[0]=sp[2];      /* OVER */
	sp[1]-=sp[0],sp++;   /* - */
	SPACES();
	TYPE();
}

void DOT()  /* n -- */
{
	*--sp=0;
	DOT_R();
	SPACE();
}

/* ==== Input stream ==== */
struct
{
	Cell *i;
	Cell *r;
	Cell *e;
	Cell q[257];
}keyq_={&keyq_.q[256],&keyq_.q[256],&keyq_.q[256]};

void keyq()  /* -- q */
{
	*--sp=(Cell)&keyq_;
}

/* decimal 256; enough for one string */

void KEY_QUERY()  /* -- flag */
{
	keyq();
	Q_QUERY();
}

void KEY()  /* -- char */
{
	for(;;)    /* BEGIN */
	{
		KEY_QUERY();
		if(*sp++)break;       /* 0= WHILE */
		BARON();
	};	 /* REPEAT */
	keyq();
	Q_FROM();
}

/* ==== Text input buffer ==== */
Cell in_=0;

void in()  /* -- a */  /* buffer index */
{
	*--sp=(Cell)&in_;
}

struct
{
	Cell name8;
	char name9[LINELENGTH + 1];
	/* pointer; terminal input buffer */
}tib_;

void tib()  /* -- a */  
{
	*--sp=(Cell)&tib_;
}


void ZERO_TIB()  /* -- */
{
	in_ = 0;
	*--sp = 0;
	INPUT();
	C_STORE();
}

void INPUT()  /* -- addr */
{
	*--sp = tib_.name8 + in_;
}

void PLUS_IN()  /* addr -- */
{
	in_ = *sp++ - tib_.name8;
}

/* ==== Input scanner ==== */

void SKIP()  /* char -- */
{
	*--rp=*sp++;    /* >R */
	INPUT();
	for(;;)       /* BEGIN */
	{
		C_FETCH_PLUS();
		*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
		QUESTION_DUP();
		if(*sp++)    /* IF */
		{
			*--sp=*rp;      /* R */
			sp[1]-=sp[0],sp++;      /* - */
		}
		else	  /* ELSE */
		{
			*--sp=-1;
		}
		if(*sp++)break;     /* UNTIL */
	}
	sp[0]-=1;   /* - */
	PLUS_IN();
	rp++;	 /* R> DROP */
}

void SCAN_FOR()  /* char -- flag */
{  /* scan for a certain character */
	*--rp=*sp++;	/* >R */
	INPUT();
	for(;;)	/* BEGIN */
	{
		--sp,sp[0]=*(Byte*)sp[1];	/* DUP C@ */
		QUESTION_DUP();
		if(!*sp++)break;	/* WHILE */
		*--sp=*rp;	/* R */
		if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;	/* = */
		if(*sp++)	/* IF */
		{
			sp[0]+=1;	/* + */
			PLUS_IN();
			rp++;	/* R> DROP */
			*--sp=-1;	/* YES */
			return;	/* EXIT */
		}
		sp[0]+=1;	/* + */
	}	/* REPEAT */
	PLUS_IN();
	rp++;	/* R> DROP */
	*--sp=0;	/* NO */
}

void SCAN()  /* char -- */
{
	*--rp=*sp++;      /* >R */
	INPUT();
	for(;;)       /* BEGIN */
	{
		C_FETCH_PLUS();
		*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
		QUESTION_DUP();
		if(*sp++)    /* IF */
		{
			*--sp=*rp;      /* R */
			if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;
		}
		else	 /* ELSE */
		{
			sp[0]-=1;    /* - */
			*--sp=-1;
		}
		if(*sp++)break;	  /* UNTIL */
	}
	PLUS_IN();
	rp++;   /* R> DROP */
}

void PARSE()  /* char -- */
{
	*--rp=*sp++;     /* >R */
	HERE();
	--sp,sp[0]=sp[1];      /* DUP */
	sp[0]+=1;       /* + */
	--sp,sp[0]=sp[1];       /* DUP */
	INPUT();
	for(;;)      /* BEGIN */
	{
		C_FETCH_PLUS();
		--sp,sp[0]=sp[2];	 /* OVER */
		QUESTION_DUP();
		if(*sp++)    /* IF */
		{
			*--sp=*rp;      /* R */
			sp[1]^=sp[0],sp++;      /* XOR */
		}
		else	/* ELSE */
		{
			sp[0]-=1;    /* - */
			*--sp=0;
		}
		if(!*sp++)break;	  /* WHILE */
		*--rp=*sp++;	/* >R */
		*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
		C_STORE_PLUS();
		*--sp=*rp++;	 /* R> */
	};     /* REPEAT */
	PLUS_IN();
	sp++;   /* DROP */
	*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
	sp[1]-=sp[0],sp++;   /* - */
	*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
	*(Byte *)(sp[0])=(Byte)(sp[1]);
	sp+=2;     /* 2DROP */     /* C! */
	rp++;   /* R> DROP */
}

/* ==== Error recovery ==== */

void ABORT()  /* -- */
{
	SHELL_ABORT();
}

void ERROR()  /* -- */
{
	HERE();
	COUNT();
	TYPE();
	--sp,*(char**)sp="\006<- eh?";
	COUNT();
	TYPE();
	ABORT();
}

void QUESTION_ERROR()  /* flag -- */
{
	if(*sp++)	 /* IF */
	{
		ERROR();
	}
}

/* ==== Number Conversion ==== */
void DIGIT()  /* char -- n \ flag */
{
	sp[0]-=48;	/* - */
	--sp,sp[0]=sp[1];	/* DUP */
	*--sp=9;
	if((Integer)sp[1]>(Integer)sp[0]) *++sp=-1; else *++sp=0;	/* > */
	if(*sp++)	/* IF */
	{
		sp[0]-=7;	/* - */
		--sp,sp[0]=sp[1];	/* DUP */
		*--sp=35;
		if((Integer)sp[1]>(Integer)sp[0]) *++sp=-1; else *++sp=0;	/* > */
		if(*sp++)	/* IF */
		{
			sp[0]-=32;	/* - */
		}
		--sp,sp[0]=sp[1];	/* DUP */
		*--sp=10;
		if((Integer)sp[1]<(Integer)sp[0]) *++sp=-1; else *++sp=0;	/* < */
		sp[1]|=sp[0],sp++;	/* OR */
	}
	--sp,sp[0]=sp[1];	/* DUP */
	base();
	sp[0]=*(Cell *)sp[0];	/* @ */
	if(sp[1]<sp[0]) *++sp=-1; else *++sp=0;	/* U< */
}

void DIGITS_QUERY()  /* a \ c -- f */
{
	--sp,sp[0]=sp[1];	/* DUP */
	*--sp=0;
	if((Integer)sp[1]>(Integer)sp[0]) *++sp=-1; else *++sp=0;	/* > */
	if(*sp++)	/* IF */
	{
		*--sp=-1;	/* YES */
		*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
		for(*--rp=*sp++;*rp;(*rp)--)	/* FOR */
		{
			*--rp=*sp++;	/* >R */
			C_FETCH_PLUS();
			*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
			DIGIT();
			sp[1]=sp[0],sp++;	/* NIP */
			*--sp=*rp++;	/* R> */
			sp[1]&=sp[0],sp++;	/* AND */
		}	/* NEXT */
		rp++;
		sp[1]=sp[0],sp++;	/* NIP */
	}
	else	/* ELSE */
	{
		sp+=2;	/* 2DROP */
		*--sp=0;	/* NO */
	}
}

void NUMBER_QUERY()  /* string -- f */
{
	base();
	sp[0]=*(Cell *)sp[0];	/* @ */
	*--rp=*sp++;	/* >R */
	COUNT();
	--sp,sp[0]=*(Byte*)sp[2];	/* OVER C@ */
	*--sp=45;
	if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;	/* = */
	if(*sp++)	/* IF */
	{
		sp[0]-=1;	/* - */
		*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
		sp[0]+=1;	/* + */
		*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
	}
	if(sp[0] > 2 )
	{
		--sp,sp[0]=*(Byte*)sp[2];	/* OVER C@ */
		*--sp=48;
		if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;	/* = */
		if(*sp++)	/* IF */
		{
			--sp,sp[0]=sp[2];	/* OVER */
			sp[0]+=1;	/* + */
			sp[0]=(Cell)(*(Byte *)sp[0]);	/* C@ */
			--sp,sp[0]=sp[1];	/* DUP */
			*--sp=88;
			if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;	/* = */
			*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
			*--sp=120;
			if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;	/* = */
			sp[1]|=sp[0],sp++;	/* OR */
			if(*sp++)	/* IF */
			{
				sp[0]-=2;	/* - */
				*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
				sp[0]+=2;	/* + */
				*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
				HEX();
			}
		}
	}
	DIGITS_QUERY();
	*--sp=*rp++;	/* R> */
	base();
	*(Cell *)sp[0]=sp[1],sp+=2;	/* ! */
}

void DIGITS()  /* a \ c -- n */
{
	--sp,sp[0]=sp[1];	/* DUP */
	*--sp=1;
	if((Integer)sp[1]<(Integer)sp[0]) *++sp=-1; else *++sp=0;	/* < */
	QUESTION_ERROR();
	--sp,sp[0]=sp[1],sp[1]=0;	/* SWAP */
	for(*--rp=*sp++;*rp;(*rp)--)	/* FOR */
	{
		*--rp=*sp++;	/* >R */
		C_FETCH_PLUS();
		*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
		DIGIT();
		sp[0]=sp[0]==0?-1:0;	/* 0= */
		QUESTION_ERROR();
		*--sp=*rp++;	/* R> */
		base();
		sp[0]=*(Cell *)sp[0];	/* @ */
		sp[1]*=sp[0],sp++;	/* * */
		sp[1]+=sp[0],sp++;	/* + */
	}	/* NEXT */
	rp++;
	sp[1]=sp[0],sp++;	/* NIP */
}

void NUMBER()  /* string -- n */
{
	base();
	sp[0]=*(Cell *)sp[0];	/* @ */
	*--rp=*sp++;	/* >R */
	COUNT();
	--sp,sp[0]=*(Byte*)sp[2];	/* OVER C@ */
	*--sp=45;
	if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;	/* = */
	*--rp=sp[0];	/* DUP >R */  /* check for negative numbers */
	if(*sp++)	/* IF */
	{
		sp[0]-=1;	/* - */
		*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
		sp[0]+=1;	/* + */
		*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
	}
	if (sp[0] > 2 )
	{
		--sp,sp[0]=*(Byte*)sp[2];	/* OVER C@ */
		*--sp=48;
		if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;	/* = */  /* check for hex numbers */
		if(*sp++)	/* IF */
		{
			--sp,sp[0]=sp[2];	/* OVER */
			sp[0]+=1;	/* + */
			sp[0]=(Cell)(*(Byte *)sp[0]);	/* C@ */
			--sp,sp[0]=sp[1];	/* DUP */
			*--sp=88;
			if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;	/* = */
			*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
			*--sp=120;
			if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;	/* = */
			sp[1]|=sp[0],sp++;	/* OR */
			if(*sp++)	/* IF */
			{
				sp[0]-=2;	/* - */
				*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
				sp[0]+=2;	/* + */
				*--rp=sp[1],sp[1]=sp[0],sp[0]=*rp++;	/* SWAP */
				HEX();
			}
		}
	}
	DIGITS();
	*--sp=*rp++;	/* R> */
	if(*sp++)	/* IF */
	{
		sp[0]=-(sp[0]);	/* NEGATE */
	}
	*--sp=*rp++;	/* R> */
	base();
	*(Cell *)sp[0]=sp[1],sp+=2;	/* ! */
}

/* ==== Compiler ==== */
Cell compile_=0;

void compile()  /* -- a */  /* state of interpreting/ compiling */
{
	*--sp=(Cell)&compile_;
}


void RIGHT_SQUARE_BRACKET()  /* -- */
{
	*--sp=128;
	compile();
	*(Cell *)(sp[0])=sp[1];
	sp+=2;      /* 2DROP */     /* ! */
}

void LEFT_SQUARE_BRACKET()  /* -- */
{
	*--sp=0;
	compile();
	*(Cell *)(sp[0])=sp[1];
	sp+=2;	 /* 2DROP */     /* ! */
}

void COMPILE()  /* tick -- */
{
	COMMA();
}

/* ==== Inner interpreters for cells ==== */

void TO_I()  /* n -- */
{
	*--rp=*sp++;
}

void I_FROM()  /* -- n */
{
	*--sp=*rp++;
}

void LIT()  /* -- n */
{
	*--sp=(Cell)*ip++;
}

void BRANCH()  /* -- */
{	
	ip = (void (***)())(*ip);
}

void ZERO_BRANCH()  /* f -- */
{
	if(!*sp++)	  /* IF */
		ip = (void (***)())(*ip);
	else	 /* ELSE */
		ip++;
}

void MINUS_BRANCH()  /* -- */
{
	if(*rp)       /* IF */
	{
		*rp-=1;      /* 1 - */
		ip = (void (***)())(*ip);  /* branch back */
	}
	else	 /* ELSE */
	{
		rp++;  /* drop count */
		ip++;  /* skip over */
	}
}

/* ==== Outer interpreter for text ==== */

void COMMENT()  /* -- */  /* Scan for multi line comments*/
{
	for(;;)	/* BEGIN */
	{
		*--sp=41;
		SCAN_FOR();
		if(*sp++)	/* IF */
		{
			return;	/* EXIT */
		}
		INPUT_LINE();
		sp[0]=sp[0]==0?-1:0;	/* 0= */
		if(*sp++)break;	/* UNTIL */
	}
}

void WORD()  /* char -- */
{
	--sp,sp[0]=sp[1];     /* DUP */
	SKIP();
	PARSE();
}

void CHAR()  /* -- c */
{
	BL();
	SKIP();
	INPUT();
	sp[0]=(Cell)(*(Byte *)(sp[0]));     /* C@ */
}

void FIND_QUERY()  /* -- tick \ status | -- string \ no */
{
	BL();
	WORD();
	HERE();
	latest();
	SEARCH_DICTIONARY();
	if(sp[0])
	{	 /* DUP IF */
		sp++;  /* DROP */
		--sp,sp[0]=sp[1];    /* DUP */
		L_TO_TICK();
		*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
		L_TO_NAME();
		sp[0]=(Cell)(*(Byte *)(sp[0]));    /* C@ */
		sp[0]&=224;      /* AND */
	}
}

void QUESTION_FIND()  /* -- tick */
{
	FIND_QUERY();
	sp[0]=sp[0]==0?-1:0;     /* 0= */
	QUESTION_ERROR();
}

void QUESTION_EXECUTE()  /* tick \ status -- */
{
	compile();
	sp[0]=*(Cell *)(sp[0]);	  /* @ */
	if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;
	if(*sp++)      /* IF */
	{
		COMPILE();
	}
	else    /* ELSE */
	{
		tick=(void (**)())(*sp++);
		TestTick;(**tick)();	  /* EXECUTE */
	}
}

void LITERAL()  /* n -- [n] */
{
	compile();
	sp[0]=*(Cell *)(sp[0]);      /* @ */
	if(*sp++)       /* IF */
	{
		*--sp=(Cell)&_LIT.tick;      /* ' */
		COMPILE();
		COMMA();
	}
}

void INTERPRET()  /* -- */
{
	for(;;)   /* BEGIN */
	{
		CHAR();
		if(!*sp++)break;	  /* WHILE */
		FIND_QUERY();
		QUESTION_DUP();
		if(*sp++)     /* IF */
		{
			QUESTION_EXECUTE();
		}
		else   /* ELSE */
		{
			NUMBER();
			tick = (thread)&_LITERAL.tick;      /* ' */
			TestTick;(**tick)();	  /* EXECUTE */
		}
	};      /* REPEAT */
}

/* ==== Key collector ==== */
struct
{
	char nameB[32];
}prompt_={"\005bot: "};

void prompt()  /* -- a */  /* contains count prefixed string for prompt */
{
	*--sp=(Cell)&prompt_;
}


void DOT_PROMPT()  /* -- */
{
	QUESTION_CR();
	compile();
	sp[0]=*(Cell *)(sp[0]);   /* @ */
	if(*sp++)       /* IF */
	{
		--sp,*(char**)sp="\002] ";
		COUNT();
		TYPE();
	}
	else   /* ELSE */
	{
		prompt();
		COUNT();
		TYPE();
	}
}

void PROMPT()  /* -- */
{
	*--sp=0;
	WORD();
	HERE();
	prompt();
	*--sp=32;
	CMOVE();
}

void COLLECT()  /* -- */
{
	KEY();
	--sp,sp[0]=sp[1];       /* DUP */
	*--sp=13;
	if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;
	if(*sp++)       /* IF */
	{
		sp++;	/* DROP */
		*--sp=0;
		INPUT();
		*(Byte *)(sp[0])=(Byte)(sp[1]);
		sp+=2;	/* 2DROP */     /* C! */
		SPACE();
		*--sp=0;
		in();
		*(Cell *)(sp[0])=sp[1];
		sp+=2;       /* 2DROP */     /* ! */
		INTERPRET();
		ZERO_TIB();
		DOT_PROMPT();
	}
	else     /* ELSE */
	{
		--sp,sp[0]=sp[1];	  /* DUP */
		*--sp=8;
		if(sp[1]==sp[0]) *++sp=-1; else *++sp=0;
		if(*sp++)	/* IF */
		{
			in();
			sp[0]=*(Cell *)(sp[0]);    /* @ */
			if(*sp++)       /* IF */
			{
				INPUT();
				sp[0]-=1;     /* - */
				PLUS_IN();
			}
			else       /* ELSE */
			{
				sp++;      /* DROP */
				*--sp=7;
			}
		}
		else	 /* ELSE */
		{
			--sp,sp[0]=sp[1];	  /* DUP */
			*--sp=27;
			if((Integer)sp[1]<(Integer)sp[0]) *++sp=-1; else *++sp=0;
			if(*sp++)      /* IF */
			{
				sp++;	/* DROP */
				*--sp=7;
			}
			else   /* ELSE */
			{
				out();
				sp[0]=*(Cell *)(sp[0]);	 /* @ */
				*--sp=78;
				if((Integer)sp[1]<(Integer)sp[0]) *++sp=-1; else *++sp=0;
				if(*sp++)	/* IF */
				{
					--sp,sp[0]=sp[1];    /* DUP */
					INPUT();
					C_STORE_PLUS();
					PLUS_IN();
				}
				else    /* ELSE */
				{
					sp++;      /* DROP */
					*--sp=7;
				}
			}
		}
		EMIT();
	}
}

void COLLECTOR()  /* -- */
{
	KEY_QUERY();
	if(*sp++)   /* IF */
	{
		COLLECT();
	}
	*--sp=(Cell)&_COLLECTOR.tick;   /* ' */
	TO_BARON();
}

void INIT()  /* -- */
{
	keyq();
	ZERO_Q();
	tib();
	--sp,sp[0]=sp[1];       /* DUP */
	sp[0]+=sizeof(Cell);    /* CELL + */
	*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
	*(Cell *)(sp[0])=sp[1];
	sp+=2;       /* 2DROP */     /* ! */
	ZERO_TIB();
	*--sp=(Cell)&_COLLECTOR.tick;    /* ' */
	RUN();
	INIT_IO();
	SH_STORE();		/* initialize error trapping shells */
}

/* ==== Control loop ==== */

void QUIT()  /* -- */
{
	for(;;)      /* BEGIN */
	{
		sp = sp0_;	/* SP! */
		rp = rp0_;	/* RP! */
		INIT();
		LEFT_SQUARE_BRACKET();
		_CR();
		DOT_PROMPT();
 	 	shell_nest();
		if( setjmp(*ep) == 0 )
		{
			for(;;)      /* BEGIN */
			{
				BARON();
			};       /* REPEAT */
		}
		shell_unest();
	};       /* REPEAT */
}

/* ==== Compilers ==== */

/* ==== Control flow ==== */

void AHEAD()  /* -- a */
{
	*--sp=(Cell)&_BRANCH.tick;    /* ' */
	COMPILE();
	HERE();
	*--sp=0;
	COMMA();
}

void IF()  /* -- a */
{
	*--sp=(Cell)&_ZERO_BRANCH.tick;      /* ' */
	COMPILE();
	HERE();
	*--sp=0;
	COMMA();
}

void ENDIF()  /* a -- */
{
	HERE();
	*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
	*(Cell *)(sp[0])=sp[1];
	sp+=2;       /* 2DROP */     /* ! */
}

void ELSE()  /* a -- a */
{
	AHEAD();
	*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
	ENDIF();
}

void THEN()  /* a -- */
{
	ENDIF();
}

void BEGIN()  /* -- m */
{
	HERE();
}

void AGAIN()  /* m -- */
{
	*--sp=(Cell)&_BRANCH.tick;    /* ' */
	COMPILE();
	COMMA();
}

void WHILE()  /* m -- m \ n */
{
	IF();
}

void REPEAT()  /* m \ n -- */
{
	*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
	AGAIN();
	ENDIF();
}

void UNTIL()  /* m -- */
{
	*--sp=(Cell)&_ZERO_BRANCH.tick;      /* ' */
	COMPILE();
	COMMA();
}

void FOR()  /* -- m \ n */
{
	*--sp=(Cell)&_TO_I.tick;    /* ' */
	COMPILE();
	AHEAD();
	HERE();
}

void NEXT()  /* m \ n -- */
{
	*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
	THEN();
	*--sp=(Cell)&_MINUS_BRANCH.tick;     /* ' */
	COMPILE();
	COMMA();
}

void EXIT()  /* f -- */
{
	*--sp=0;
	COMMA();
}

void TICK()  /* -- tick */
{
	QUESTION_FIND();
	compile();
	sp[0]=*(Cell *)(sp[0]);   /* @ */
	if(*sp++)       /* IF */
	{
		LITERAL();
	}
}

/* ==== Strings ==== */

void STRING()  /* terminator -- string */
{
	PARSE();
	HERE();
	--sp,sp[0]=sp[1];      /* DUP */
	sp[0]=(Cell)(*(Byte *)(sp[0]));	  /* C@ */
	sp[0]+=1;	/* + */
	ALLOT();
	ALIGNED();
}

void QUOTE()  /* -- s */
{
	compile();
	sp[0]=*(Cell *)(sp[0]);	/* @ */
	if(*sp++)       /* IF */
	{
		AHEAD();
	}
	*--sp=34;
	STRING();
	compile();
	sp[0]=*(Cell *)(sp[0]);     /* @ */
	if(*sp++)       /* IF */
	{
		*--rp = sp[1], sp[1] = sp[0], sp[0] = *rp++;	/* SWAP */
		THEN();
		LITERAL();
	}
}

void DOT_QUOTE()  /* -- */
{
	compile();
	sp[0]=*(Cell *)(sp[0]);	  /* @ */
	if(*sp++)       /* IF */
	{
		tick = (thread)&_QUOTE.tick;    /* ' */
		TestTick;(**tick)();	  /* EXECUTE */
		*--sp=(Cell)&_COUNT.tick;	 /* ' */
		COMPILE();
		*--sp=(Cell)&_TYPE.tick;     /* ' */
		COMPILE();
	}
	else       /* ELSE */
	{
		*--sp=34;
		WORD();
		HERE();
		COUNT();
		TYPE();
	}
}

/* ==== Defining words ==== */
void MAKE_LINK()  /* current -- */
{
	latest();
	sp[0]=*(Cell *)(sp[0]);	 /* @ */
	COMMA();
	latest();
	*(Cell *)(sp[0])=sp[1];
	sp+=2;       /* 2DROP */     /* ! */
}

void MAKE_NAME()  /* -- */
{
	*--sp=128;
	HERE();
	--sp,sp[0]=sp[1];   /* DUP */
	sp[0]=(Cell)(*(Byte *)(sp[0]));	  /* C@ */
	sp[0]+=1;	/* + */
	ALLOT();
	PLUS_BITS();
}

void HEADER()  /* -- */
{
	ALIGNED();
	HERE();
	MAKE_LINK();
	BL();
	WORD();
	MAKE_NAME();
	ALIGNED();
	latest(), L_TO_LINK(), L_TO_NAME(),  search_dict(), FETCH(),  INSERT();
}

void SMUDGE()  /* -- */
{
	*--sp=32;
	latest();
	L_TO_LINK();
	L_TO_NAME();
	PLUS_BITS();
}

void RECURSIVE()  /* -- */
{
	*--sp=32;
	latest();
	L_TO_LINK();
	L_TO_NAME();
	MINUS_BITS();
}

void IMMEDIATE()  /* -- */
{
	*--sp=64;
	latest();
	L_TO_LINK();
	L_TO_NAME();
	PLUS_BITS();
}

void CREATE()  /* -- */
{
	HEADER();
	*--sp=(Cell)VII;
	COMMA();
}

void LESS_THAN_BUILDS()  /* -- */
{
	CREATE();
	*--sp=0;
	COMMA();
}

Cell ending = 0;  /* kludge to fake an end of a definition */

void DOES_FROM()  /* -- */
{
	*--sp=(Cell)ip;
	*--sp=(Cell)DII;
	latest();
	L_TO_LINK();
	L_TO_TICK();
	STORE_PLUS();
	*(Cell *)(sp[0])=sp[1];
	sp+=2;	  /* 2DROP */     /* ! */
	ip = (void (***)())&ending;
}

void COLON()  /* -- */
{
	HEADER();
	SMUDGE();
	*--sp=(Cell)COLON_II;
	COMMA();
	tick = (thread)&_RIGHT_SQUARE_BRACKET.tick;	 /* ' */
	TestTick;(**tick)();	  /* EXECUTE */
}

void SEMI_COLON()  /* -- */
{
	*--sp=0;
	COMMA();
	tick = (thread)&_RECURSIVE.tick;       /* ' */
	TestTick;(**tick)();	  /* EXECUTE */
	tick = (thread)&_LEFT_SQUARE_BRACKET.tick;   /* ' */
	TestTick;(**tick)();	  /* EXECUTE */
}

void CONSTANT()  /* n -- */
{
	HEADER();
	*--sp=(Cell)CII;
	COMMA();
	COMMA();
}

void VARIABLE()  /* n -- */
{
	CREATE();
	COMMA();
}

void QUEUE()  /* n -- */
{
	CREATE();
	*--sp=3;
	CELLS();
	HERE();
	sp[1]+=sp[0],sp++;	 /* + */
	--sp,sp[0]=sp[1];       /* DUP */
	COMMA();
	COMMA();
	sp[0]+=1;     /* + */
	CELLS();
	--sp,sp[0]=sp[1];      /* DUP */
	HERE();
	sp[1]+=sp[0],sp++;    /* + */
	COMMA();
	ALLOT();
}

void FORGET()  /* -- */
{
	QUESTION_FIND();
	T_TO_LINK();
	--sp,sp[0]=sp[1];	  /* DUP */
	dp();
	*(Cell *)(sp[0])=sp[1];
	sp+=2;	  /* 2DROP */     /* ! */
	sp[0]=*(Cell *)(sp[0]);     /* @ */
	latest();
	*(Cell *)(sp[0])=sp[1];
	sp+=2;	/* 2DROP */     /* ! */
	INIT_CACHE();
}

/* ==== End of Kernel ==== */

void VERSION()  /* -- */
{
	--sp,*(char**)sp="\015botForth V.15";
	COUNT();
	TYPE();
}
/* derived from botForth V.14 */

void RESET()  /* -- */
{
	rp = rp0_;	/* RP! */
	sp = sp0_;	/* SP! */
	INIT_CACHE();
	ZERO_BARON();
	*--sp=(Cell)&_INIT.tick;      /* ' */
	LATE();
	tick=(void (**)())(*sp++);
	TestTick;(**tick)();	  /* EXECUTE */
	*--sp=(Cell)&_VERSION.tick;      /* ' */
	LATE();
	tick=(void (**)())(*sp++);
	TestTick;(**tick)();	  /* EXECUTE */
	QUIT();
}

extern void *_listhead;

void main(void)  /* -- */
{
/*
 * Memory model:  | user dictionary | limit | parameter stack | return stack |
 */
	dp_ = (Cell)malloc(MSIZE);
	rp0_ = rp = (Cell *)(dp_ + MSIZE);
	sp0_ = sp = rp0_ - RCELLS;
	limit_ = (Cell)(sp0_ - DCELLS);
	latest_ = (Cell)_listhead;
	init_signals();
	RESET();
}
 