( Timbre  Rob Chapman Dec 17, 1991 )

C
#include "extkern.h"	/* ticks from kernel */
#include "extlib.h"		/* ticks from library */
#include "extii.h"		/* inner interpreters */
Forth

( ==== Version ==== )
: VERSION  ( -- )  ." Timbre V.4 " ;  ( derived from Timbre V.3 )

( ==== Define a rule set ==== )
: RULE-SET  ( -- )  DICTIONARY CONSTANT ; 

( ==== Word dictionary for rules ==== )
  0 VARIABLE rule-words  ( all words which appear in all rules )

: default  ( -- s )  " { }" ;
: IN-RULES?  ( s -- s' \ f )  DUP rule-words @ FIND  DUP  IF  NIP YES  ENDIF ;
: GET-WORD  ( -- name )  BL WORD  HERE IN-RULES?  IF  EXIT  ENDIF
   DUP C@ 1 + ALLOCATE  TUCK  OVER C@ 1 +  <CMOVE  DUP rule-words @ INSERT  ;

( ==== Rule set stack ==== )
  16 QUEUE ruleq		( stack of rule sets for input pattern matching )
  16 QUEUE dictq		( stack of check rule sets )
  NO VARIABLE activity ( flag: an activity is being defined for a rule )
  
: RULES>  ( -- rule-set )  ruleq PULL ;
: >RULES  ( rule-set -- )  ruleq STUFF ;
: RULE-SETS?  ( -- n )  ruleq Q? ;
: NEXT-RULE-SET  ( -- )  RULES>  dictq PUSH ;
: RESET-RULES  ( -- )  dictq Q?  FOR  dictq POP  >RULES  NEXT ;
: RULES  ( ruleset -- )  rule-words @ 0=  IF  DICTIONARY rule-words !  ENDIF
   NO activity !  ruleq 0Q  >RULES  dictq 0Q ;

( ==== Phrase maker ==== | >word... | 0 | )
  128 QUEUE inputq         ( hold parsed input )
  128 QUEUE outputq        ( hold matched phrase or parsed phrases )

: PHRASE  ( terminator -- phrase )  inputq
   BEGIN  OVER CHAR XOR  WHILE  GET-WORD  OVER PUSH  REPEAT  INPUT 1 + +IN
   NIP  Q?  DUP 1 + CELLS ALLOCATE
   SWAP NUP  FOR  inputq PULL  SWAP !+  NEXT  0 SWAP ! ;

: PHRASES  ( initiater \ terminater -- )
   BEGIN  DUP PHRASE  outputq STUFF  OVER INPUT C@ =
   WHILE  INPUT 1 + +IN  REPEAT  2DROP ;

: STUFF-IT  ( phrase -- )  0 SWAP  BEGIN  @+  OVER 0=  UNTIL  2DROP
   BEGIN  ?DUP  WHILE  inputq STUFF  REPEAT ;

( ==== Method maker ==== | compiled words... EXIT | )
: METHOD  ( -- method )   YES activity !  ' ] EXECUTE  HERE  :II ,
   BEGIN  CHAR ` ]  XOR  WHILE  BL WORD HERE INTERPRET-WORD  REPEAT  CHAR SKIP
   " EXIT" INTERPRET-WORD  ' [ EXECUTE  NO activity ! ;

: IN-PHRASE  ( terminator -- )  >R ' AHEAD EXECUTE  R> PHRASE >R
   ' ENDIF EXECUTE  R> ' LITERAL EXECUTE  ' STUFF-IT COMPILE ;

( ==== Rule maker ==== | >next rule | >phrase | >method | >substitute | )
: ADD-RULE  ( sub \ meth \ phr -- )
   DUP @  IF  DUP @  ELSE  default  ENDIF
   DUP ruleq Q FIND  0=  IF  DUP ruleq Q INSERT  ENDIF
   ruleq Q ADJUNCT  4 CELLS ALLOCATE  SWAP  NUP LINK  CELL +  !+ !+ ! ;

: ADD-RULES  ( sub \ meth -- )
   outputq Q?  FOR  2DUP outputq PULL ADD-RULE  NEXT  2DROP ;

: MAKE-RULES  ( initiator \ terminator -- )  2DUP PHRASES  INPUT C@ ` [ =
   IF  INPUT C@ SKIP  METHOD >R   OVER INPUT C@ =
       IF  SWAP SKIP  PHRASE  ELSE  2DROP  0  ENDIF  R>
   ELSE  2DROP  outputq PULL  0  ENDIF  ADD-RULES ;

: {  ( -- )  activity @
   IF  ` } IN-PHRASE
   ELSE  compile @  IF  ' { EXECUTE  ELSE  ` { ` } MAKE-RULES  ENDIF
   ENDIF ; IMMEDIATE

( ==== And for curly bracket rules ==== )
: |  ( -- )  activity @
   IF  ` | IN-PHRASE  ELSE  ` | ` | MAKE-RULES  ENDIF ; IMMEDIATE

( ==== Run time navigation ==== )
  0 VARIABLE check-rule  ( points to the rule which is being checked )
  0 VARIABLE check-word  ( points to a word in the phrase of the rule being checked )

: RULE?  ( -- f )  check-rule @ ;
: RULE-DONE?  ( -- f )  check-word @ @ 0= ;
: FIRST-WORD  ( -- )  check-rule @  CELL +  @  check-word ! ;
: NEXT-WORD  ( -- )  RULE-DONE? 0=  IF  CELL check-word +!  ENDIF
   inputq PULL  outputq PUSH ;
: NEXT-RULE  ( -- )  check-rule @ @  DUP check-rule !  IF  FIRST-WORD  NEXT-WORD  ENDIF ;
: SET-RULE  ( s -- )  ruleq Q ADJUNCT  @  check-rule !  FIRST-WORD ;
: FIND-RULE?  ( -- f )  inputq Q  ruleq Q  FIND  DUP  IF  DUP SET-RULE  ENDIF ;

( ==== Input parser ==== )
: GET-INPUT  ( -- )  inputq Q?  IF  EXIT  ENDIF
   BEGIN  CHAR  IF  GET-WORD  inputq PUSH  EXIT  ENDIF  INPUT-LINE  0=  UNTIL
   SHELL-END ;
: RESTORE-INPUT  ( -- )  outputq inputq  OVER Q?
   FOR  OVER POP  OVER STUFF  NEXT  2DROP ;
: WORD-MATCH?  ( -- f )  check-word @ @ inputq Q   2DUP =
   IF  2DROP  YES  ELSE  COMPARE  ENDIF ;
      
( ==== Rule execution ==== )
  0 VARIABLE rule-echo  ( echo a rule when executing for tracing )

: DO-METHOD  ( rule -- )  2 CELLS + @  ?DUP  IF  EXECUTE  ENDIF ;
: SUBSTITUTE  ( rule -- )  3 CELLS + @  ?DUP 0=  IF EXIT ENDIF  0 SWAP
   BEGIN  @+ OVER  0=  UNTIL  2DROP  BEGIN  ?DUP  WHILE  inputq STUFF  REPEAT ;
: RUN-WORD  ( s -- )  latest SEARCH-DICTIONARY  IF  L>TICK  EXECUTE  EXIT  ENDIF
   ?CR  COUNT TYPE ."  can't be found in the dictionary and isn't run." ;
: RUN-RULE  ( rule -- )  >R  rule-echo @
   IF  R CELL +  " .RULE" RUN-WORD  ENDIF
   R DO-METHOD  R> SUBSTITUTE ;

( ==== Rule cache ==== )
  0 VARIABLE rule  ( address of rule which is in the cache )
  0 VARIABLE rule-size  ( length of phrase in cached rule )

: CACHED?  ( -- f )  rule-size @ ;
: CACHE-RULE  ( -- )  outputq Q?  rule-size !  check-rule @  rule ! ;
: TURF-PHRASE  ( -- )  rule-size @  FOR  inputq PULL DROP  NEXT ;
: RUN-CACHE  ( -- )  TURF-PHRASE
   rule @  RUN-RULE  0 rule-size !  0 check-rule ! ;
: LONGER?  ( -- f )  outputq Q?  rule-size @  U> ;

( ==== Timbre engine ==== )
: ?DEFAULT  ( -- )
   BEGIN  RULE-SETS?  WHILE
     default ruleq Q FIND
     IF  default ruleq Q ADJUNCT @ RUN-RULE  RESET-RULES  EXIT  ENDIF
     NEXT-RULE-SET
   REPEAT  RESET-RULES  ." No rules match input. "  ABORT ;
: CHECK-RULE  ( -- )  GET-INPUT  WORD-MATCH?
   IF  NEXT-WORD  ELSE  RESTORE-INPUT  NEXT-RULE  ENDIF ;
: CHECK-RULES  ( -- )
   BEGIN  RULE-DONE?
     IF  LONGER?  IF  CACHE-RULE  ENDIF  RESTORE-INPUT  NEXT-RULE
     ELSE  CHECK-RULE  ENDIF  RULE? 0=
   UNTIL  RULE-SETS?  IF  NEXT-RULE-SET  ENDIF ;
: CHECK-RULE-SETS  ( -- )  RULE-SETS?
   IF  GET-INPUT  FIND-RULE?  IF  NEXT-WORD  ELSE  NEXT-RULE-SET  ENDIF
   ELSE  RESET-RULES  CACHED?  IF  RUN-CACHE  ELSE  ?DEFAULT  ENDIF  ENDIF ;
: TRANSLATE-FILE  ( -- )  SHELL{
   BEGIN  RULE?  IF  CHECK-RULES  ELSE  CHECK-RULE-SETS  ENDIF  AGAIN 
   }SHELL  0=  IF  SHELL-END  ENDIF ;
: TRANSLATE  ( -- )  ( for running Timbre without affecting open files )
   PUSH-FILE  0 file !  TRANSLATE-FILE  POP-FILE ;

( ==== File interface ==== )
: TRANSLATOR-INIT  ( -- )  inputq 0Q  outputq 0Q  dictq 0Q  0 check-rule ! ;
: TLD  ( "file" | s -- )  ( translate a file )
   BL WORD  HERE C@  IF  HERE  ENDIF  FILE  ?FILE
   ' TRANSLATOR-INIT LATE EXECUTE  SHELL{ TRANSLATE-FILE }SHELL  CLOSE
   0= IF  ABORT  ENDIF ;

( Rules Tools )
( ==== Rule printer ==== )
: GET-PHRASE  ( -- a )  rule @  CELL +  @ ;
: .PHRASE  ( a -- )  BEGIN  @+ SWAP ?DUP  WHILE  COUNT TYPE SPACE  REPEAT  DROP ;
: .{PHRASE}  ( a -- )  ." { "  .PHRASE  ." }" FLUSH-EMITS ;
: .METHOD  ( a -- )  ?DUP
   IF  CELL +  ." [ "  " SEE" latest SEARCH-DICTIONARY
      IF  L>TICK EXECUTE  ELSE  2DROP  ENDIF   ." ]"
   ENDIF ;
: .SUBSTITUTE  ( a -- )  ?DUP  IF  .{PHRASE}  ENDIF ;
: .RULE  ( rule -- )  ?CR
   @+ SWAP .{PHRASE}  @+ SWAP .METHOD  @ .SUBSTITUTE  FLUSH-EMITS ?STOP ;
: DOTRULES  ( -- )  check-rule @  BEGIN  ?DUP  WHILE  @+ .RULE  REPEAT ;
: RULE  ( -- )  inputq 0Q  GET-INPUT  0TIB
   IF  FIND-RULE?  IF  DOTRULES  ELSE  ." no rule."  ENDIF  ENDIF ;

( ==== Q printer ==== )
: .Q  ( q -- )  DUP Q?  FOR  DUP Q>  DUP COUNT TYPE SPACE OVER >Q  NEXT  DROP ;
: .QS  ( -- )  ?CR  FLUSH-EMITS ?STOP
   ." inputq: " inputq .Q  ."  outputq: " outputq .Q ;

( ==== Rule display ==== )
: .REPORT  ( n \ n \ n \ n -- )  - ?DUP
   IF  DUP NEGATE ALLOT  ." Dictionary grew by " .D  ." bytes.  "  ENDIF
   - ?DUP  IF  .S SP! ENDIF  ' [ EXECUTE ;
: SHOWME  ( -- )  ?CR  TRANSLATOR-INIT  HERE >R  DEPTH >R
   YES rule-echo !  SHELL{ TRANSLATE }SHELL DROP  DEPTH R>  HERE R>  .REPORT
   0TIB  NO rule-echo ! ;

( ==== Dump a rule ==== )
: DOTRULE  ( rule -- )  DUP 0=  IF  DROP  ." No rule."  EXIT  ENDIF 
   ?CR  ." Next rule: " @+ SWAP .
   CR  ." Phrase:"  @+ SWAP DUP .H  .PHRASE
   CR  ." Method:"  @+ SWAP  ?DUP
   IF  DUP .H  ( #IFDEF DECODE_TILL_EXIT  DECODE_TILL_EXIT  #ELSE  DROP  #ENDIF)
       DROP
   ELSE  ."  not!"  ENDIF
   CR  ." Substitute:"  @ ?DUP  IF  DUP .H  .PHRASE  ELSE  ."  not!"  ENDIF ;
: .LAST-RULE  ( -- )  ?CR  ." cache-size: " rule-size @ .
   rule @  ?DUP  IF  CELL + .RULE  ENDIF
   CR ." rule: " check-rule @ ?DUP  IF  CELL + .RULE  ENDIF ;

( ==== Timber status ==== )
: .TIMBRE  ( -- )  ?CR  ruleq Q?  .  ." unchecked rule sets.  "
   dictq Q? .  ." checked rule sets.  "  .QS   .LAST-RULE
   RULE?  IF  RULE-DONE?  IF  ." current RULE done"  ENDIF  ENDIF ;

( ==== Print phrase in outputq ==== )
: ?RMARGIN  ( n -- n )  DUP 2 + 77 >  IF  CR  ENDIF ;

: D>TABLE    ( dict -- string table \ capacity )  @+ @ ;
: D>ADJUNCT  ( dict -- adjunct table \ capacity )  CELL + @+  CELL + @  SWAP ;

: .DICT  ( dictionary -- )  D>TABLE
   FOR  @+ SWAP ?DUP  IF  COUNT ?RMARGIN TYPE SPACE  ENDIF  NEXT  DROP ;

: .WORDS  ( -- )  rule-words @ .DICT ;
: .RULES  ( -- )  ruleq Q  DUP .DICT  D>ADJUNCT
   FOR  @+ SWAP ?DUP  IF  check-rule ! DOTRULES  ENDIF  NEXT  DROP ;
