/*
 * forth2c.c
 *	Library functions for the Forth to C converter.
 *	(C) Martin Maierhofer 1994
 *	m.maierhofer@ieee.org
 */

#include "forth.h"
#include "forth2c.h"
	    
/*
 * Some variables/functions for runtime support.
 * 1) _C_base:          contains the current base for convertion
 * 2) _C_data et al.:   this variable(s)/functions add support for the
 *              runtime data allocation provided by the 'comma' operator(s)
 * 3) _C_pic_buffer:    numeric pictured output buffer (<# .. #>)
 * 4) _C_pad:           pad
 */

Cell    _C_state = 0;		/* represents the current state */
Cell	_C_base = 10;		/* represents current base */
Char *	_C_data = NULL;		/* base pointer of data space */
Char *	_C_here = NULL;		/* represents 'here' */
int	_C_pic_buffer_index = 0;	/* index into _C_pic_buffer */
char	_C_pic_buffer[_C_PICBUFFER_SIZE];	/* pictured numeric output */
char    _C_pad[_C_PAD_SIZE];	/* the forth pad */
      
/*
 * _C_request_mem
 *      Takes the number of bytes for which to reserve data space and
 *      adjusts _C_here. When called for the first time (_C_data == NULL),
 *      we have to request some memory (fixed size, but this way it is
 *      _much_ easier and faster to implement)
 */
void _C_request_mem(Cell nrbytes)
{
        if (_C_data == NULL)
        {
                if ((_C_data = malloc(_C_DATA_SIZE)) == NULL)
                {
                        fprintf(stderr, "Can't allocate memory: %s\n",
                                strerror(errno));
                        exit(1);
                }
                
                _C_here =  _C_data;
        }
                 
        _C_here += nrbytes;             /* and a sanity check */
        if (_C_here >= _C_data + _C_DATA_SIZE)
        {
                fprintf(stderr, "Too much memory allocated by forth. "
                        "Maybe you should increment _C_DATA_SIZE.\n");
                exit(1);
        }
}
		       
/*
 * _C_fetch_here
 *	Returns 'here'. Take care to reserve some data space if this
 *	has not yet been done.
 */
Char * _C_fetch_here(void)
{
	if (_C_here == NULL) _C_request_mem(0);
	return _C_here;
}

/*
 * _C_pic_add_char
 *      Add a character to the pictured output buffer.
 *      NOTE: the buffer is filled from the end to the beginning and
 *            we have to leave room for the terminating '\0'
 */
void _C_pic_add_char(char c)
{
int     index = _C_PICBUFFER_SIZE - _C_pic_buffer_index - 1;

        if (_C_pic_buffer_index + 1 < _C_PICBUFFER_SIZE)
                _C_pic_buffer[index] = c;
                
        _C_pic_buffer_index++;
}
 
/*
 * _C_pic_add_digit
 *      Add the digit contained in the parameter to the pictured output
 *      buffer.
 */
void _C_pic_add_digit(int dig)
{
                                        /* ugly and non portable ! */
        if (dig > 9) dig += 'A' - '9' - 1;
        dig += '0';

        _C_pic_add_char(dig);        
}

/*
 * _C_to_digit
 *      Converts the character in d into a digit according to base
 *      in _C_base. Returns -1 for invalid character.
 */
int _C_to_digit(Char d)
{
char *          errstr;
unsigned long   result;
char            convstr[2];

        convstr[0] = (char) d;
        convstr[1] = '\0';
        result = strtoul(convstr, &errstr, _C_base);
        if (*errstr != '\0') return -1;
        return (result);
}

