\ *
\ * _c_funcs.fs
\ *	This file contains all of the words/variables etc.
\ *	for the Forth 2 C converter.
\ *
\ *	(C) Martin Maierhofer 1994
\ *	m.maierhofer@ieee.org
\ *

( $Id: _c_funcs.fs,v 1.32 1994/12/06 23:31:24 martin Exp $ )

\ *
\ * NOTE: all the constants used in this file are in HEX. We store the
\ * current base and restore it at the end of our file.
\ *
BASE @ HEX

\ *
\ * Constants
\ *
$5000 constant 	_C_code_size	( maximum length of C code for a word )
$30 constant 	_C_funcname_size	( maximum length of function name )
                                                                
$20 constant 	_C_cf_buffer_size	( size of control flow buffer )

$a constant 	_C_nl_char		( the newline character )
$9 constant	_C_tab_char		( the tab character )
$20 constant 	_C_space_char		( the space character )

\ *
\ * Variables
\ *
				( the name of the function we are )
				( building in _C_code )
create 		_C_funcname _C_funcname_size chars allot

				( the name of the variable/constant )
				( currently being defined )                                              
create 		_C_wordname _C_funcname_size chars allot
				( buffer for a 'virtual stack' holding )
				( 1. label numbers for control flow )
				( 2. local register number at 'if' -- )
				(	this is needed by 'else' )
				( 3. local register number for return )
				(	stack at 'if' )
create 		_C_cf_buffer _C_cf_buffer_size $3 * cells allot
variable 	_C_cf_buffer_index	( index into _C_cf_buffer )

				( buffer for 'leave' with loops; this is )
				( quite tricky ;-)
create		_C_cf_leave_buffer _C_cf_buffer_size cells allot
variable	_C_cf_leave_index
											 							  
				( storage for the C code )
create 		_C_code _C_code_size chars allot

				( index into _C_code where to store the )
variable 	_C_code_index	( next characters )
variable 	_C_nrparams	( the number of parameters to the func )
variable 	_C_maxlocals	( the number of local vars needed )
variable 	_C_curlocal	( the local var being the top of stack )
variable 	_C_maxrstack	( the number of vars for the return stack )
variable 	_C_currstack	( the top of the return stack )
variable 	_C_curlabel	( the label counter for jumps )

variable 	_C_curtab	( the number of tabs for pretty printing )

variable 	_C_wordlist	( holds the wordlist id for forth2c )
variable 	_C_allot_size	( accumulated number of cells for allot )
variable 	_C_creating	( flag, true if currently creating vars )
variable 	_C_inserting	( true if currently inserting a new name )
variable	_C_constant_value	( holds the last constant )

\ *
\ * Words
\ *

\ words handling the 'virtual stack' for control flow support, push
\ and pop entries from _C_cf_buffer
 
\ _C_cf_overflow
\	Called whenever the 'virtual stack' runs out of memory, i.e.
\	whenever control flow is nested too deep.
\ 	Currently only dumps a message.

: _C_cf_overflow		( -- )
	cr ." forth2c: This word is too complicated for me to convert it." cr
	." (I only have " _C_cf_buffer_size . ."  nesting levels for control flow)" cr
	." [This message could appear soon again !]" cr
;

\ _C_cf_underflow
\	Called whenever the 'virtual stack' underflows, i.e. there is
\	something wrong with the control flow code (in forth2c or in the
\	code to be converted). Currently only dumps a message.

: _C_cf_underflow		( -- )
	cr ." forth2c: Oops, control flow buffer underflow !" cr
	." (Maybe your control flow words are incorrectly nested -- or mine are)" cr
	." [This message could appear soon again !]" cr
;

\ _C_cf_push
\	Push an entry to the 'virtual stack'. Check for overflow, push the
\	three parameters at 3 * _C_cf_buffer_index (one entry == 3 cells).
\	and increment _C_cf_buffer_index.

: _C_cf_push			( n1 n2 n3 -- )
	_C_debug if 
		." forth2c: at _C_cf_push(" 
		2>r dup . 2r>
		over .
		dup .
		." )" cr 
	then
	
	_C_cf_buffer_index @		( n1 n2 n3 index )
   
	dup _C_cf_buffer_size >= if
		_C_cf_overflow
		2drop 2drop exit
	then

   	3 * cells _C_cf_buffer +	( n1 n2 n3 3*index )
	tuck ! cell+			( n1 n2 3*index+1 )
	tuck ! cell+			( n1 3*index+2 )
	!				( -- )
   
   	1 _C_cf_buffer_index +!
;

\ _C_cf_pop
\	Pop an entry off the 'virtual stack'. Check for underflow, decrement
\	_C_cf_buffer_index and fetch three cells -- their order on the stack
\	should be the same as it was when _C_cf_push has been called.

: _C_cf_pop			( -- n1 n2 n3 )
	_C_debug if ." forth2c: at _C_cf_pop" cr then
	
	_C_cf_buffer_index @		( index )
   
	1- dup 0< if
		_C_cf_underflow
		drop -1 -1 -1 exit	\ need some dummy values
	then

	3 * cells _C_cf_buffer +	( 3*index )
        
	dup @ swap cell+		( n1 3*index+1 )
	dup @ swap cell+		( n1 n2 3*index+2 )
	@        			( n1 n2 n3 )
	
	swap rot             	\ restore order as in _C_cf_push
	
	-1 _C_cf_buffer_index +!
;

\ _C_cf_leave_push
\	Push one cell to the leave stack. Check for overflow and adjust
\	_C_cf_leave_index.

: _C_cf_leave_push		( n -- )
	_C_debug if ." forth2c: at _C_cf_leave_push( " dup . ." )" cr then

	_C_cf_leave_index @		( n index )
	dup _C_cf_buffer_size >= if
		_C_cf_overflow
		2drop exit
	then
	
	cells _C_cf_leave_buffer + !
	1 _C_cf_leave_index +!	\ adjust the index into the leave stack
;

\ _C_cf_leave_pop
\	Pop one cell off the leave stack.

: _C_cf_leave_pop		( -- n )
	_C_debug if ." forth2c: at _C_cf_leave_pop" cr then

	_C_cf_leave_index @		( n index )
	1- dup 0< if		\ ignore underflow -- like kernal.fs
		drop 0 exit
	then
	
	cells _C_cf_leave_buffer + @
	-1 _C_cf_leave_index +!	\ adjust the index into the leave stack
;
 
\ _C_cf_pick
\	Pick the n'th item from the control flow 'stack' and copy it to
\	the top of the stack. n is zero-based.

: _C_cf_pick              	( n -- n )
	_C_debug if ." forth2c: at _C_cf_pick(" dup . ." )" cr then
                                   
   	>r           			( ; r: n )
                        	\ pop n items and save them at the stack
   	r@ 0 ?do
      		_C_cf_pop
   	loop
	                          	( 3*n cells ; r: n )
	                           
	                   	\ make a copy of the item to pick
	_C_cf_pop                  	( 3*n_cells n1 n2 n3 ; r: n )
	2dup                       	( 3*n_cells n1 n2 n3 n2 n3 ; r: n )
	4 pick                     	( 3*n_cells n1 n2 n3 n2 n3 n1 ; r: n )
	rot rot                    	( 3*n_cells n1 n2 n3 n1 n2 n3 ; r: n )
	_C_cf_push                 	( 3*n_cells n1 n2 n3 ; r: n )
	
	r@ rot rot 2>r swap >r     	( 3*n_cells n ; r: n n2 n3 n1 )
	
	0 ?do                	\ restore the cf 'stack'
		_C_cf_push
	loop                          
	                           	( r: n n2 n3 n1 )
                        	\ and push the picked item
    	r> 2r> _C_cf_push         	( r: n )
    	r>                        	( n )
;

\ _C_cf_roll
\	Pick the n'th item from the control flow 'stack' and move it to
\	the top of the stack. n is zero-based.

: _C_cf_roll           		( n -- n )
	_C_debug if ." forth2c: at _C_cf_roll(" dup . ." )" cr then
   
	>r                     		( ; r: n )
	r@ 0 ?do		\ pop n items and save them at the stack
		_C_cf_pop
	loop
	                           	( 3*n cells ; r: n )
	                     	\ fetch the item to roll
	_C_cf_pop                  	( 3*n_cells n1 n2 n3 ; r: n )
	r@ rot rot 2>r swap >r     	( 3*n_cells n ; r: n n2 n3 n1 )
	
	0 ?do                	\ restore the cf 'stack'
		_C_cf_push
	loop                          
	                          	( r: n n2 n3 n1 )
	                               
	                    	\ and push the picked item
	r> 2r> _C_cf_push         	( r: n )
	r>                        	( n )
;
  


\ words dealing with storing strings to the temporary storage for the
\ current function (i.e. into _C_code) 

\ _C_mem_code_too_long
\	Called whenever _C_mem_store or _C_mem_store_char would result in
\	writing out of bounds. Currently only dumps a message.

: _C_mem_code_too_long		( -- )
	cr ." forth2c: This word is too long for me to convert it." cr
	." (I only have " _C_code_size . ."  characters to store the code)" cr
	." [This message could appear soon again !]" cr
;

\ _C_mem_store
\	Add the string on the stack to _C_code; adjust _C_code_index.
\	MM: could (and should) be optimized

: _C_mem_store			( c-addr u -- )
	dup 				( c-addr u u )
	
			\ now check whether we would write out of bounds
	_C_code_index @ dup >r		( c-addr u u index ; r: index )
	+ _C_code_size > if		( c-addr u ; r: index )
		_C_mem_code_too_long	
		2drop rdrop
		exit	
	then
	
	r>				( c-addr u index )
	_C_code swap chars +		( c-addr u c-addr2 )
	over >r				( c-addr u c-addr2 ; r: u )
	swap cmove			( -- ; r: u )
	r> _C_code_index +!		( -- )
;

\ _C_mem_store_char
\	Add the character on the stack to _C_code; update _C_code_index

: _C_mem_store_char		( c -- )
	_C_code_index @ dup >r		( c index ; r: index )
	
	_C_code_size 1- > if		( c ; r: index )
		_C_mem_code_too_long
		drop rdrop
		exit
	then
	
	_C_code r> chars + c!		( -- )
	1 _C_code_index +!		( -- )
;

\ _C_mem_nl
\	Write a newline to _C_code (via _C_mem_store_char). In the new line,
\	add _C_curtab tab characters (for pretty printing)

: _C_mem_nl			( -- )
	_C_nl_char _C_mem_store_char
	_C_curtab @ 0 ?do
		_C_tab_char _C_mem_store_char
	loop
;

\ _C_mem_store_nl
\	Short for '_C_mem_store _C_mem_nl'

: _C_mem_store_nl      		( c-addr u -- )
	_C_mem_store _C_mem_nl
;
						 
\ _C_mem_beginblock
\ 	Begin a block of code for a primitive or a function call etc, i.e.
\	write a curly brace to _C_code; then write the string given on the
\	stack; afterwards increase _C_curtab and dump a newline.

: _C_mem_beginblock		( c-addr u -- )
	_C_mem_nl 
	[char] { _C_mem_store_char
	_C_tab_char _C_mem_store_char
	1 _C_curtab +!
	_C_mem_store_nl		\ write the string
;

\ _C_mem_endblock
\	The reverse of _C_mem_beginblock

: _C_mem_endblock		( -- )
	-1 _C_curtab +!
	_C_mem_nl 
	[char] } _C_mem_store_char
	_C_mem_nl
;



\ words dealing with storing strings and characters to the output file

\ _C_store
\	Write the given string to the output file. Watch out if _C_fileid
\	is 0 (something that never should happen, as it means that the
\	output file has not yet been opened)

: _C_store			( c-addr u -- )
	_C_fileid @ 0= 			( c-addr u )	
	if
		." forth2c: oops, _C_store called with _C_fileid == 0 !" cr
		2drop exit
	then
	
	_C_fileid @ write-file		( flag )
	if
		." forth2c: write-file failed. No more C code is generated." cr
		." [Some more error messages may be the result of this one]." cr
		
		0 _C_fileid !	\ invalidate the file
	then
;

\ _C_store_char
\	Store a character to the output file via _C_store

: _C_store_char			( c -- )
												 
				\ build a 'string' containing the character
				\ in the pad and call _C_store
	1 pad dup >r dup >r		( c 1 c-addr ; r: c-addr c-addr )
	! r> char+ c! 			( -- ; r: c-addr )
	r> count _C_store		( -- )
;

\ _C_nl
\ 	Write a newline to the output file (via _C_store_char)

: _C_nl				( -- )
	_C_nl_char  _C_store_char
;

\ _C_tab
\ 	Write a tab to the output file (via _C_store_char)

: _C_tab			( -- )
	_C_tab_char _C_store_char
;



\ some words providing low level functionality dealing with local
\ variables and parameters (e.g. you can get the next parameter in
\ string representation and modify the counter variables accordingly)

\ _C_one_more_local
\ 	Adjust _C_maxlocals if _C_curlocal > _C_maxlocals. Called
\ 	whenever a new local variable is needed

: _C_one_more_local		( -- )
	1 _C_curlocal +!	\ increment the index of the next local var
	_C_curlocal @ _C_maxlocals @	( n n )
	max _C_maxlocals !
;

\ _C_one_more_rstacklocal
\ 	Adjust _C_maxrstack if _C_currstack > _C_maxrstack. Called
\ 	whenever a new rstack variable is needed.

: _C_one_more_rstacklocal	( -- )
	1 _C_currstack +!	\ increment the index of the next local var
	_C_currstack @ _C_maxrstack @	( n n )
	max _C_maxrstack !
;
  		
\ _C_get_next_param
\ 	Get the next parameter for a word. Returns either 'x?' if
\ 	there are still local variables as parameters; otherwise
\ 	'p?' is returned. _C_curlocal and _C_nrparams are adjusted
\ 	accordingly.

: _C_get_next_param		( -- c-addr u )
	-1 _C_curlocal +!
	_C_curlocal @ dup 0 >= if
				\ there are still local variables
		s>d <# #s [char] x hold #>	( c-addr u )
	else						
                        	\ we have to switch to parameters
                        	
		abs dup >r			( n r: n )
				\ use 1- because parameters start with
                        	\ p0 and not p1		
		1- s>d <# #s [char] p hold #>	( c-addr u r: n)

                        	\ adjust _C_nrparams if we have used a new
                        	\ parameter
      		r@ _C_nrparams @ > if
         		r> _C_nrparams !
      		else
         		rdrop
      		then
	then
;

\ _C_next_locvar
\ 	Get the next free local variable and increment _C_curlocal
\	(and maybe _C_maxlocal) via _C_one_more_local

: _C_next_locvar 		( -- c-addr u )
	_C_curlocal @ 		\ create string represenation
   	dup 0 < if
		abs 1- s>d <# #s [char] p hold #>
   	else
	   	s>d <# #s [char] x hold #>
   	then
	
	_C_one_more_local	\ reserve next local variable
;

\ _C_next_rstackvar
\ 	Get the next free local variable for the return stack and increment 
\ 	_C_currstack (and maybe _C_maxrstack) via _C_one_more_rstacklocal

: _C_next_rstackvar 		( -- c-addr u )

	_C_currstack @		\ create string represenation
	s>d <# #s [char] r hold #>
	_C_one_more_rstacklocal		
;

\ _C_conc_param
\	This word takes a string on the stack, concatenates the next
\	parameter and a semicolon and writes the whole thing to _C_code
\	e.g. you use it like 
\		s" n = " _C_conc_param
\

: _C_conc_param			( c-addr u -- )
	_C_mem_store
	_C_get_next_param _C_mem_store
	[char] ; _C_mem_store_char _C_mem_nl
;

\ _C_conc_rstack
\	This word takes a string on the stack, concatenates the 'top of
\	the return stack' and a semicolon and writes the whole thing to 
\	_C_code. _C_currstack is decremented by one.

: _C_conc_rstack		( c-addr u -- )
   	-1 _C_currstack +!   \ adjust _C_currstack
	_C_mem_store         \ save the string
   	_C_currstack @       \ create something like 'r?'
   	s>d <# #s [char] r hold #> _C_mem_store
	[char] ; _C_mem_store_char _C_mem_nl
;
    
\ _C_locvar_conc
\	This word assigns the whatever contained in the word on the stack
\	to the next free local variable (_C_next_locvar).
\	NOTE: Do NOT use this word for strings generated via <# ... #>;
\	      Usage with strings generated by s" should be safe.
\	you could use it like
\		s"  = x / y;" _C_locvar_conc

: _C_locvar_conc		( c-addr u -- )
	_C_next_locvar _C_mem_store
	_C_mem_store_nl
;

\ _C_rstackvar_conc
\	This word assigns the whatever contained in the word on the stack
\	to the next free local variable of the 'return stack' 
\	(_C_next_rstackvar).
\	NOTE: Do NOT use this word for strings generated via <# ... #>;
\	      Usage with strings generated by s" should be safe.

: _C_rstackvar_conc		( c-addr u -- )
	_C_next_rstackvar _C_mem_store
	_C_mem_store_nl
;


\ Some words helping with the generation of C code, e.g. you can find
\ Words for common binary operations (+, -, and, mod etc.) here

\ _C_binary_op
\ 	Generate C code for a binary operating (i.e. taking two parameters
\	and returning one). The top of stack contains the number of characters
\	constituating the ascii representation of the operator in C (in
\	reverse order ! 
\	E.g. to dump a '!=' you would have to write
\		[char] = [char] ! 2 _C_binary_op

: _C_binary_op			( i*c n -- )
	s" Cell n1, n2, n;" _C_mem_store_nl
       
                        	\ get two parameters
   	s" n1 = " _C_conc_param
   	s" n2 = " _C_conc_param _C_mem_nl													  
	s" n = n2 " _C_mem_store 
				\ now write the operator characters
	0 ?do	_C_mem_store_char loop
   	s"  n1;" _C_mem_store_nl
				\ assign 'n' to topmost local variable
	s"  = n;" _C_locvar_conc
;

\ _C_boolean_binary
\	Mostly the same as _C_binary_op. The only difference is the
\	generation of 'n = FLAG'. I wonder whether these two words
\	could be factore better ?
 
: _C_boolean_binary		( i*c n -- )
	s" Cell n1, n2, n;" _C_mem_store_nl
                        	\ get two parameters
   	s" n1 = " _C_conc_param
   	s" n2 = " _C_conc_param _C_mem_nl													  
	s" n = FLAG(n2 " _C_mem_store 
   				\ now write the operator characters
	0 ?do	_C_mem_store_char loop
	s"  n1);" _C_mem_store_nl
				\ assign 'n' to topmost local variable
	s"  = n;" _C_locvar_conc
;

\ _C_boolean_zero_unary
\     	Mostly the same as _C_boolean_binary, but use '0' as the
\     implicit second operand.
 
: _C_boolean_zero_unary		( i*c n -- )
	s" Cell n1, n;" _C_mem_store_nl
                               	\ get one parameter
   	s" n1 = " _C_conc_param _C_mem_nl													  
	s" n = FLAG(n1 " _C_mem_store 
   				\ now write the operator characters
	0 ?do	_C_mem_store_char loop
	s"  0);" _C_mem_store_nl
				\ assign 'n' to topmost local variable
	s"  = n;" _C_locvar_conc
;

\ words for flow control support generating code for labels und
\ conditional and unconditional jumps

\ _C_make_goto_if
\	Generate C code like 'if (!x?) goto labely;' where 'x?' is the next
\     	parameter on the virtual stack and 'y' is the label number given
\     	as an input parameter. This word is used by _C_if_hook, _C_while_hook

: _C_make_goto_if          	( n -- )

	_C_mem_nl
	s" if (!" _C_mem_store
	_C_get_next_param _C_mem_store
	s" ) goto label" _C_mem_store 
	s>d <# [char] ; hold #s #> _C_mem_store_nl
;

\ _C_make_label
\     	Generates C code for a label like 'labely:' where 'y' is given
\     	as an input parameter.

: _C_make_label            	( n -- )

	_C_mem_nl		
	s" label" _C_mem_store
	s>d <# [char] : hold #s #> _C_mem_store_nl _C_mem_nl
;

\ _C_make_goto
\     	Generates C code for a goto statement like 'goto labely;' where
\     	'y' is given as an input parameter.

: _C_make_goto             	( n -- )

	_C_mem_nl s" goto label" _C_mem_store
	s>d <# [char] ; hold #s #> _C_mem_store_nl
;

\ words generating C code for the forth primitives. The primitive to be
\ converted can easily by recognized from the name of the generating word.

\ _C_+
\ 	Generate C code for '+'.

: _C_+				( -- )
	_C_debug if ." forth2c: at _C_+" cr then
											
	s" /* plus */" _C_mem_beginblock
	[char] + 1 _C_binary_op	
	_C_mem_endblock
;

\ _C_-
\ 	Generate C code for '-'.

: _C_-				( -- )
	_C_debug if ." forth2c: at _C_-" cr then

	s" /* minus */" _C_mem_beginblock
	[char] - 1 _C_binary_op
	_C_mem_endblock
;

\ _C_*
\ 	Generate C code for '*'.

: _C_*				( -- )
	_C_debug if ." forth2c: at _C_*" cr then

	s" /* star */" _C_mem_beginblock
	[char] * 1 _C_binary_op
	_C_mem_endblock
;

\ _C_/
\ 	Generate C code for '/'.

: _C_/				( -- )
	_C_debug if ." forth2c: at _C_/" cr then

	s" /* slash */" _C_mem_beginblock
	[char] / 1 _C_binary_op
	_C_mem_endblock
;

\ _C_and
\ 	Generate C code for 'and'.

: _C_and			( -- )
	_C_debug if ." forth2c: at _C_and" cr then
											
	s" /* and */" _C_mem_beginblock
	[char] & 1 _C_binary_op	
	_C_mem_endblock
;

\ _C_or
\ 	Generate C code for 'or'.

: _C_or				( -- )
	_C_debug if ." forth2c: at _C_or" cr then
											
	s" /* or */" _C_mem_beginblock
	[char] | 1 _C_binary_op	
	_C_mem_endblock
;

\ _C_xor
\ 	Generate C code for 'xor'.

: _C_xor			( -- )
	_C_debug if ." forth2c: at _C_xor" cr then
											
	s" /* xor */" _C_mem_beginblock
	[char] ^ 1 _C_binary_op	
	_C_mem_endblock
;

\ _C_.
\ 	Generate C code for '.'.

: _C_. 				( -- )
	_C_debug if ." forth2c: at _C_." cr then
	
	s" /* dot */" _C_mem_beginblock
       				\ generate a printf statement, which is
       				\ a bit complicated because of the '"' ...
	s" printf(" _C_mem_store	\ 'printf('
	[char] " _C_mem_store_char	\ 'printf("'
	s" %d " _C_mem_store		\ 'printf(" %d'
	[char] " _C_mem_store_char	\ 'printf(" %d"'
	s" , " _C_mem_store		\ 'printf(" %d", '
	_C_get_next_param _C_mem_store	\ 'printf(" %d", x?'
	s" );" _C_mem_store_nl		\ 'printf(" %d", x?);'
	s" fflush(stdout);" _C_mem_store
	_C_mem_endblock
;	

\ _C_over
\	Generate C code for 'over'. 

: _C_over			( -- )
	_C_debug if ." forth2c: at _C_over" cr then

	s" /* over */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl _C_mem_nl
	s" n2 = " _C_conc_param
	s" n1 = " _C_conc_param _C_mem_nl
	s"  = n1;" _C_locvar_conc
	s"  = n2;" _C_locvar_conc
	s"  = n1;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_drop
\	Generate C code for 'drop'. Doesn't generate 'real' C code

: _C_drop			( -- )
	_C_debug if ." forth2c: at _C_drop" cr then
	
	s" /* drop */" _C_mem_beginblock
	_C_get_next_param 2drop
	_C_mem_endblock
;

\ _C_swap
\ 	Generate C code for 'swap'.

: _C_swap 			( -- )
	_C_debug if ." forth2c: at _C_swap" cr then
	
	s" /* swap */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param _C_mem_nl
	s"  = n1;" _C_locvar_conc
	s"  = n2;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_dup
\	Generate C code for 'dup'.

: _C_dup			( -- )
	_C_debug if ." forth2c: at _C_dup" cr then
	
	s" /* dup */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param _C_mem_nl
	s"  = n;" _C_locvar_conc
	s"  = n;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_rot
\	Generate C code for 'rot'.

: _C_rot			( -- )
	_C_debug if ." forth2c: at _C_rot" cr then
									
	s" /* rot */" _C_mem_beginblock
	s" Cell n1, n2, n3;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" n3 = " _C_conc_param _C_mem_nl
	s"  = n2;" _C_locvar_conc
	s"  = n1;" _C_locvar_conc
	s"  = n3;" _C_locvar_conc
	_C_mem_endblock
;


\ _C_2dup
\	Generate C code for '2dup'.

: _C_2dup			( -- )
	_C_debug if ." forth2c: at _C_2dup" cr then
	
	s" /* 2dup */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param _C_mem_nl
	s"  = n2;" _C_locvar_conc
	s"  = n1;" _C_locvar_conc
	s"  = n2;" _C_locvar_conc
	s"  = n1;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_2over
\	Generate C code for '2over'.

: _C_2over			( -- )
	_C_debug if ." forth2c: at _C_2over" cr then

	s" /* 2over */" _C_mem_beginblock
	s" Cell n1, n2, n3, n4;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" n3 = " _C_conc_param
	s" n4 = " _C_conc_param _C_mem_nl
	s"  = n4;" _C_locvar_conc
	s"  = n3;" _C_locvar_conc
	s"  = n2;" _C_locvar_conc
	s"  = n1;" _C_locvar_conc
	s"  = n4;" _C_locvar_conc
	s"  = n3;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_2drop
\	Generate C code for '2drop'.

: _C_2drop			( -- )
	_C_debug if ." forth2c: at _C_2drop" cr then
	
	s" /* 2drop */" _C_mem_beginblock
	_C_get_next_param 2drop
	_C_get_next_param 2drop
	_C_mem_endblock
;

\ _C_2swap
\	Generate C code for '2swap'.

: _C_2swap			( -- )
	_C_debug if ." forth2c: at _C_2swap" cr then
	
	s" /* 2swap */" _C_mem_beginblock
	s" Cell n1, n2, n3, n4;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" n3 = " _C_conc_param
	s" n4 = " _C_conc_param _C_mem_nl
	s"  = n2;" _C_locvar_conc
	s"  = n1;" _C_locvar_conc
	s"  = n4;" _C_locvar_conc
	s"  = n3;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_2rot
\	Generate C code for '2rot'.

: _C_2rot			( -- )
	_C_debug if ." forth2c: at _C_2rot" cr then
									
	s" /* 2rot */" _C_mem_beginblock
	s" Cell n1, n2, n3, n4, n5, n6;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" n3 = " _C_conc_param
	s" n4 = " _C_conc_param
	s" n5 = " _C_conc_param
	s" n6 = " _C_conc_param _C_mem_nl
	s"  = n4;" _C_locvar_conc
	s"  = n3;" _C_locvar_conc
	s"  = n2;" _C_locvar_conc
	s"  = n1;" _C_locvar_conc
	s"  = n6;" _C_locvar_conc
	s"  = n5;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_<
\ 	Generate C code for '<'. 

: _C_<				( -- )
	_C_debug if ." forth2c: at _C_<" cr then

	s" /* less-than */" _C_mem_beginblock
	[char] < 1 _C_boolean_binary
	_C_mem_endblock
;

\ _C_<>
\ 	Generate C code for '<>'.

: _C_<>				( -- )
	_C_debug if ." forth2c: at _C_<>" cr then

	s" /* not-equals */" _C_mem_beginblock
	[char] = [char] ! 2 _C_boolean_binary
	_C_mem_endblock
;

\ _C_=
\ 	Generate C code for '='. 

: _C_=				( -- )
	_C_debug if ." forth2c: at _C_=" cr then

	s" /* equals */" _C_mem_beginblock
	[char] = dup 2 _C_boolean_binary
	_C_mem_endblock
;

\ _C_>
\ 	Generate C code for '>'. 

: _C_>				( -- )
	_C_debug if ." forth2c: at _C_>" cr then

	s" /* greater-than */" _C_mem_beginblock
	[char] > 1 _C_boolean_binary
	_C_mem_endblock
;

\ _C_unloop
\     Discards two parameters from the return stack (holding the loop
\     index and bound)

: _C_unloop                	( -- )
	_C_debug if ." forth2c: at _C_unloop" cr then
   
	s" /* unloop */" _C_mem_beginblock
   	-2 _C_currstack +!
   	_C_mem_endblock
;

\ _C_noop
\	Generate C code for noop, i.e. nothing.

: _C_noop 			( -- )
	_C_debug if ." forth2c: at _C_noop" cr then
	
	s" /* noop */" _C_mem_beginblock
	[char] ; _C_mem_store_char
	_C_mem_endblock
;

\ _C_move
\	Generate C code for move

: _C_move			( -- )
	_C_debug if ." forth2c: at _C_move" cr then
	
	s" /* move */" _C_mem_beginblock
	s" UCell u;" _C_mem_store_nl
	s" Char *dst, *src;" _C_mem_store_nl _C_mem_nl
	s" u = (UCell) " _C_conc_param
	s" dst = (Char *) " _C_conc_param
	s" src = (Char *) " _C_conc_param
	s" memmove(dst, src, u);" _C_mem_store
	_C_mem_endblock
;

\ _C_fill
\	Generate C code for fill;

: _C_fill			( -- )
	_C_debug if ." forth2c: at _C_fill" cr then
	
	s" /* fill */" _C_mem_beginblock
	s" UCell u;" _C_mem_store_nl
	s" Char *addr, c;" _C_mem_store_nl _C_mem_nl
	s" c = (Char) " _C_conc_param
	s" u = (UCell) " _C_conc_param
	s" addr = (Char *) " _C_conc_param
	s" memset(addr, c, u);" _C_mem_store
	_C_mem_endblock
;

		 
\ _C_negate
\	Generate C code for 'negate'.

: _C_negate			( -- )
	_C_debug if ." forth2c: at _C_negate" cr then
	
	s" /* negate */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s"  = -n;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_invert
\	Generate C code for 'invert'.

: _C_invert			( -- )
	_C_debug if ." forth2c: at _C_invert" cr then
	
	s" /* invert */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s"  = ~n;" _C_locvar_conc
	_C_mem_endblock
;


\ _C_1+
\	Generate C code for '1+'.

: _C_1+				( -- )
	_C_debug if ." forth2c: at _C_1+" cr then
	
	s" /* one_plus */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s"  = n + 1;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_1-
\	Generate C code for '1-'.

: _C_1-				( -- )
	_C_debug if ." forth2c: at _C_1-" cr then
	
	s" /* one_minus */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s"  = n - 1;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_max
\	Generate C code for 'max'.

: _C_max			( -- )
	_C_debug if ." forth2c: at _C_max" cr then
	
	s" /* max */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param _C_mem_nl
	s" if (n1 < n2) " _C_mem_store
	s"  = n2;" _C_locvar_conc 
	-1 _C_curlocal +!
	s" else " _C_mem_store
	s"  = n1;" _C_locvar_conc 
	_C_mem_endblock
;

\ _C_min
\		Generate C code for 'min'.

: _C_min			( -- )
	_C_debug if ." forth2c: at _C_min" cr then
	
	s" /* min */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param _C_mem_nl
	s" if (n1 < n2) " _C_mem_store
	s"  = n1;" _C_locvar_conc 
	-1 _C_curlocal +!
	s" else " _C_mem_store
	s"  = n2;" _C_locvar_conc 
	_C_mem_endblock
;

\ _C_abs
\	Generate C code for 'abs'.

: _C_abs			( -- )
	_C_debug if ." forth2c: at _C_abs" cr then
	
	s" /* abs */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param _C_mem_nl
	s" if (n < 0) " _C_mem_store
	s"  = -n;" _C_locvar_conc 
	-1 _C_curlocal +!
	s" else " _C_mem_store
	s"  = n;" _C_locvar_conc 
	_C_mem_endblock
;

\ _C_mod
\ 	Generate C code for 'mod'.

: _C_mod			( -- )
	_C_debug if ." forth2c: at _C_mod" cr then

	s" /* mod */" _C_mem_beginblock
	[char] % 1 _C_binary_op
	_C_mem_endblock
;

\ _C_/mod
\	Generate C code for '/mod'.

: _C_/mod			( -- )
	_C_debug if ." forth2c: at _C_/mod" cr then
	
	s" /* slash_mod */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param _C_mem_nl
	s"  = n2 % n1; /* !! is this correct? look into C standard! */" 
	_C_locvar_conc 
	s"  = n2 / n1;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_2*
\	Generate C code for '2*'.

: _C_2*				( -- )
	_C_debug if ." forth2c: at _C_2*" cr then
	
	s" /* two_star */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s"  = 2 * n;" _C_locvar_conc 
	_C_mem_endblock
;

\ _C_2/
\	Generate C code for '2/'.

: _C_2/				( -- )
	_C_debug if ." forth2c: at _C_2/" cr then
	
	s" /* two_slash */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s"  = n >> 1; /* is this still correct ? */" _C_locvar_conc 
	_C_mem_endblock
;


\ _C_fm/mod
\	Generate C code for 'fm/mod'.

: _C_fm/mod			( -- )
	_C_debug if ." forth2c: at _C_fm/mod" cr then
	
	s" /* f_m_slash_mod */" _C_mem_beginblock
	s" Cell n, n1, n2;" _C_mem_store_nl 
	s" DCell d;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" n1 = d / n;" _C_mem_store_nl
	s" n2 = d % n;" _C_mem_store_nl
	s" if (1 % -3 > 0 && (d < 0) != (n < 0) && n2 != 0)" _C_mem_store
	0 0 _C_mem_beginblock
	s" n1--;" _C_mem_store_nl
	s" n2 += n;" _C_mem_store 
	_C_mem_endblock
	s"  = n2;" _C_locvar_conc
	s"  = n1;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_sm/rem
\	Generate C code for 'sm/rem'.

: _C_sm/rem			( -- )
	_C_debug if ." forth2c: at _C_sm/rem" cr then
	
	s" /* s_m_slash_rem */" _C_mem_beginblock
	s" Cell n, n1, n2;" _C_mem_store_nl 
	s" DCell d;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" n1 = d / n;" _C_mem_store_nl
	s" n2 = d % n;" _C_mem_store_nl
	s" if (1 % -3 < 0 && (d < 0) != (n < 0) && n2 != 0)" _C_mem_store
	0 0 _C_mem_beginblock
	s" n1++;" _C_mem_store_nl
	s" n2 -= n;" _C_mem_store 
	_C_mem_endblock
	s"  = n2;" _C_locvar_conc
	s"  = n1;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_m*
\	Generate C code for 'm*'.

: _C_m*				( -- )
	_C_debug if ." forth2c: at _C_m*" cr then
	
	s" /* m_star */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl 
	s" DCell d;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param _C_mem_nl
	s" d = (DCell) n1 * (DCell) n2;" _C_mem_store_nl
	s" = _C_LOWHALF(d);" _C_locvar_conc
	s" = _C_HIGHHALF(d);" _C_locvar_conc
	_C_mem_endblock
;	

\ _C_um*
\	Generate C code for 'um*'.

: _C_um*			( -- )
	_C_debug if ." forth2c: at _C_um*" cr then
	
	s" /* u_m_star */" _C_mem_beginblock
	s" UCell u1, u2;" _C_mem_store_nl 
	s" UDCell ud;" _C_mem_store_nl _C_mem_nl
	s" u1 = (UCell) " _C_conc_param
	s" u2 = (UCell) " _C_conc_param _C_mem_nl
	s" ud = (UDCell) u1 * (UDCell) u2;" _C_mem_store_nl
	s" = _C_ULOWHALF(ud);" _C_locvar_conc
	s" = _C_UHIGHHALF(ud);" _C_locvar_conc
	_C_mem_endblock
;	


\ _C_um/mod
\	Generate C code for 'um/mod'.

: _C_um/mod			( -- )
	_C_debug if ." forth2c: at _C_um/mod" cr then
	
	s" /* u_m_slash_mod */" _C_mem_beginblock
	s" UCell u, u1, u2;" _C_mem_store_nl 
	s" UDCell ud;" _C_mem_store_nl _C_mem_nl
	s" u = (UCell) " _C_conc_param
	s" u1 = (UCell) " _C_conc_param
	s" u2 = (UCell) " _C_conc_param
	s" _C_MAKEUDCELL(ud, u1, u2);" _C_mem_store_nl _C_mem_nl
	s"  = (Cell) (ud % u);" _C_locvar_conc
	s"  = (Cell) (ud / u);" _C_locvar_conc
	_C_mem_endblock
;

\ _C_rshift
\	Generate C code for 'rshift'.

: _C_rshift			( -- )
	_C_debug if ." forth2c: at _C_rshift" cr then
	
	s" /* rshift */" _C_mem_beginblock
	s" UCell u;" _C_mem_store_nl 
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s" u = (UCell) " _C_conc_param _C_mem_nl
	s"  = u >> n;" _C_locvar_conc
	_C_mem_endblock		 
;


\ _C_lshift
\	Generate C code for 'lshift'.

: _C_lshift			( -- )
	_C_debug if ." forth2c: at _C_lshift" cr then
	
	s" /* lshift */" _C_mem_beginblock
	s" UCell u;" _C_mem_store_nl 
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s" u = (UCell) " _C_conc_param _C_mem_nl
	s"  = u << n;" _C_locvar_conc
	_C_mem_endblock		 
;

\ _C_>r
\	Generate C code for '>r'.

: _C_>r 			( -- )
	_C_debug if ." forth2c: at _C_>r" cr then
	
	s" /* to_r */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s"  = n;" _C_rstackvar_conc
	_C_mem_endblock		 
;

\ _C_r>
\	Generate C code for 'r>'.

: _C_r> 			( -- )
	_C_debug if ." forth2c: at _C_r>" cr then
	
	s" /* r_from */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_rstack
	s"  = n;" _C_locvar_conc
	_C_mem_endblock		 
;

\ _C_rdrop
\	Generate C code for 'rdrop', which is nothing at all.

: _C_rdrop  			( -- )
	_C_debug if ." forth2c: at _C_rdrop" cr then
	
	s" /* rdrop */" _C_mem_beginblock
   	-1 _C_currstack +!
	_C_mem_endblock		 
;

\ _C_2>r
\	Generate C code for '2>r'.

: _C_2>r			( -- )
	_C_debug if ." forth2c: at _C_2>r" cr then
	
	s" /* two_to_r */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param _C_mem_nl
	s"  = n2;" _C_rstackvar_conc
	s"  = n1;" _C_rstackvar_conc
	_C_mem_endblock		 
;

\ _C_2r>
\	Generate C code for '2r>'.

: _C_2r>			( -- )
	_C_debug if ." forth2c: at _C_2r>" cr then
	
	s" /* two_r_from */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_rstack
	s" n2 = " _C_conc_rstack _C_mem_nl
	s"  = n2;" _C_locvar_conc
	s"  = n1;" _C_locvar_conc
	_C_mem_endblock		 
;
 
\ _C_2rdrop
\	Generate C code for '2rdrop', which is nothing at all.

: _C_2rdrop  			( -- )
	_C_debug if ." forth2c: at _C_2rdrop" cr then
	
	s" /* two_r_drop */" _C_mem_beginblock
   	-2 _C_currstack +!
	_C_mem_endblock		 
;

\ _C_r@
\	Generate C code for 'r@'.

: _C_r@ 			( -- )
	_C_debug if ." forth2c: at _C_r@" cr then
	
	s" /* r_fetch */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_rstack
   	1 _C_currstack +!    \ undo _C_next_rstackvar as this is a copy !
	s"  = n;" _C_locvar_conc
	_C_mem_endblock		 
;

\ _C_2r@
\	Generate C code for '2r@'.

: _C_2r@			( -- )
	_C_debug if ." forth2c: at _C_2r@" cr then
	
	s" /* two_r_fetch */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_rstack
	s" n2 = " _C_conc_rstack _C_mem_nl
   	2 _C_currstack +!    \ undo _C_next_rstackvar as this is a copy !
	s"  = n2;" _C_locvar_conc
	s"  = n1;" _C_locvar_conc
	_C_mem_endblock		 
;
       
\ _C_cell+
\	Generate C code for 'cell+'.

: _C_cell+                 	( -- )
	_C_debug if ." forth2c: at _C_cell+" cr then
	
	s" /* cell+ */" _C_mem_beginblock
	s" Cell * a;" _C_mem_store_nl _C_mem_nl
	s" a = (Cell *) " _C_conc_param
	s"  = (Cell) (a + 1);" _C_locvar_conc
	_C_mem_endblock
;

\ _C_cells
\     	Generate C code for 'cells'.

: _C_cells                 	( -- )
	_C_debug if ." forth2c: at _C_cells" cr then
	
	s" /* cells */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s"  = n * sizeof(Cell);" _C_locvar_conc
	_C_mem_endblock
;

\ _C_char+
\     	Generate C code for 'char+'.

: _C_char+                 	( -- )
	_C_debug if ." forth2c: at _C_char+" cr then
	
	s" /* char+ */" _C_mem_beginblock
	s" Char * a;" _C_mem_store_nl _C_mem_nl
	s" a = (Char *) " _C_conc_param
	s"  = (Cell) (a + 1);" _C_locvar_conc
	_C_mem_endblock
;

\ _C_chars
\     	Generate C code for 'chars'.

: _C_chars                 	( -- )
	_C_debug if ." forth2c: at _C_chars" cr then
	
	s" /* chars */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s"  = n * sizeof(Char);" _C_locvar_conc
	_C_mem_endblock
;

\ _C_count
\     	Generate C code for 'count'.

: _C_count                 	( -- )
	_C_debug if ." forth2c: at _C_count" cr then
	
	s" /* count */" _C_mem_beginblock
   	s" Char *a;" _C_mem_store_nl
	s" UCell u;" _C_mem_store_nl _C_mem_nl
	s" a = (Char *) " _C_conc_param
   	s" u = (UCell) *a;" _C_mem_store_nl
	s"  = (Cell) (a + 1);" _C_locvar_conc
   	s"  = (Cell) u;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_,
\     	Generate C code for ','.

: _C_,                     	( -- )
	_C_debug if ." forth2c: at _C_," cr then
	
	s" /* comma */" _C_mem_beginblock
	s" Cell *a, n;" _C_mem_store_nl _C_mem_nl
	s" a = (Cell *) _C_fetch_here();" _C_mem_store_nl
	s" n = " _C_conc_param _C_mem_nl
	s" *a = n;" _C_mem_store_nl
	s" _C_request_mem(sizeof(Cell));" _C_mem_store
	_C_mem_endblock
;

       
\ _C_!
\     	Generate C code for '!'.

: _C_!                     	( -- )
	_C_debug if ." forth2c: at _C_!" cr then
	
	s" /* store */" _C_mem_beginblock
	s" Cell *a, n;" _C_mem_store_nl _C_mem_nl
	s" a = (Cell *) " _C_conc_param
	s" n = " _C_conc_param _C_mem_nl
	s" *a = n;" _C_mem_store
	_C_mem_endblock
;

\ _C_+!
\     	Generate C code for '+!'.

: _C_+!                    	( -- )
	_C_debug if ." forth2c: at _C_+!" cr then
	
	s" /* plus_store */" _C_mem_beginblock
	s" Cell *a, n;" _C_mem_store_nl _C_mem_nl
	s" a = (Cell *) " _C_conc_param
	s" n = " _C_conc_param _C_mem_nl
	s" *a += n;" _C_mem_store
	_C_mem_endblock
;

\ _C_0<
\     	Generate C code for '0<'.

: _C_0<                    	( -- )
	_C_debug if ." forth2c: at _C_0<" cr then

	s" /* zero-less */" _C_mem_beginblock
	[char] < 1 _C_boolean_zero_unary
	_C_mem_endblock
;

\ _C_0<>
\     	Generate C code for '0<>'.

: _C_0<>                   	( -- )
	_C_debug if ." forth2c: at _C_0<>" cr then

	s" /* zero-not-equals */" _C_mem_beginblock
	[char] = [char] ! 2 _C_boolean_zero_unary
	_C_mem_endblock
;

\ _C_0>
\     	Generate C code for '0>'.

: _C_0>                    	( -- )
	_C_debug if ." forth2c: at _C_0>" cr then

	s" /* zero-greater */" _C_mem_beginblock
	[char] > 1 _C_boolean_zero_unary
	_C_mem_endblock
;

\ _C_0=
\     	Generate C code for '0='.

: _C_0=                    	( -- )
	_C_debug if ." forth2c: at _C_0=" cr then

	s" /* zero-equals */" _C_mem_beginblock
	[char] = [char] = 2 _C_boolean_zero_unary
	_C_mem_endblock
;

\ _C_2!
\     	Generate C code for '2!'.

: _C_2!                    	( -- )
	_C_debug if ." forth2c: at _C_+!" cr then
	
	s" /* two_store */" _C_mem_beginblock
	s" Cell *a, n1, n2;" _C_mem_store_nl _C_mem_nl
	s" a = (Cell *) " _C_conc_param
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param _C_mem_nl
	s" a[0] = n1;" _C_mem_store_nl
	s" a[1] = n2;" _C_mem_store
	_C_mem_endblock
;

\ _C_2@
\     	Generate C code for '2@'.

: _C_2@                    	( -- )
	_C_debug if ." forth2c: at _C_2@" cr then
	
	s" /* two_fetch */" _C_mem_beginblock
	s" Cell *a;" _C_mem_store_nl _C_mem_nl
	s" a = (Cell *) " _C_conc_param
	s"  = a[1];" _C_locvar_conc
	s"  = a[0];" _C_locvar_conc
	_C_mem_endblock
;

\ _C_@
\     	Generate C code for '@'.

: _C_@                     	( -- )
	_C_debug if ." forth2c: at _C_@" cr then
	s" /* fetch */" _C_mem_beginblock
	s" Cell *a;" _C_mem_store_nl _C_mem_nl
	s" a = (Cell *) " _C_conc_param
	s"  = *a;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_c@
\     	Generate C code for 'c@'.

: _C_c@                    	( -- )
	_C_debug if ." forth2c: at _C_c@" cr then

	s" /* cfetch */" _C_mem_beginblock
	s" Char *a;" _C_mem_store_nl _C_mem_nl
	s" a = (Char *) " _C_conc_param
	s"  = (Cell) *a;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_c!
\     	Generate C code for 'c!'.

: _C_c!                    	( -- )
	_C_debug if ." forth2c: at _C_c!" cr then

	s" /* cstore */" _C_mem_beginblock
   	s" Char *a; " _C_mem_store_nl
	s" Cell c;" _C_mem_store_nl _C_mem_nl
	s" a = (Char *) " _C_conc_param
	s" c = (Char) " _C_conc_param _C_mem_nl
	s" *a = c;" _C_mem_store
	_C_mem_endblock
;

\ _C_c,
\     	Generate C code for 'c,'.

: _C_c,                     	( -- )
	_C_debug if ." forth2c: at _C_c," cr then
	
	s" /* c_comma */" _C_mem_beginblock
	s" Char c, *a;" _C_mem_store_nl _C_mem_nl
	s" a = (Char *) _C_fetch_here();" _C_mem_store_nl
	s" c = (Char) " _C_conc_param _C_mem_nl
	s" *a = c;" _C_mem_store_nl
	s" _C_request_mem(sizeof(Char));" _C_mem_store
	_C_mem_endblock
;

\ _C_cr
\     	Generate C code for 'cr'.

: _C_cr                    	( -- )
	_C_debug if ." forth2c: at _C_cr" cr then

	s" /* c_r */" _C_mem_beginblock
   	s" printf(" _C_mem_store
   	[char] " _C_mem_store_char
   	s" \n" _C_mem_store
   	[char] " _C_mem_store_char
   	s" );" _C_mem_store_nl
	_C_mem_endblock
;

\ _C_i
\     	Generate C code for 'i'. The index is the topmost variable of
\     	the return stack.

: _C_i                     	( -- )
	_C_debug if ." forth2c: at _C_i" cr then
	                  
	s" /* i */" _C_mem_beginblock
	s" Cell n = " _C_conc_rstack
	s"  = n;"  _C_locvar_conc
	1 _C_currstack +!    \ has been decremented by _C_conc_rstack !
	_C_mem_endblock
;

\ _C_j
\     	Generate C code for 'j'. The index is the third variable of
\     	the return stack.

: _C_j                     	( -- )
	_C_debug if ." forth2c: at _C_j" cr then
	                  
	s" /* j */" _C_mem_beginblock
	s" Cell n = r" _C_mem_store
	_C_currstack @ 3 -   \ generate variable number
	s>d <# [char] ; hold #s #> _C_mem_store_nl
	s"  = n;"  _C_locvar_conc
	_C_mem_endblock
;      

\ _C_decimal
\     	Set _C_base to 10.

: _C_decimal               	( -- )
	_C_debug if ." forth2c: at _C_decimal" cr then
	
	s" /* decimal */" _C_mem_beginblock
	s" _C_base = 10;" _C_mem_store
	_C_mem_endblock
;

\ _C_hex
\     	Set _C_base to 16.

: _C_hex                   	( -- )
	_C_debug if ." forth2c: at _C_hex" cr then
	
	s" /* hex */" _C_mem_beginblock
	s" _C_base = 16;" _C_mem_store
	_C_mem_endblock
;

\ _C_base
\     	Put the address of variable _C_base onto the stack.

: _C_base                  	( -- )
	_C_debug if ." forth2c: at _C_base" cr then
	
	s" /* base */" _C_mem_beginblock
	s" = (Cell) &_C_base;" _C_locvar_conc
	_C_mem_endblock
;      

\ _C_# 
\     	Generate C code for '#'.

: _C_#                     	( -- )
	_C_debug if ." forth2c: at _C_#" cr then
	
	s" /* number_sign */" _C_mem_beginblock
	s" UDCell n, quot;" _C_mem_store_nl
	s" int rem;" _C_mem_store_nl
	s" UCell u1, u2;" _C_mem_store_nl _C_mem_nl
	
	s" u1 = (UCell) " _C_conc_param
	s" u2 = (UCell) " _C_conc_param
	s" _C_MAKEUDCELL(n, u1, u2);" _C_mem_store_nl _C_mem_nl
	
	s" quot = n / _C_base;" _C_mem_store_nl
	s" rem = n % _C_base;" _C_mem_store_nl _C_mem_nl
	
	s" u1 = _C_ULOWHALF(quot);" _C_mem_store_nl
	s" u2 = _C_UHIGHHALF(quot);" _C_mem_store_nl
	
	s"  = u1;" _C_locvar_conc
	s"  = u2;" _C_locvar_conc
	s" _C_pic_add_digit(rem);" _C_mem_store
	_C_mem_endblock   
;

\ _C_#>
\     	Generate C code for '#>'.

: _C_#>                    	( -- )
	_C_debug if ." forth2c: at _C_#>" cr then
	
	-2 _C_curlocal +!    \ drop xd
	
	s" /* number_sign_greater */" _C_mem_beginblock
	s" int index = _C_PICBUFFER_SIZE - _C_pic_buffer_index;" 
	_C_mem_store_nl _C_mem_nl
	
	s" _C_pic_buffer[_C_PICBUFFER_SIZE - 1] = '\0';" _C_mem_store_nl
	s" _C_pic_buffer[index - 1] = _C_pic_buffer_index - 1;" 
	_C_mem_store_nl
	
	s"  = (Cell) &_C_pic_buffer[index];" _C_locvar_conc
	s"  = (Cell) _C_pic_buffer_index - 1;" _C_locvar_conc
	
	s" _C_pic_buffer_index = 1;" _C_mem_store_nl
	
	_C_mem_endblock   
;

\ _C_<#
\     	Generate C code for '<#'.

: _C_<#                    	( -- )
	_C_debug if ." forth2c: at _C_<#" cr then
	
	s" /* number_sign_less */" _C_mem_beginblock
	s" _C_pic_buffer_index = 1;" _C_mem_store_nl
	_C_mem_endblock
;

\ _C_#s
\     	Generate C code for '#s'.

: _C_#s                    	( -- )
	_C_debug if ." forth2c: at _C_#s" cr then
	
	s" /* number_sign_s */" _C_mem_beginblock
	s" UDCell n, quot;" _C_mem_store_nl
	s" int rem;" _C_mem_store_nl
	s" UCell u1, u2;" _C_mem_store_nl _C_mem_nl
	
	s" do" _C_mem_store 0 0 _C_mem_beginblock
	
	s" u1 = (UCell) " _C_conc_param
	s" u2 = (UCell) " _C_conc_param
	s" _C_MAKEUDCELL(n, u1, u2);" _C_mem_store_nl _C_mem_nl
	
	s" quot = n / _C_base;" _C_mem_store_nl
	s" rem = n % _C_base;" _C_mem_store_nl _C_mem_nl
	
	s" u1 = _C_ULOWHALF(quot);" _C_mem_store_nl
	s" u2 = _C_UHIGHHALF(quot);" _C_mem_store_nl
	
	s"  = u1;" _C_locvar_conc
	s"  = u2;" _C_locvar_conc
	s" _C_pic_add_digit(rem);" _C_mem_store
	_C_mem_endblock
	s" while (quot != 0);" _C_mem_store_nl
	
	_C_mem_endblock   
;

\ _C_s>d
\     	Generate C code for 's>d'.

: _C_s>d                   	( -- )
	_C_debug if ." forth2c: at _C_s>d" cr then
	
	s" /* s_to_d */" _C_mem_beginblock
	s" DCell d;" _C_mem_store_nl
	s" d = (DCell) " _C_conc_param
	s"  = _C_LOWHALF(d);" _C_locvar_conc
	s"  = _C_HIGHHALF(d);" _C_locvar_conc
	_C_mem_endblock   
;      

\ _C_hold
\     	Generate C code for 'hold'.

: _C_hold                     	( -- )
	_C_debug if ." forth2c: at _C_hold" cr then
	
	s" /* hold */" _C_mem_beginblock
	s" Char c;" _C_mem_store_nl
	s" c = " _C_conc_param
	s" _C_pic_add_char(c);" _C_mem_store
	_C_mem_endblock   
;

\ _C_type
\     	Generate C code for 'type'.

: _C_type                     	( -- )
	_C_debug if ." forth2c: at _C_type" cr then
	
	s" /* type */" _C_mem_beginblock
	s" Cell u;" _C_mem_store_nl
	s" Char *a;" _C_mem_store_nl
	
	s" u = " _C_conc_param
	s" a = (Char *) " _C_conc_param
	
	s" while (u-- != 0) printf(" _C_mem_store
	[char] " _C_mem_store_char
	s" %c" _C_mem_store
	[char] " _C_mem_store_char
	s" , *a++);" _C_mem_store_nl
	s" fflush(stdout);" _C_mem_store
	_C_mem_endblock   
;

       
\ _C_aligned
\     	Generate code for 'aligned'.

: _C_aligned               	( -- )
	_C_debug if ." forth2c: at _C_aligned" cr then
	
	s" /* aligned */" _C_mem_beginblock
	s" Cell *a;" _C_mem_store_nl _C_mem_nl
	s" a = (Cell *) " _C_conc_param
	s" _C_ALIGN(a);" _C_mem_store_nl
	s"  = (Cell) a;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_align
\     	Generate code for 'align'.

: _C_align              	( -- )
	_C_debug if ." forth2c: at _C_align" cr then
	
	s" /* align */" _C_mem_beginblock
	s" _C_ALIGN(_C_here);" _C_mem_store   
	_C_mem_endblock
;              

\ _C_allot
\     	Generate code for 'allot' in compilation state.

: _C_allot                 	( -- )
	_C_debug if ." forth2c: at _C_allot" cr then
	
	s" /* allot */" _C_mem_beginblock
	s" Cell n; " _C_mem_store_nl _C_mem_nl
	s" n = " _C_conc_param
	s" _C_request_mem(n);" _C_mem_store
	_C_mem_endblock
;

\ _C_create
\     	Generate code for 'create' in compilation state.

: _C_create           		( -- )
	_C_debug if ." forth2c: at _C_create" cr then
	
	s" /* create */" _C_mem_beginblock
	s" _C_ALIGN(_C_here);" _C_mem_store_nl
	s" #error 'create' is impossible to do without an input stream"
	_C_mem_store
	_C_mem_endblock
	   
	cr ." forth2c: encountered 'create' -- C output is an #error directive."
;      

\ _C_create
\     	Generate code for 'constant' in compilation state.

: _C_constant          		( -- )
	_C_debug if ." forth2c: at _C_constant" cr then
	
	s" /* constant */" _C_mem_beginblock
	s" #error 'constant' is impossible to do without an input stream"
	_C_mem_store
	_C_mem_endblock
	   
	cr ." forth2c: encountered 'constant' -- C output is an #error directive."
;      

\ _C_abort
\     	Generate code for 'abort'.

: _C_abort              	( -- )
	_C_debug if ." forth2c: at _C_abort" cr then
	
	s" /* abort */" _C_mem_beginblock
	s" printf(" _C_mem_store
	[char] " _C_mem_store_char
	s" \nAbort.\n" _C_mem_store
	[char] " _C_mem_store_char
	s" );" _C_mem_store_nl
	s" exit(1);" _C_mem_store
	_C_mem_endblock
;

\ _C_quit
\     	Generate code for 'quit'.

: _C_quit               	( -- )
   	_C_debug if ." forth2c: at _C_quit" cr then
   
	s" /* quit */" _C_mem_beginblock
	s" printf(" _C_mem_store
	[char] " _C_mem_store_char
	s" \nQuit.\n" _C_mem_store
	[char] " _C_mem_store_char
	s" );" _C_mem_store_nl
	s" exit(1);" _C_mem_store
	_C_mem_endblock
;

\ _C_cell
\     	Generate code for 'cell'.

: _C_cell               	( -- )
	_C_debug if ." forth2c: at _C_cell" cr then
	
	s" /* cell */" _C_mem_beginblock
	s"  = sizeof(Cell);" _C_locvar_conc
	_C_mem_endblock
;

\ _C_char
\     	Generate code for 'char'.

: _C_char               	( -- )
	_C_debug if ." forth2c: at _C_char" cr then
	
	s" /* char */" _C_mem_beginblock
	s" #error 'char' is impossible to do without an input stream !" 
	_C_mem_store
	cr ." forth2c: encountered 'char' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_bl
\     	Generate code for 'bl'.

: _C_bl              		( -- )
	_C_debug if ." forth2c: at _C_bl" cr then
	
	s" /* bl */" _C_mem_beginblock
	s"  = (Cell) ' ';" _C_locvar_conc
	_C_mem_endblock
;

\ _C_blank
\     	Generate code for 'blank'.

: _C_blank              	( -- )
	_C_debug if ." forth2c: at _C_blank" cr then
	
	s" /* blank */" _C_mem_beginblock
	s" Char *ca;" _C_mem_store_nl
	s" UCell u; " _C_mem_store_nl
	s" u = " _C_conc_param
	s" ca = (Char *) " _C_conc_param _C_mem_nl
	s" while (u--) *ca++ = ' ';" _C_mem_store_nl
	_C_mem_endblock
;

\ _C_*/
\     	Generate C code for '*/'.

: _C_*/                    	( -- )
   	_C_debug if ." forth2c: at _C_*/" cr then
   
   	s" /* star_slash */" _C_mem_beginblock
	s" Cell n1, n2, n3;" _C_mem_store_nl 
	s" DCell d1, d2;" _C_mem_store_nl _C_mem_nl
	s" n3 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" n1 = " _C_conc_param _C_mem_nl
	s" d1 = (DCell) n1 * (DCell) n2;" _C_mem_store_nl
	s" d2 = d1 / (DCell) n3;" _C_mem_store_nl
	s" if (1 % -3 > 0 && (d1 < 0) != (n3 < 0) && d1 % (DCell) n3 != 0) d2--;"
   	_C_mem_store_nl
   	s"  = (Cell) d2;" _C_locvar_conc
   	_C_mem_endblock
;

\ _C_*/mod
\     	Generate C code for '*/mod'.

: _C_*/mod                    	( -- )
   	_C_debug if ." forth2c: at _C_*/mod" cr then
   
   	s" /* star_slash_mod */" _C_mem_beginblock
	s" Cell n1, n2, n3;" _C_mem_store_nl 
	s" DCell d1, d2, d3;" _C_mem_store_nl _C_mem_nl
	s" n3 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" n1 = " _C_conc_param _C_mem_nl
	s" d1 = (DCell) n1 * (DCell) n2;" _C_mem_store_nl
	s" d2 = d1 / (DCell) n3;" _C_mem_store_nl
   	s" d3 = d1 % (DCell) n3;" _C_mem_store_nl
	s" if (1 % -3 > 0 && (d1 < 0) != (n3 < 0) && d3 != 0)" _C_mem_store
   	0 0 _C_mem_beginblock
	s" d2--;" _C_mem_store_nl
	s" d3 += (DCell) n3;" _C_mem_store 
   	_C_mem_endblock
   	s"  = (Cell) d3;" _C_locvar_conc
   	s"  = (Cell) d2;" _C_locvar_conc
   	_C_mem_endblock
;      

\ _C_>number
\     	Generate C code for '>number'.

: _C_>number               	( -- )
	_C_debug if ." forth2c: at _C_>number" cr then
	
	s" /* to_number */" _C_mem_beginblock
	s" UCell u, u1, u2;" _C_mem_store_nl
	s" Char *a;" _C_mem_store_nl
	s" int digit;" _C_mem_store_nl
	s" UDCell ud;" _C_mem_store_nl _C_mem_nl
	s" u = (UCell) " _C_conc_param
	s" a = (Char *) " _C_conc_param
	s" u1 = (UCell) " _C_conc_param
	s" u2 = (UCell) " _C_conc_param
	s" _C_MAKEUDCELL(ud, u1, u2);" _C_mem_store_nl _C_mem_nl
	
	s" while (u > 0)" _C_mem_store
	0 0 _C_mem_beginblock
	s" if ((digit = _C_to_digit(*a)) < 0) break;" _C_mem_store_nl
	s" ud *= (UDCell) _C_base;" _C_mem_store_nl
	s" ud += (UDCell) digit;" _C_mem_store_nl
	s" u--; a++;" _C_mem_store
	_C_mem_endblock
	
	s"  = (Cell) _C_ULOWHALF(ud);" _C_locvar_conc
	s"  = (Cell) _C_UHIGHHALF(ud);" _C_locvar_conc
	s"  = (Cell) a;" _C_locvar_conc
	s"  = (Cell) u;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_accept
\     	Generate C code for 'accept'.

: _C_accept                   	( -- )
	_C_debug if ." forth2c: at _C_accept" cr then
	
	s" /* accept */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl
	s" Char *a;" _C_mem_store_nl 
	s" char *temp;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" a = (Char *) " _C_conc_param 
	s" if ((temp = malloc(n1+1)) == NULL)" _C_mem_store
	0 0 _C_mem_beginblock
	s" fprintf(stderr, " _C_mem_store
	[char] " _C_mem_store_char
	s" malloc failed: %s\n" _C_mem_store
	[char] " _C_mem_store_char
	s" , strerror(errno));" _C_mem_store_nl
	s" exit(1);" _C_mem_store
	_C_mem_endblock
	s" if (fgets(temp, n1+1, stdin) == NULL) n2 = 0;" _C_mem_store_nl
	s" else" _C_mem_store 
	0 0 _C_mem_beginblock
	s" char *nl = strchr(temp, '\n');" _C_mem_store_nl
	s" if (nl != NULL) *nl = '\0';" _C_mem_store_nl
	s" n2 = strlen(temp);" _C_mem_store_nl
	s" memcpy(a, temp, n2);" _C_mem_store
	_C_mem_endblock
	s" free(temp);" _C_mem_store_nl
	s"  = n2;" _C_locvar_conc
	_C_mem_endblock
;
       
\ _C_compare
\     	Generate C code for 'compare'.

: _C_compare                 	( -- )
	_C_debug if ." forth2c: at _C_compare" cr then
	
	s" /* compare */" _C_mem_beginblock
	s" UCell u1, u2;" _C_mem_store_nl
	s" Cell n;" _C_mem_store_nl
	s" Char *a1, *a2;" _C_mem_store_nl _C_mem_nl
	s" u2 = (UCell) " _C_conc_param
	s" a2 = (Char *) " _C_conc_param
	s" u1 = (UCell) " _C_conc_param
	s" a1 = (Char *) " _C_conc_param _C_mem_nl
	
	s" n = memcmp(a1, a2, u1 < u2 ? u1 : u2);" _C_mem_store_nl
	s" if (n == 0)  n = u1 - u2;" _C_mem_store_nl
	s" if (n < 0)  n = -1;" _C_mem_store_nl
	s" else if (n > 0) n = 1;" _C_mem_store_nl
	s"  = n;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_emit
\     	Generate C code for 'emit'.

: _C_emit                     	( -- )
	_C_debug if ." forth2c: at _C_emit" cr then
	
	s" /* emit */" _C_mem_beginblock
	s" Char c;" _C_mem_store_nl
	s" c = (Char) " _C_conc_param
	s" putchar(c);" _C_mem_store_nl
	s" fflush(stdout);" _C_mem_store
	_C_mem_endblock
;

\ _C_key
\     	Generate C code for 'key'.

: _C_key                   	( -- )
	_C_debug if ." forth2c: at _C_key" cr then
	
	s" /* key */" _C_mem_beginblock
	s" Char c;" _C_mem_store_nl
	s" c = getchar();" _C_mem_store_nl
	s"  = (Cell) c;" _C_locvar_conc
	_C_mem_endblock
;      

\ _C_d+
\     	Generate C code for 'd+'.

: _C_d+                    	( -- )
	_C_debug if ." forth2c: at _C_d+" cr then
	
	s" /* d_plus */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl
	s" DCell d1, d2, d;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d2, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d1, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" d = d1 + d2;" _C_mem_store_nl _C_mem_nl
	s"  = _C_LOWHALF(d);" _C_locvar_conc
	s"  = _C_HIGHHALF(d);" _C_locvar_conc
	_C_mem_endblock
;

\ _C_d-
\     	Generate C code for 'd-'.

: _C_d-                    	( -- )
	_C_debug if ." forth2c: at _C_d-" cr then
	
	s" /* d_minus */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl
	s" DCell d1, d2, d;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d2, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d1, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" d = d1 - d2;" _C_mem_store_nl _C_mem_nl
	s"  = _C_LOWHALF(d);" _C_locvar_conc
	s"  = _C_HIGHHALF(d);" _C_locvar_conc
	_C_mem_endblock
;

\ _C_d2* 
\     	Generate C code for 'd2* '.

: _C_d2*                      	( -- )
	_C_debug if ." forth2c: at _C_d2* " cr then
	
	s" /* d_two_star  */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl
	s" DCell d1, d;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d1, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" d = d1 << 1;" _C_mem_store_nl _C_mem_nl
	s"  = _C_LOWHALF(d);" _C_locvar_conc
	s"  = _C_HIGHHALF(d);" _C_locvar_conc
	_C_mem_endblock
;

\ _C_d2/ 
\     	Generate C code for 'd2/ '.

: _C_d2/                      	( -- )
	_C_debug if ." forth2c: at _C_d2/ " cr then
	
	s" /* d_two_slash  */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl
	s" DCell d1, d;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d1, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" d = d1 >> 1;" _C_mem_store_nl _C_mem_nl
	
	s"  = _C_HIGHHALF(d);" _C_locvar_conc
	_C_mem_endblock
;

\ _C_d=
\     	Generate C code for 'd='.

: _C_d=                    	( -- )
	_C_debug if ." forth2c: at _C_d=" cr then
	
	s" /* d_equals */" _C_mem_beginblock
	s" Cell flag;" _C_mem_store_nl
	s" Cell n1, n2;" _C_mem_store_nl
	s" DCell d1, d2;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d2, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d1, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" flag = FLAG(d1 == d2);" _C_mem_store_nl _C_mem_nl
	s"  = flag;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_dabs
\     	Generate C code for 'dabs'.

: _C_dabs                     	( -- )
	_C_debug if ." forth2c: at _C_dabs" cr then
	
	s" /* d_abs */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl
	s" DCell d1, d;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d1, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" if (d1 < 0) d = -d1;" _C_mem_store_nl _C_mem_nl
	s"  = _C_LOWHALF(d);" _C_locvar_conc
	s"  = _C_HIGHHALF(d);" _C_locvar_conc
	_C_mem_endblock
;

\ _C_dmax
\     	Generate C code for 'dmax'.

: _C_dmax                     	( -- )
	_C_debug if ." forth2c: at _C_dmax" cr then
	
	s" /* d_max */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl
	s" DCell d1, d2, d;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d2, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d1, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" if (d1 > d2) d = d1;" _C_mem_store_nl
	s" else d = d2;" _C_mem_store_nl _C_mem_nl
	s"  = _C_LOWHALF(d);" _C_locvar_conc
	s"  = _C_HIGHHALF(d);" _C_locvar_conc
	_C_mem_endblock
;

\ _C_dmin
\     	Generate C code for 'dmin'.

: _C_dmin                     	( -- )
	_C_debug if ." forth2c: at _C_dmin" cr then
	
	s" /* d_min */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl
	s" DCell d1, d2, d;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d2, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d1, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" if (d1 < d2) d = d1;" _C_mem_store_nl
	s" else d = d2;" _C_mem_store_nl _C_mem_nl
	s"  = _C_LOWHALF(d);" _C_locvar_conc
	s"  = _C_HIGHHALF(d);" _C_locvar_conc
	_C_mem_endblock
	_C_mem_endblock
;

\ _C_dnegate
\     	Generate C code for 'dnegate'.

: _C_dnegate                	( -- )
	_C_debug if ." forth2c: at _C_dnegate" cr then
	
	s" /* d_negate */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl
	s" DCell d1, d;" _C_mem_store_nl _C_mem_nl
	s" n1 = " _C_conc_param
	s" n2 = " _C_conc_param
	s" _C_MAKEDCELL(d1, n1, n2);" _C_mem_store_nl _C_mem_nl
	s" d = -d1;" _C_mem_store_nl _C_mem_nl
	s"  = _C_LOWHALF(d);" _C_locvar_conc
	s"  = _C_HIGHHALF(d);" _C_locvar_conc
	_C_mem_endblock
;
       
\ _C_nip
\     	Generate C code for 'nip'.

: _C_nip                   	( -- )
	_C_debug if ." forth2c: at _C_nip" cr then
	
	s" /* nip */" _C_mem_beginblock
	s" Cell n1, n2;" _C_mem_store_nl _C_mem_nl
	s" n2 = " _C_conc_param
	s" n1 = " _C_conc_param
	s"  = n2;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_pad
\     	Generate C code for 'pad'.

: _C_pad                   	( -- )
	_C_debug if ." forth2c: at _C_pad" cr then
	
	s" /* pad */" _C_mem_beginblock
	s"  = (Cell) &_C_pad;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_parse
\     	Generate C code for 'parse'.

: _C_parse                    	( -- )
	_C_debug if ." forth2c: at _C_parse" cr then
	
	s" /* parse */" _C_mem_beginblock
	s" #error 'parse' is impossible to do without an input stream !" 
	_C_mem_store
	cr ." forth2c: encountered 'parse' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_pick
\     	Generate C code for 'pick'.

: _C_pick                     	( -- )
	_C_debug if ." forth2c: at _C_pick" cr then
	
	s" /* pick */" _C_mem_beginblock
	s" #error 'pick' is impossible to do without a real stack !"
	_C_mem_store
	cr ." forth2c: encountered 'pick' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_?dup
\     	Generate C code for '?dup'.

: _C_?dup                     	( -- )
	_C_debug if ." forth2c: at _C_?dup" cr then
	
	s" /* ?dup */" _C_mem_beginblock
	s" #error '?dup' is impossible to do without a real stack !"
	_C_mem_store
	cr ." forth2c: encountered '?dup' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_environment?
\     	Generate C code for 'environment?'.

: _C_environment?		( -- )
	_C_debug if ." forth2c: at _C_environment?" cr then
	
	s" /* environment? */" _C_mem_beginblock
	s" #error 'environment?' is impossible to do without a real stack !"
	_C_mem_store
	cr ." forth2c: encountered 'environment?' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_evaluate
\     	Generate C code for 'evaluate'.

: _C_evaluate			( -- )
	_C_debug if ." forth2c: at _C_evaluate" cr then
	
	s" /* evaluate */" _C_mem_beginblock
	s" #error 'evaluate' is impossible to do without an input stream !"
	_C_mem_store
	cr ." forth2c: encountered 'evaluate' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_source
\     	Generate C code for 'source'.

: _C_source			( -- )
	_C_debug if ." forth2c: at _C_source" cr then
	
	s" /* source */" _C_mem_beginblock
	s" #error 'source' is impossible to do without an input stream !"
	_C_mem_store
	cr ." forth2c: encountered 'source' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_find
\     	Generate C code for 'find'.

: _C_find			( -- )
	_C_debug if ." forth2c: at _C_find" cr then
	
	s" /* find */" _C_mem_beginblock
	s" #error 'find' is impossible to do without a 'real forth' !"
	_C_mem_store
	cr ." forth2c: encountered 'find' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_'
\    	Generate C code for '''.

: _C_'                     	( -- )
	_C_debug if ." forth2c: at _C_'" cr then
	
	s" /* tick */" _C_mem_beginblock
	s" #error 'tick' is impossible to do without a real forth !"
	_C_mem_store
	cr ." forth2c: encountered 'tick' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_execute
\     	Generate C code for 'execute'.

: _C_execute                	( -- )
	_C_debug if ." forth2c: at _C_execute" cr then
	
	s" /* execute */" _C_mem_beginblock
	s" #error 'execute' is impossible to do without a real forth !"
	_C_mem_store
	cr ." forth2c: encountered 'execute' -- C output is an #error directive." 
	cr
	_C_mem_endblock
;

\ _C_>body
\    	Generate C code for '>body'.

: _C_>body                     	( -- )
	_C_debug if ." forth2c: at _C_>body" cr then
	
	s" /* to-body */" _C_mem_beginblock
	s"  = 0;" _C_locvar_conc
	s" #error '>body' is impossible to do without a real forth !"
	_C_mem_store
	cr ." forth2c: encountered '>body' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_>in
\    	Generate C code for '>in'.

: _C_>in                     	( -- )
	_C_debug if ." forth2c: at _C_>in" cr then
	
	s" /* to-in */" _C_mem_beginblock
	s"  = 0;" _C_locvar_conc
	s" #error '>in' is impossible to do without a real forth !"
	_C_mem_store
	cr ." forth2c: encountered '>in' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_depth
\     	Generate C code for 'depth'.

: _C_depth                  	( -- )
	_C_debug if ." forth2c: at _C_depth" cr then
	
	s" /* depth */" _C_mem_beginblock
	_C_curlocal @ _C_nrparams @ +
	_C_next_locvar _C_mem_store
	s"  = " _C_mem_store
	s>d <# [char] ; hold #s #> _C_mem_store
	_C_mem_endblock
	
	cr ." forth2c: encountered 'depth' -- C output will probably be wrong." cr
;

\ _C_sign
\     	Generate C code for 'sign'.

: _C_sign                     	( -- )
	_C_debug if ." forth2c: at _C_sign" cr then
	
	s" /* sign */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl
	s" n = " _C_conc_param
	s" if (n < 0) _C_pic_add_char('-');" _C_mem_store
	_C_mem_endblock
;

\ _C_space
\     	Generate C code for 'space'.

: _C_space                    	( -- )
	_C_debug if ." forth2c: at _C_space" cr then
	
	s" /* space */" _C_mem_beginblock
	s" putchar(' ');" _C_mem_store_nl
	s" fflush(stdout);" _C_mem_store
	_C_mem_endblock
;

\ _C_spaces
\     	Generate C code for 'spaces'.

: _C_spaces                   	( -- )
	_C_debug if ." forth2c: at _C_spaces" cr then
	
	s" /* spaces */" _C_mem_beginblock
	s" Cell n;" _C_mem_store_nl _C_mem_nl
	
	s" n = " _C_conc_param	
	s" while (n--) putchar(' ');" _C_mem_store_nl
	s" fflush(stdout);" _C_mem_store
	_C_mem_endblock
;

\ _C_state
\     	Generate C code for 'state'.

: _C_state                    	( -- )
	_C_debug if ." forth2c: at _C_state" cr then
	
	s" /* state */" _C_mem_beginblock
	s"  = (Cell) &_C_state;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_tuck
\     	Generate C code for 'tuck'.

: _C_tuck                     	( -- )
	_C_debug if ." forth2c: at _C_tuck" cr then
	
	s" /* tuck */" _C_mem_beginblock
	s" Cell c1, c2;" _C_mem_store_nl _C_mem_nl
	s" c2 = " _C_conc_param
	s" c1 = " _C_conc_param _C_mem_nl
	s"  = c2;" _C_locvar_conc
	s"  = c1;" _C_locvar_conc
	s"  = c2;" _C_locvar_conc
	
	_C_mem_endblock
;
       
\ _C_u.
\     	Generate C code for 'u.'.

: _C_u.                    	( -- )
   	_C_debug if ." forth2c: at _C_u." cr then
   
   	s" /* u_dot */" _C_mem_beginblock
				\ generate a printf statement, which is
				\ a bit complicated because of the '"' ...
	s" printf(" _C_mem_store	\ 'printf('
	[char] " _C_mem_store_char	\ 'printf("'
	s" %u " _C_mem_store		\ 'printf(" %u'
	[char] " _C_mem_store_char	\ 'printf(" %u"'
	s" , " _C_mem_store		\ 'printf(" %u", '
	_C_get_next_param _C_mem_store	\ 'printf(" %u", x?'
	s" );" _C_mem_store_nl		\ 'printf(" %u", x?);'
   	s" fflush(stdout);" _C_mem_store
   	_C_mem_endblock
;

\ _C_u<
\     	Generate C code for 'u<'.

: _C_u<                    	( -- )
	_C_debug if ." forth2c: at _C_u<" cr then
	
	s" /* u_less_than */" _C_mem_beginblock
	s" UCell u1, u2;" _C_mem_store_nl 
	s" Cell flag;" _C_mem_store_nl _C_mem_nl
	s" u2 = (UCell) " _C_conc_param
	s" u1 = (UCell) " _C_conc_param _C_mem_nl
	s" flag = FLAG(u1 < u2);" _C_mem_store_nl
	s"  = flag;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_u>
\     	Generate C code for 'u>'.

: _C_u>                    	( -- )
	_C_debug if ." forth2c: at _C_u>" cr then
	
	s" /* u_greater_than */" _C_mem_beginblock
	s" UCell u1, u2;" _C_mem_store_nl 
	s" Cell flag;" _C_mem_store_nl _C_mem_nl
	
	s" u2 = (UCell) " _C_conc_param
	s" u1 = (UCell) " _C_conc_param _C_mem_nl
	
	s" flag = FLAG(u1 > u2);" _C_mem_store_nl
	s"  = flag;" _C_locvar_conc
	_C_mem_endblock
;

\ _C_word
\     	Generate C code for 'word'.

: _C_word                     	( -- )
	_C_debug if ." forth2c: at _C_word" cr then
	
	s" /* word */" _C_mem_beginblock
	s" #error 'word' is impossible to do without an input stream !" 
	_C_mem_store
	cr ." forth2c: encountered 'word' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_here
\     	Generate C code for 'here'.

: _C_here                  	( -- )
	_C_debug if ." forth2c: at _C_word" cr then
	
	s" /* here */" _C_mem_beginblock
	s"  = (Cell) _C_fetch_here();" _C_locvar_conc
	_C_mem_endblock
;



\ Words used by the hooks. These deal primarily with output of
\ function headers etc. to the C file.

\ _C_>lower
\	Takes a character and converts it to lowercase if it is in
\	uppercase.
\	NOTE: this word relies on ASCII encoding (or something similar),
\		so it is not very portable.

: _C_>lower               	( c -- c )
	dup [char] A [char] Z 1+ within	if
		[char] a [char] A - +
   	then
;

\ _C_cname_buffer
\	This buffer holds a converted function name (used by >cname)

create _C_cname_buffer _C_funcname_size chars allot

\ _C_>cname
\	Takes a string and converts it to a valid C name (all characters
\	that would result in an invalid C name are converted to underscores)
\	NOTE: this can result in name clashes, e.g. (loop) and +loop+ both
\		are converted to _loop_.
\	c-addr1 in the stack diagrams denotes _C_cname_buffer.

: _C_>cname                	( c-addr u -- c-addr u )
	2>r _C_cname_buffer 2r@		( c-addr1 c-addr u ;r: c-addr u )
	0 ?do
		dup c@                  ( c-addr1 c-addr c ;r: c-addr u )
		_C_>lower               ( c-addr1 c-addr c ;r: c-addr u )
		dup [char] a [char] z 1+ within	
		over [char] _ = or      ( c-addr1 c-addr c flag; r: c-addr u )
		           	\ flag == (c is a letter | underscore)
				
		i if		\ somewhere in the middle of the name, so
			        \ there may also be a digit
			over [char] 0 [char] 9 1+ within or
		then		\ now flag == (c is letter | digit | '_')
		
		0= if           \ replace invalid character by an underscore
			drop
			[char] _
		then                    ( c-addr1 c-addr c ;r: c-addr u )
		
		rot tuck c!		( c-addr c-addr1 )
		char+ swap char+	( c-addr1 c-addr )
	loop
	2drop 
	2r> nip _C_cname_buffer swap	( c-addr1 u )
;
 
\ _C_write_fileheader
\	Writes some preliminary code to the output file (like the
\	include statements and so on)

: _C_write_fileheader		( -- )
	s" /* Generated by forth2c, (C) Martin Maierhofer 1994 */"
	_C_store _C_nl _C_nl
      	s"                      /* size of data space, can be changed ! */"
      	_C_store _C_nl
      	s" int _C_DATA_SIZE = 8192;" _C_store _C_nl _C_nl
	s" #include " _C_store
	[char] " _C_store_char
	s" forth.h" _C_store
	[char] " _C_store_char
	s"    /* general forth definitions */" _C_store _C_nl
	s" #include " _C_store
	[char] " _C_store_char
	s" forth2c.h" _C_store
	[char] " _C_store_char
	s"  /* forth2c specific definitions */"	_C_store _C_nl _C_nl
;

\ _C_write_funcname
\     	Dump the function name and the parameters

: _C_write_funcname		( -- )
				\ write the function name							
	_C_funcname count _C_>cname _C_store
	[char] ( _C_store_char
				\ dump all parameters execpt the last one...
	_C_nrparams @ dup dup if	( n n )
				\ at least one parameter here
		1- 0 ?do
				\ the parameters are name 'p0' through 'p?'
			s" Cell p" _C_store
			i s>d <# #s #> _C_store
			s" , " _C_store
		loop								( n )
																						  
				\ ...which has to be written 'manually' as
				\ there is no trailing ','
		s" Cell p" _C_store
		1- s>d <# #s #> _C_store
		[char] ) _C_store_char
	else
				\ no parameters at all
		2drop
		s" void)" _C_store
	then
	
	_C_nl [char] { _C_store_char _C_nl
;
 
\ _C_write_funcheader
\	Write the return value (i.e. 'void', 'Cell' or 'Cell??'), the
\	function name (via _C_write_funcname) and the definition of the
\	local variable holding the return value to the output file.

: _C_write_funcheader		( -- )
				\ get the number of return values
	_C_nrparams @
	_C_curlocal @ + dup		( n n )
	
	0= if			\ no return value
		drop 
		s" void " _C_store
		_C_write_funcname
		exit
	then
	dup 1 = if		\ one Cell to return
		drop 
		s" Cell " _C_store
		_C_write_funcname
		s" Cell _c_result;" _C_store _C_nl
	else			\ we have only defined structures with 
				\ at most 20 Cells so dump a warning if 
				\ the number of Cells to be returned 
				\ exceeds this limit
		dup $14 > if
			." forth2c: Sure you need > 20 return values ? " cr
			." forth2c: Ok, so you got to redefine a struct in forth2c.h yourself !"
			cr
		then
		dup			( n n )
		s" Cells" _C_store
		s>d <# _C_space_char hold #s #> _C_store
		_C_write_funcname
				\ write definition for the structure 
				\ to be returned			
		s" Cells" _C_store
		s>d <# #s #> _C_store
		s"  _c_result;" _C_store _C_nl
	then
;
 
\ _C_write_localvar
\	Write the definition of the local variables to the output
\	file.

: _C_write_localvar		( -- )
				\ _C_maxlocals holds the number of local 
				\ variables needed to implement the ordinary 
				\ stack. The variables are named 'x?'
	_C_maxlocals @ 0 ?do
		s" Cell x" _C_store
		i s>d <# [char] ; hold #s #> _C_store _C_nl
	loop
				\ _C_maxrstack holds the number of local 
				\ variables needed to implement the return 
				\ stack. These variables are named 'r?'
	_C_maxrstack @ 0 ?do
		s" Cell r" _C_store
		i s>d <# [char] ; hold #s #> _C_store
		_C_nl
	loop
;
 
\ _C_write_returnstruct
\	Write code filling in the return structure to the output file.
\	NOTE: Code is written to _C_code as this word can be called by
\		compiling the forth 'exit'-word.

: _C_write_returnstruct		( -- )
   	_C_nrparams @		\ fetch the number of Cells to return
	_C_curlocal @ + dup		( n n )
	0= if			\ nothing to do
      		s" return;" _C_mem_store_nl
		drop exit
	then		
	dup 1 = if		\ a simple assignment does the job
      		s" _c_result = " _C_conc_param
		drop
	else			\ iterate through the number of local 
				\ variables and copy them to the return 
				\ structure
		0 ?do
			s" _c_result.cell" _C_mem_store
			i s>d <# #s #> _C_mem_store
			s"  = " _C_conc_param
		loop
	then
	
	s" return (_c_result);" _C_mem_store_nl		
;

\ _C_create_var
\	This word is called whenever there is a variable being created
\	in interpretation state. It writes the name of the variable to
\	the output file and sets some state variables. The end of a
\	variable definition is handled by _C_create_endvar (cf. below)
\	Forth variables are translated to C arrays of Cells and the
\	number of cells allocated are added, e.g.
\		create foo 2 cells allot 7 cells allot
\	will generate code like
\		Cell foo[9];
\	(At least that's what I'd like it to generate ;-)

: _C_create_var            	( c-addr u -- )
	0 _C_allot_size !	\ accumulated number of cells for the var
   	1 _C_creating !		\ flag to prevent infinite recursion
     
   	_C_nl s" Char " _C_store
   	_C_store		\ dump the variable name
   	[char] [ _C_store_char
;

\ _C_create_endvar
\	Ends the definition of a variable. See also the comment for
\	_C_create_var above.
\	_C_allot_size contains the accumulated number of cells to
\	reserve for the variable.

: _C_create_endvar         	( -- )
   	_C_creating @ if
      		_C_allot_size @ s>d  <# #s #> _C_store
      		s" ];" _C_store _C_nl _C_nl
      		0 _C_creating !
   	then
;

\ _C_create_const
\	This word is called whenever there is a constant being created
\	in interpretation state. It generates a '#define' for the C code.

: _C_create_const            	( c-addr u -- )
	_C_nl s" #define " _C_store
   	_C_store		\ dump the constant name
	_C_tab_char _C_store_char
	_C_constant_value @ s>d <# #s #> _C_store _C_nl
;

\ _C_insert_word
\	This word adds a word to our wordlist. The parameters are
\	the type of the entry (see below) (type in the stack diagram), 
\	and maybe two cells of type specific data (nx) for functions
\	and the name of the function or variable or constant as in forth 
\	nomenclatura.
\
\	The contents of the wordlist entry are as follows:
\		* type -- 1 cell (0=function, 1=variable, 2=constant)
\		* data -- 2 cells (functions only, nr of returned cells
\				   and nr of parameters)
\	The word as spelled by forth is entered into the wordlist and
\	thereafter the name is converted to a valid C identifier.

: _C_insert_word           	( i*n type c-addr u -- c-addr u )
	1 _C_inserting !	\ prevent infinite recursion
	get-current >r		\ enable our wordlist
	2>r 				( i*n type ; r: wid c-addr u )
	_C_wordlist @ set-current
   	2r@ nextname create   	\ insert the new word into our wordlist
	1 			\ the number of cells to allot
	over 0= if 2 + then	\ need two additional cells for functions
	cells here swap allot		( i*n type addr ; r: wid c-addr u )
	2dup !  			( i*n type addr ; r: wid c-addr u )
	swap 0= if		\ save an additional two cells for functions
		cell+ tuck !		( n addr+1 ; r: wid c-addr u )
		cell+ !			( r: wid c-addr u )
	else
		drop		\ don't need address any longer
	then
	2r> r> set-current		\ restore the old wordlist
	_C_>cname			( c-addr u )
	0 _C_inserting !
;

\ _C_code_function
\	Generate C code for a function call. The (forth-ish) name
\	of the variable and the number of parameters (n2) and the
\	number of cells returned (n1).
\	MM: This word definitely is ugly.

: _C_code_function		( c-addr u n1 n2 -- )
	_C_debug if ." forth2c: at _C_code_function" cr then
			     
	>r >r				( c-addr u ; r: n2 n1 )	   
	0 0 _C_mem_beginblock
	r@ if			\ we get some return values
		r@ 1 = if	\ only one return value
			s" Cell _C_locret;" _C_mem_store_nl _C_mem_nl
		else		\ we get more return values
			s" Cells" _C_mem_store
			r@ s>d <# #s #> _C_mem_store
			s"  _C_locret;" _C_mem_store_nl _C_mem_nl
		then
		
		s" _C_locret = " _C_mem_store
	then
	_C_>cname _C_mem_store	\ write the function name
	[char] ( _C_mem_store_char
	r> r> 				( n1 n2 )
	dup 0> if		\ do we have parameters ?
		1- 0 ?do	\ yes, write them one after the other
			_C_get_next_param _C_mem_store
			s" , " _C_mem_store
		loop
		_C_get_next_param _C_mem_store
	else
		drop		\ don't need nr of parameters
	then
	s" );" _C_mem_store_nl
	
	dup 0> if		\ do we have returned cells ?
		dup 1 = if	\ yes 1 cell, simple
			drop
			s"  = _C_locret;" _C_locvar_conc
		else		\ copy the cells of the returned structure
				\ into local vars (in reversed order !)
			0 swap 1- ?do
				_C_next_locvar _C_mem_store
				s"  = _C_locret.cell" _C_mem_store
				i s>d <# [char] ; hold #s #> _C_mem_store_nl
			-1 +loop
		then
	else
		drop		\ don't need nr of retcells any longer
	then
	_C_mem_endblock
;
 
\ _C_code_variable
\	Generate C code for a variable access. The (forth-ish) name
\	of the variable is our parameter.

: _C_code_variable		( c-addr u -- )
	_C_debug if ." forth2c: at _C_code_variable" cr then
				   
	0 0 _C_mem_beginblock
	_C_next_locvar _C_mem_store
	s"  = (Cell) (&" _C_mem_store
	_C_>cname _C_mem_store
	s" [0]);" _C_mem_store
	_C_mem_endblock
;

\ _C_code_constant
\	Generate C code for a constant. The data field of a constant
\	represents its value.

: _C_code_constant		( c-addr u -- )
	_C_debug if ." forth2c: at _C_code_constant" cr then

	0 0 _C_mem_beginblock	
	_C_next_locvar _C_mem_store
	s"  = (Cell) " _C_mem_store
	_C_>cname _C_mem_store
	[char] ; _C_mem_store_char
	_C_mem_endblock
;
 
\ _C_no_primitive
\	This word gets called when there is something to be compiled
\	which is not a primitive. We look up the word in our wordlist
\	where we store all our functions and variables and generate
\	the corresponding C code.
\	Flag is false when no matching entry has been found, true 
\	otherwise.

: _C_no_primitive		( c-addr u -- flag )
	_C_debug if ." forth2c: at _C_no_primitive -- " 2dup type cr then
	
	2dup _C_wordlist @ search-wordlist
	0= if 			\ no such name in our wordlist
		2drop false exit 
	then
	
	execute				( c-addr u addr )	
	dup @ dup			( c-addr u addr type type )
	0= if			\ it's a function
		drop
		cell+ dup @		( c-addr u addr n1 )
		swap cell+ @		( c-addr u n1 n2 )
		_C_code_function
	else
		dup 1 = if	\ it's a variable
			2drop _C_code_variable
		else		\ it must be a constant
			2drop _C_code_constant
		then
	then
	true			\ everything worked out fine
;



\ Words being the interface to kernal.fs. They can be easily recognized
\ by the '_hook' at the end of their names.

\ _C_compile,_hook
\ 	This word is called by compile, in kernal.fs whenever there is
\	a new word to be compiled. The argument (which should never be
\	changed is) is the execution token of the word to be inserted.
\	We compare xt with the xts of the primitives and call the
\	resulting function to generate the code. If this fails, we try
\	to obtain the function name and generate code for calling this
\	function in C.
\	MM: This is ugly and should be changed.
			       
: _C_compile,_hook 		( xt -- xt )
	_C_debug if ." forth2c: at _C_compile,_hook(" dup . ." )" cr then
				\ arithmetics stuff
	dup ['] + = if _C_+ exit then
	dup ['] - = if _C_- exit then
	dup ['] * = if _C_* exit then
	dup ['] / = if _C_/ exit then
	dup ['] negate = if _C_negate exit then
	dup ['] 1+ = if _C_1+ exit then
	dup ['] 1- = if _C_1- exit then
	dup ['] max = if _C_max exit then
	dup ['] min = if _C_min exit then
	dup ['] abs = if _C_abs exit then
	dup ['] mod = if _C_mod exit then
	dup ['] /mod = if _C_/mod exit then
	dup ['] 2* = if _C_2* exit then
	dup ['] 2/ = if _C_2/ exit then
	dup ['] fm/mod = if _C_fm/mod exit then
	dup ['] sm/rem = if _C_sm/rem exit then
	dup ['] m* = if _C_m* exit then
	dup ['] um* = if _C_um* exit then
	dup ['] um/mod = if _C_um/mod exit then
	dup ['] rshift = if _C_rshift exit then
	dup ['] lshift = if _C_lshift exit then
	dup ['] and = if _C_and exit then
	dup ['] or = if _C_or exit then
	dup ['] xor = if _C_xor exit then
	dup ['] invert = if _C_invert exit then
	dup ['] */ = if _C_*/ exit then
	dup ['] */mod = if _C_*/mod exit then
	dup ['] # = if _C_# exit then
	dup ['] s>d = if _C_s>d exit then
				\ double arithmetics stuff
	dup ['] d+ = if _C_d+ exit then
	dup ['] d- = if _C_d- exit then
	dup ['] d2* = if _C_d2* exit then
	dup ['] d2/ = if _C_d2/ exit then
	dup ['] d= = if _C_d= exit then
	dup ['] dabs = if _C_dabs exit then
	dup ['] dmax = if _C_dmax exit then
	dup ['] dmin = if _C_dmin exit then
	dup ['] dnegate = if _C_dnegate exit then
				\ comparison stuff
	dup ['] < = if _C_< exit then
	dup ['] <> = if _C_<> exit then
	dup ['] = = if _C_= exit then
	dup ['] > = if _C_> exit then
	dup ['] 0< = if _C_0< exit then
	dup ['] 0<> = if _C_0<> exit then
	dup ['] 0> = if _C_0> exit then
	dup ['] 0= = if _C_0= exit then
	dup ['] u< = if _C_u< exit then
	dup ['] u> = if _C_u> exit then
				\ stack operations stuff
	dup ['] over = if _C_over exit then
	dup ['] drop = if _C_drop exit then
	dup ['] swap = if _C_swap exit then
	dup ['] dup = if _C_dup exit then
   	dup ['] nip = if _C_nip exit then
	dup ['] rot = if _C_rot exit then
	dup ['] pick = if _C_pick exit then
	dup ['] tuck = if _C_tuck exit then
	dup ['] 2dup = if _C_2dup exit then
	dup ['] 2drop = if _C_2drop exit then
	dup ['] 2over = if _C_2over exit then
	dup ['] 2rot = if _C_2rot exit then
	dup ['] 2swap = if _C_2swap exit then
				\ return stack stuff
	dup ['] r> = if _C_r> exit then	
	dup ['] >r = if _C_>r exit then	
	dup ['] 2>r = if _C_2>r exit then	
	dup ['] 2r> = if _C_2r> exit then	
	dup ['] rdrop = if _C_rdrop exit then	
	dup ['] 2rdrop = if _C_2rdrop exit then	
	dup ['] r@ = if _C_r@ exit then	
	dup ['] 2r@ = if _C_2r@ exit then	
				\ misc stuff
	dup ['] decimal = if _C_decimal exit then
	dup ['] hex = if _C_hex exit then
	dup ['] depth = if _C_depth exit then
	dup ['] fill = if _C_fill exit then
	dup ['] noop = if _C_noop exit then
	dup ['] move = if _C_move exit then
	dup ['] >number = if _C_>number exit then
	dup ['] count = if _C_count exit then
	dup ['] abort = if _C_abort exit then
	dup ['] accept = if _C_accept exit then
	dup ['] base = if _C_base exit then
	dup ['] bl = if _C_bl exit then
	dup ['] blank = if _C_blank exit then
	dup ['] char = if _C_char exit then
	dup ['] compare = if _C_compare exit then
[IFDEF] environment?	
   	dup ['] environment? = if _C_environment? exit then
[THEN]
   	dup ['] find = if _C_find exit then
   	dup ['] parse = if _C_parse exit then
      	dup ['] ' = if _C_' exit then
   	dup ['] execute = if _C_execute exit then
   	dup ['] ?dup = if _C_?dup exit then
   	dup ['] quit = if _C_quit exit then
	dup ['] spaces = if _C_spaces exit then
	dup ['] state = if _C_state exit then
	dup ['] word = if _C_word exit then
	dup ['] >body = if _C_>body exit then
	dup ['] >in = if _C_>in exit then
	dup ['] evaluate = if _C_evaluate exit then
	dup ['] source = if _C_source exit then
				\ data space stuff
	dup ['] , = if _C_, exit then
	dup ['] ! = if _C_! exit then
	dup ['] +! = if _C_+! exit then
	dup ['] 2! = if _C_2! exit then
	dup ['] 2@ = if _C_2@ exit then
	dup ['] @ = if _C_@ exit then
	dup ['] c! = if _C_c! exit then
	dup ['] c@ = if _C_c@ exit then
	dup ['] c, = if _C_c, exit then
	dup ['] cell+ = if _C_cell+ exit then
	dup ['] cells = if _C_cells exit then
	dup ['] cell = if _C_cell exit then
	dup ['] char+ = if _C_char+ exit then
	dup ['] chars = if _C_chars exit then
	dup ['] aligned = if _C_aligned exit then
	dup ['] align = if _C_align exit then
	dup ['] allot = if _C_allot exit then
   	dup ['] here = if _C_here exit then
   	dup ['] constant = if _C_constant exit then
      	dup ['] create = if _C_create exit then
				\ i/o stuff
	dup ['] . = if _C_. exit then
	dup ['] #> = if _C_#> exit then
	dup ['] <# = if _C_<# exit then
	dup ['] #s = if _C_#s exit then
	dup ['] cr = if _C_cr exit then
	dup ['] hold = if _C_hold exit then
	dup ['] key = if _C_key exit then
   	dup ['] pad = if _C_pad exit then
   	dup ['] emit = if _C_emit exit then
	dup ['] type = if _C_type exit then
	dup ['] u. = if _C_u. exit then
	dup ['] sign = if _C_sign exit then
	dup ['] space = if _C_space exit then
				\ control flow stuff	
   	dup ['] unloop = if _C_unloop exit then
	dup ['] i = if _C_i exit then
   	dup ['] j = if _C_j exit then
                        \ some words used by IF, THEN etc. that
			\ should not generate an error message
	dup ['] ?branch = if exit then
	dup ['] dest? = if exit then
	dup ['] ?branch-lp+!# = if exit then
	dup ['] (loop) = if exit then
	dup ['] (+loop) = if exit then
	dup ['] (loop)-lp+!# = if exit then
	dup ['] (+loop)-lp+!# = if exit then
	dup ['] lit = if exit then
	dup ['] (do) = if exit then
	dup ['] ;s = if exit then
	dup ['] branch = if exit then
	dup ['] (abort") = if exit then
	dup ['] (.") = if exit then
			\ 'recurse'ing should not give us a
			\ warning
	dup lastxt = if exit then

	dup >name name>string		( xt c-addr u )
	_C_no_primitive 0= if   
		cr ." forth2c: Unknown execution token: " dup . cr
		." forth2c: (Textual representation: '"
		dup >name name>string type ." ')" cr
		." forth2c: [Ignoring this word.]" cr
	then
;

\ _C_:_hook
\ 	This word is called by : in kernal.fs whenever a new word is begun.
\	This word resets all counter variables for the new function and 
\	records the word (= function) name in _C_funcname.
\	In order to extract the function name, the input stream has to
\	be saved (via >in @) and later restored.

: _C_:_hook 			( -- )
	_C_debug if ." forth2c: at _C_:hook" cr then
   	_C_create_endvar	\ : ends a 'created' variable...
	>in @ >r bl word r> >in !
	dup c@ 1+		\ make c-addr a c-addr u
	_C_funcname swap cmove			
	0 _C_code_index !	\ now initialize all the counter variables
	0 _C_nrparams !
	0 _C_maxlocals !
	0 _C_curlocal !
	0 _C_maxrstack !
	0 _C_currstack !
	0 _C_curlabel !
	1 _C_curtab !
   	0 _C_cf_buffer_index !
	0 _C_cf_leave_index !
;

\ _C_;_hook
\	This word is called by ; in kernal.fs which means that the definition
\	of the word/function is complete. The state of the variables follows:
\ 	_C_code..._C_code+_C_code_index
\ 		contains the C code for the function
\ 	_C_maxlocals
\ 		contains the number of local variables needed (x0 to x?)
\ 	_C_curlocal (can be negative -> is the parameter number - 1)
\ 		contains the number of local variables still in use
\ 		i.e. _C_nrparams + _C_curlocal == the number of cells to
\           	return
\ 	_C_nrparams
\ 		contains the number of Cells the function takes as
\ 		its parameters (p0 to p?)
\	_C_maxrstack
\		contains the number of local variables needed for the return
\		stack (r0 to r?)

: _C_;_hook 			( -- )
	_C_debug if ." forth2c: at _C_;"  cr then
	
	_C_nrparams @		\ the number of parameters
	dup _C_curlocal @ +	\ the number of cells returned
	0			\ id for a function
	_C_funcname count
	_C_insert_word		\ insert the function into our wordlist
	2drop			\ don't need 'C'ish name
	
	_C_nl
	_C_write_funcheader	\ write the function header
	_C_write_localvar	\ write definitions for the local variables
   
				\ and now put the whole stuff out to the fs
	_C_code _C_code_index @ _C_store _C_nl
	[char] } _C_store_char _C_nl
;

\ _C_Literal_hook
\ 	Generate C code for a literal whose value is given at the stack.

: _C_Literal_hook 		( n -- n )
	_C_debug if ." forth2c: at _C_Literal_hook(" dup . ." )" cr then
	
	s" /* Literal */" _C_mem_beginblock
	_C_next_locvar _C_mem_store	
	s"  = " _C_mem_store
	dup s>d tuck dabs <# [char] ; hold #s rot sign #> _C_mem_store
	_C_mem_endblock
;      

\ _C_if_hook
\	Generate C code for 'if'. This is a little bit tricky: when it comes
\	to else, we should use the same local variables (i.e. our virtual
\	stack) as in the if-branch. E.g.
\		if 2 else 3 then 
\	clearly, '2' and '3' should go into the same local variable so that
\	code after 'then' can continue no matter which branch has been
\	taken. This means also that variable stack effects will not be coded
\	properly by this implementation.
\	To reach this goal, we have to store the current top of stack var
\	number somewhere -- this somewhere is of course a 'stack' !
\     	To avoid a rather complex interface with the control flow words in
\     	kernal.fs, we keep a stack of our own for this purpose instead of
\     	using the forth stack. 
\	In addition, we have to record the 'top of stack' variable of the
\     	return stack and the label number for later use by _C_else_hook.

: _C_if_hook			( -- )
	_C_debug if ." forth2c: at _C_if_hook" cr then
																							
   	_C_curlabel @      		( labelnr )
   	dup _C_make_goto_if        	( labelnr )
	_C_curlocal @			( labelnr curlocal )
   	_C_currstack @             	( labelnr curlocal currstack )
   	_C_cf_push                 	( -- )
	1 _C_curlabel +!	\ labels for the next control structures
;

\ _C_then_hook
\	Generate C code for 'then', i.e. write a new label number (the 
\     	third stack parameter as returned from _C_cf_pop).

: _C_then_hook			( -- )
	_C_debug if ." forth2c: at _C_then_hook" cr then

   	_C_cf_pop 2drop            	( labelnr )
   	_C_make_label	
;
                                           
\ _C_ahead_hook
\     	Generate C code for 'ahead', i.e. a goto instruction to the next
\     	free label and put this label number to the cf stack via _C_cf_push

: _C_ahead_hook            	( -- )
	_C_debug if ." forth2c: at _C_ahead_hook" cr then

   	_C_curlabel @        		( label )
   	dup _C_make_goto           	( label )
	_C_curlocal @
	_C_currstack @
	_C_cf_push
   	1 _C_curlabel +!     	\ make room for next control flow words
;

\ _C_else_hook
\	Generate C code for 'else'. 'ahead' and 'then' are called by
\     	'else' in kernal.fs and they handle the label and goto stuff,
\     	so we only have to reset the local variable numbers.

: _C_else_hook 			( -- )
	_C_debug if ." forth2c: at _C_else_hook" cr then

   	_C_cf_pop                  	( labelnr curlocal currstack )
   	dup _C_currstack !         	( labelnr curlocal currstack )
	over _C_curlocal !         	( labelnr curlocal currstack )
   	_C_cf_push                 	( -- )
;

\ _C_begin_hook
\	Generate C code for 'begin'. First, a label is generated and an
\     	entry is put on the cf 'stack' for later use by _C_until_hook.
\     	Next, _C_curlabel is incremented to make room for the next 
\     	control flow words.

: _C_begin_hook			( -- )
	_C_debug if ." forth2c: at _C_begin_hook" cr then
   
   	_C_curlabel @           	( labelnr )
   	dup _C_make_label		( labelnr )
	_C_curlocal @
	_C_currstack @
	_C_cf_push
   	1 _C_curlabel +!	\ labels for the next control structures
;

\ _C_until_hook
\	Generate C code for 'until', write a conditional goto to the 
\     	label the number of which is returned by _C_cf_pop.

: _C_until_hook			( -- )
	_C_debug if ." forth2c: at _C_until_hook" cr then
                     
   	_C_cf_pop 2drop            	( labelnr )
   	_C_make_goto_if
;

\ _C_again_hook
\	Generate C code for 'again', i.e. generate a goto to the label
\     	number indicated by the entry returned from _C_cf_pop.

: _C_again_hook			( -- )
	_C_debug if ." forth2c: at _C_again_hook" cr then
   
   	_C_cf_pop 2drop           	( labelnr )
   	_C_make_goto
;

\ _C_cs-pick_hook
\     	We have to change our control flow 'stack' accordingly. This is
\     	done by _C_cf_pick.

: _C_cs-pick_hook          	( n -- n )
	_C_debug if ." forth2c: at _C_cs-pick_hook" cr then
   
   	_C_cf_pick
;

\ _C_cs-roll_hook
\     We have to change our control flow 'stack' accordingly. This is
\     done by _C_cf_roll.

: _C_cs-roll_hook          ( n -- n )
	_C_fileid @ 0= if exit then
	_C_debug if ." forth2c: at _C_cs-roll_hook" cr then
   
   _C_cf_roll
;

\ _C_do_hook
\	Generate C code for 'do'. We reserve two local variables from
\     	the 'rstack' to save the index and the bounds of the loop. These
\     	are accessed later on in _C_loop_hook.

: _C_do_hook			( -- )
	_C_debug if ." forth2c: at _C_do_hook" cr then

   	s" /* do */" _C_mem_beginblock
   	s" Cell bound, index;" _C_mem_store_nl _C_mem_nl
   	s" index = " _C_conc_param
   	s" bound = " _C_conc_param _C_mem_nl
	s"  = bound;" _C_rstackvar_conc
	s"  = index;" _C_rstackvar_conc 
	_C_mem_endblock
;

\ _C_?do_hook
\	Generate C code for '?do'. We reserve two local variables from
\     	the 'rstack' to save the index and the bounds of the loop. These
\     	are accessed later on in _C_loop_hook.

: _C_?do_hook			( -- )
	_C_debug if ." forth2c: at _C_?do_hook" cr then

	s" /* ?do */" _C_mem_beginblock
	s" Cell bound, index;" _C_mem_store_nl _C_mem_nl
	s" index = " _C_conc_param
	s" bound = " _C_conc_param _C_mem_nl
	s"  = bound;" _C_rstackvar_conc
	s"  = index;" _C_rstackvar_conc 
	s" if (bound == index) goto label" _C_mem_store
	_C_curlabel @ dup          	( n n )
	s>d <# [char] ; hold #s #> _C_mem_store_nl
	_C_mem_endblock
	_C_curlocal @
	_C_currstack @
	_C_cf_push
	1 _C_curlabel +!     	\ make room for next control structure
;

\ _C_loop_hook
\	Generate C code for 'loop'. Fetch the index and the bound from
\     	the 'rstack' and check for the exit condition.

: _C_loop_hook			( -- )
	_C_debug if ." forth2c: at _C_loop_hook" cr then
              
	s" /* loop */" _C_mem_beginblock
	s" Cell index, bound;" _C_mem_store_nl _C_mem_nl
	s" index = " _C_conc_rstack
	s" bound = " _C_conc_rstack _C_mem_nl
	s" index++;" _C_mem_store_nl
	s" if (index != bound)" _C_mem_store
	0 0 _C_mem_beginblock
	s"  = bound;" _C_rstackvar_conc
	s"  = index;" _C_rstackvar_conc       
	s" goto label" _C_mem_store
	_C_cf_pop 2>r dup    	\ fetch the label number
	s>d <# [char] ; hold #s #> _C_mem_store_nl
	2r> _C_cf_push		
	_C_mem_endblock
	_C_mem_endblock
   	\ -2 _C_currstack +!	\ adjust rstack number (called _C_r.._conc !)
				\ unloop will do this for us...
;


\ _C_+loop_hook
\	Generate C code for '+loop'. Fetch the index and the bound from
\     	the 'rstack' and check for the exit condition.

: _C_+loop_hook			( -- )
	_C_debug if ." forth2c: at _C_+loop_hook" cr then
              
	s" /* +loop */" _C_mem_beginblock
	s" Cell step, index, bound;" _C_mem_store_nl _C_mem_nl
	s" index = " _C_conc_rstack
	s" bound = " _C_conc_rstack _C_mem_nl
	s" step = " _C_conc_param                 
	s" index += step;" _C_mem_store_nl
	s" if ((step > 0 && index < bound) ||" _C_mem_store _C_mem_nl
	_C_tab_char _C_mem_store_char
	s" (step <= 0 && index >= bound))" _C_mem_store
	0 0 _C_mem_beginblock
	s"  = bound;" _C_rstackvar_conc
	s"  = index;" _C_rstackvar_conc       
	s" goto label" _C_mem_store
	_C_cf_pop 2>r dup	\ fetch the label number
	s>d <# [char] ; hold #s #> _C_mem_store_nl
	2r> _C_cf_push
	_C_mem_endblock
	_C_mem_endblock
   	\ -2 _C_currstack +!	\ adjust rstack number (called _C_r.._conc !)
				\ unloop will do this for us
;
                           
\ _C_loop_purge_hook
\     	Called at the end of 'loop' and '+loop' in kernal.fs, this word
\     	removes the 'dodest' entries (?!) from the control flow stack.

: _C_loop_purge_hook       	( -- )
	_C_debug if ." forth2c: at _C_loop_purge_hook" cr then
   
   	_C_cf_pop _C_cf_pop 2drop 2drop 2drop
;

\ _C_>leave_hook
\	Move one entry from the cf stack to the leave stack.

: _C_>leave_hook            	( -- )
	_C_debug if ." forth2c: at _C_>leave_hook" cr then
		 
	_C_cf_pop 2drop			( labelnr )
	_C_cf_leave_push
;

\ _C_leave>_hook
\	Move one entry from the leave stack to the cf stack.

: _C_leave>_hook            	( -- )
	_C_debug if ." forth2c: at _C_leave>_hook" cr then

	_C_cf_leave_pop			( labelnr )
	0 0 _C_cf_push		 
;

\ _C_s"_hook
\ 	Generate C code for 's"'.

: _C_s"_hook	            	( -- )
	state @ 0= if exit then
	_C_debug if ." forth2c: at _C_squote" cr then
	
	base @ hex
	s" /* s_quote */" _C_mem_beginblock
	>in @ >r [char] " parse r> >in !
	s" Cell n = " _C_mem_store
	dup s>d <# [char] ; hold #s #> _C_mem_store_nl
	s" Char *str = " _C_mem_store
	[char] " _C_mem_store_char
	s" \x" _C_mem_store
	dup s>d <# # # #> _C_mem_store
	0 ?do
		s" \x" _C_mem_store
		dup c@ s>d <# # # #> _C_mem_store
		char+
	loop
	drop
	[char] " _C_mem_store_char	
	[char] ; _C_mem_store_char _C_mem_nl _C_mem_nl
	s"  = (Cell) (&str[1]);" _C_locvar_conc
	s"  = n;" _C_locvar_conc
	_C_mem_endblock
	base !
;
   
\ _C_."_hook
\ 	Generate C code for '."'.

: _C_."_hook	            	( -- )
	_C_debug if ." forth2c: at _C_.quote" cr then
	
	s" /* dot_quote */" _C_mem_beginblock
				\ generate a printf statement, which is
				\ a bit complicated because of the '"' ...
	s" fputs(" _C_mem_store         \ 'fputs('
	[char] " _C_mem_store_char	\ 'fputs("'
	>in @ >r [char] " parse r> >in !
   	_C_mem_store                    \ 'fputs(".....'
	[char] " _C_mem_store_char	\ 'fputs("......"'
	s" , stdout);" _C_mem_store_nl	\ 'fputs("......", stdout);'
   	s" fflush(stdout);" _C_mem_store
	_C_mem_endblock
;

\ _C_allot_hook
\     	Called for an 'allot' while interpreting. If we are in the midst
\	of creating a definition of a variable (i.e. _C_creating == 1),
\	increment the accumulated size by the stack argument.

: _C_allot_hook          	( n -- n )
   	state @ _C_creating @ 0= or if exit then
   	_C_debug if ." forth2c: at _C_allot_hook" cr then
   
   	dup _C_allot_size +!
;      
				  
\ _C_save_word_hook
\	Called by 'create' and 'constant' in interpretative state.
\	Parse the next word of the input stream and save it into
\	a secure location, where it can be found by _C_create_hook
\	and _C_constant_hook.
\	In addition, we save the topmost stack element (if any) into
\	_C_constant_value where it can be retrieved later (the
\	original cell will be discarded by 'constant'.

: _C_save_word_hook		( n -- n )
	state @ _C_inserting @ or if exit then
   	_C_debug if ." forth2c: at _C_save_word_hook" cr then
					       
   	>in @ >r bl word r> >in !
	dup c@ 1+
	_C_wordname swap cmove
	depth 0> if dup _C_constant_value ! then
;

\ _C_create_hook
\     	Called by 'create' in interpretative state. If we are currently
\	inserting a word into our wordlist (i.e. _C_inserting == true),
\	exit to prevent an infinite recursion. Otherwise, terminate the
\	previous variable definition (if any), insert the word into our
\	wordlist and begin a new variable definition.

: _C_create_hook           	( -- )
	state @ _C_inserting @ or if exit then
   	_C_debug if ." forth2c: at _C_create_hook" cr then
   
   	_C_create_endvar  	\ terminate previous variable declaration
	_C_wordname count	\ fetch the name
   	2>r 1 2r> _C_insert_word        ( c-addr u )
   	_C_create_var     	\ and create a new variable
;

\ _C_constant_hook
\     	Called by 'constant' in interpretative state. Terminate the
\	previous variable definition (if any), insert the word into our
\	wordlist and add a constant definition (#define)

: _C_constant_hook           	( -- )
	state @ if exit then
   	_C_debug if ." forth2c: at _C_constant_hook" cr then

   	_C_create_endvar  	\ terminate previous variable declaration
	_C_wordname count	\ fetch the name
   	2>r 2 2r> _C_insert_word        ( c-addr u )
   	_C_create_const     	\ and create a new constant
;

\ _C_abort"_hook
\     	Generate code for 'abort"'.

: _C_abort"_hook           	( -- )
   	_C_debug if ." forth2c: at _C_abort_quote_hook" cr then
   
	s" /* abort_quote */" _C_mem_beginblock
	s" Cell n; " _C_mem_store_nl
	s" n = " _C_conc_param
	s" if (n != 0) " _C_mem_store 
	
	0 0 _C_mem_beginblock
	s" fputs(" _C_mem_store
	[char] " _C_mem_store_char
	s" \n" _C_mem_store
	>in @ >r [char] " parse	r> >in !
	_C_mem_store
	s" \n" _C_mem_store
	[char] " _C_mem_store_char
	s" , stdout);" _C_mem_store_nl
	s" exit(n);" _C_mem_store   
	_C_mem_endblock
	
	_C_mem_endblock
;

\ _C_exit_hook
\	Generate C code for 'exit'. Most of the work is done by 
\	_C_write_returnstruct but we have to save _C_curlocal as
\	it is changed by _C_write_returnstruct.

: _C_exit_hook             	( -- )
	_C_debug if ." forth2c: at _C_exit" cr then

   	_C_curlocal @
	s" /* exit */" _C_mem_beginblock
   	_C_write_returnstruct
	_C_mem_endblock
   	_C_curlocal !
;      

\ _C_postpone_hook
\	Generate C code for 'postpone'.

: _C_postpone_hook		( -- )
	_C_debug if ." forth2c: at _C_postpone_hook" cr then
	
	s" /* postpone */" _C_mem_beginblock
	s" #error 'postpone' is impossible to do without an input stream"
	_C_mem_store
	_C_mem_endblock
	   
	cr ." forth2c: encountered 'postpone' -- C output is an #error directive."
;

\ _C_does>_hook
\	Generate C code for 'does>'.

: _C_does>_hook			( -- )
	_C_debug if ." forth2c: at _C_does>_hook" cr then
	
	s" /* does> */" _C_mem_beginblock
	s" #error 'does>' is impossible to do without a real stack !"
	_C_mem_store
	cr ." forth2c: encountered 'does>' -- C output is an #error directive."
	_C_mem_endblock
;

\ _C_recurse_hook
\	Generate C code for 'recurse'. This may fail so we give the user
\	a warning.

: _C_recurse_hook		( -- )
	_C_debug if ." forth2c: at _C_recurse_hook" cr then
	
	s" /* recurse */" _C_mem_beginblock    
	_C_funcname count	\ the function name
	_C_nrparams @		\ the number of parameters
	_C_curlocal @ 		\ the number of cells returned
	swap
	_C_code_function
	_C_mem_endblock
	
	cr ." forth2c: encountered 'recurse': C code may be wrong !"
;

\ words helping _C_init (the initialization word) with interactive
\ filename specification.
       
\ _C_temp_filename
\	Used by _C_get_filename

variable _C_temp_filename $20 chars allot

\ _C_get_filename
\	Reads a string from the keyboard (via accept) and returns
\	its address and count.

: _C_get_filename		( -- c-addr u )
	cr ." forth2c: please enter a filename for the C output file."
	cr $20 ." [maximum length: " . ." , press <enter> to disable forth2c] ? "
	_C_temp_filename $20 accept cr	( u )
	_C_temp_filename swap		( c-addr u )
;

\ _C_init
\ 	Initializes the converter. Called by in forth2c.fs when starting.
\	If a name has been specified to gforth by the command line,
\	the global variable _C_fileName contains this name, otherwise
\	we read the filename interactively via _C_get_filename.
\	This word tries to open the indicated name for writing.
\ 	If this is successful, the file id is stored to the global
\ 	variable _C_fileId in kernal.fs; otherwise an error message is 
\	written to the screen.

: _C_init			( -- )
	_C_debug if cr ." forth2c: at _C_init" cr then
	
	_C_fileName 2@ dup	\ as defined in kernal.fs
	0= if			\ when no name has yet been specified
		2drop
		_C_get_filename	\ read filename from keyboard
		dup 0= if 2drop exit then
	then
				\ try to open the output file
	2dup w/o open-file 0= if
		>r		\ save the file-id and create a wordlist
      		wordlist _C_wordlist !
		2drop		\ filename is no longer needed
		r> _C_fileId !	\ store the file-id
		_C_write_fileheader
	else			\ open-file failed
		drop		\ forget about the (invalid) fileid
		cr ." forth2c: Could not open '" type 
		." ' for writing; no C code will be generated." cr
	then
;

\ and now bind the deferred words from kernal.fs to our hooks.

' _C_;_hook is _C_;_hook_defer
' _C_:_hook is _C_:_hook_defer
' _C_save_word_hook is _C_save_word_hook_defer
' _C_create_hook is _C_create_hook_defer
' _C_constant_hook is _C_constant_hook_defer
' _C_compile,_hook is _C_compile,_hook_defer
' _C_abort"_hook is _C_abort"_hook_defer
' _C_."_hook is _C_."_hook_defer
' _C_s"_hook is _C_s"_hook_defer
' _C_exit_hook is _C_exit_hook_defer
' _C_+loop_hook is _C_+loop_hook_defer
' _C_loop_hook is _C_loop_hook_defer
' _C_loop_purge_hook is _C_loop_purge_hook_defer
' _C_?do_hook is _C_?do_hook_defer
' _C_do_hook is _C_do_hook_defer
' _C_until_hook is _C_until_hook_defer
' _C_again_hook is _C_again_hook_defer
' _C_begin_hook is _C_begin_hook_defer
' _C_else_hook is _C_else_hook_defer
' _C_then_hook is _C_then_hook_defer
' _C_if_hook is _C_if_hook_defer
' _C_ahead_hook is _C_ahead_hook_defer
' _C_>leave_hook is _C_>leave_hook_defer
' _C_leave>_hook is _C_leave>_hook_defer
' _C_cs-roll_hook is _C_cs-roll_hook_defer
' _C_cs-pick_hook is _C_cs-pick_hook_defer
' _C_Literal_hook is _C_Literal_hook_defer
' _C_allot_hook is _C_allot_hook_defer
' _C_recurse_hook is _C_recurse_hook_defer
' _C_does>_hook is _C_does>_hook_defer
' _C_postpone_hook is _C_postpone_hook_defer

\ *
\ * See comment at the beginning of the file
\ *
BASE !
